﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/fs.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/time.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

using namespace nn::fs;

namespace {

    nn::Result FindSaveDataId(nn::fs::SaveDataId *outValue, nn::ncm::ApplicationId applicationId, nn::fs::UserId userId) NN_NOEXCEPT
    {
        const nn::fs::SaveDataSpaceId SpaceIds[] = {nn::fs::SaveDataSpaceId::User, nn::fs::SaveDataSpaceId::System};
        for(auto spaceId : SpaceIds)
        {
            std::unique_ptr<nn::fs::SaveDataIterator> iter;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, spaceId));

            while (NN_STATIC_CONDITION(true))
            {
                int64_t count;
                nn::fs::SaveDataInfo info;
                NNT_EXPECT_RESULT_SUCCESS(iter->ReadSaveDataInfo(&count, &info, 1));
                if (count == 0)
                {
                    break;
                }

                if (info.applicationId.value == applicationId.value && info.saveDataUserId == userId)
                {
                    *outValue = info.saveDataId;
                    NN_RESULT_SUCCESS;
                }
            }
        }
        return nn::fs::ResultTargetNotFound();
    }

    nn::Bit64 GetSaveDataApplicationId() NN_NOEXCEPT
    {
        return 0x0005000C10000000;
    }

    nn::fs::UserId GetUserIdByIndex(int index) NN_NOEXCEPT
    {
        nn::fs::UserId userId = {{ 0xFull, static_cast<uint32_t>(index) }};
        return userId;
    }

    nn::Bit64 GetOwnerIdByIndex(int index) NN_NOEXCEPT
    {
        return GetSaveDataApplicationId() + index;
    }

    uint32_t GetFlagsByIndex(int index) NN_NOEXCEPT
    {
        return 1024 - index;
    }

    nnt::fs::util::String GetMountNameByIndex(int index) NN_NOEXCEPT
    {
        nnt::fs::util::String mountName = "save" + nnt::fs::util::ToString(index);
        return mountName;
    }

    nnt::fs::util::String GetDuplexMountNameByIndex(int index) NN_NOEXCEPT
    {
        nnt::fs::util::String mountName = "saveD" + nnt::fs::util::ToString(index);
        return mountName;
    }
}

TEST(SaveDataExtraData, Basic)
{
    const size_t SaveDataSize = 1024 * 1024;
    const size_t SaveDataJournalSize = 1024 * 1024;
    const int SaveDataNum = 2;

    nn::ncm::ApplicationId appId = { GetSaveDataApplicationId() };

    nn::fs::SaveDataId saveDataId = 0;

    nn::time::PosixTime beforeCreate;
    NNT_EXPECT_RESULT_SUCCESS(nn::time::StandardUserSystemClock::GetCurrentTime(&beforeCreate));
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(2000));

    for (int i = 1; i <= SaveDataNum; i++)
    {
        auto result = FindSaveDataId(&saveDataId, appId, GetUserIdByIndex(i));
        if (result.IsSuccess())
        {
            nn::fs::DeleteSaveData(saveDataId);
        }

        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSaveData(appId, GetUserIdByIndex(i), GetOwnerIdByIndex(i), SaveDataSize, SaveDataJournalSize, GetFlagsByIndex(i)));
    }

    // 拡張データの取得及び設定
    for (int i = 1; i <= SaveDataNum; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(FindSaveDataId(&saveDataId, appId, GetUserIdByIndex(i)));

        {
            nn::Bit64 tmpOwnerId = 0;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetSaveDataOwnerId(&tmpOwnerId, saveDataId));
            EXPECT_EQ(tmpOwnerId, GetOwnerIdByIndex(i));

            uint32_t tmpFlags = 0;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetSaveDataFlags(&tmpFlags, saveDataId));
            EXPECT_EQ(tmpFlags, GetFlagsByIndex(i));

            nn::time::PosixTime saveDataTimeStamp = { -1 };
            {
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetSaveDataTimeStamp(&saveDataTimeStamp, saveDataId));
                EXPECT_GE(saveDataTimeStamp.value, beforeCreate.value);
            }

            {
                uint32_t flags = tmpFlags + 1024;
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::SetSaveDataFlags(saveDataId, nn::fs::SaveDataSpaceId::User, flags));
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetSaveDataFlags(&tmpFlags, saveDataId));
                EXPECT_EQ(tmpFlags, flags);

                NNT_EXPECT_RESULT_SUCCESS(nn::fs::SetSaveDataFlags(saveDataId, nn::fs::SaveDataSpaceId::User, GetFlagsByIndex(i)));
            }

            // SetFlags でタイムスタンプは更新されない
            {
                nn::time::PosixTime  tmpTimeStamp = { -1 };
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetSaveDataTimeStamp(&tmpTimeStamp, saveDataId));

                EXPECT_EQ(saveDataTimeStamp.value, tmpTimeStamp.value);
            }

            // SetTimeStamp
            {
                nn::time::PosixTime  tmpTimeStamp = { -1 };
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::SetSaveDataTimeStamp(nn::fs::SaveDataSpaceId::User, saveDataId, tmpTimeStamp));
                tmpTimeStamp.value = 0;
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetSaveDataTimeStamp(&tmpTimeStamp, saveDataId));
                EXPECT_EQ(tmpTimeStamp.value, -1);
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::SetSaveDataTimeStamp(nn::fs::SaveDataSpaceId::User, saveDataId, saveDataTimeStamp));
            }

            int64_t tmpAvailableSize = 0;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetSaveDataAvailableSize(&tmpAvailableSize, saveDataId));
            EXPECT_EQ(tmpAvailableSize, static_cast<int64_t>(SaveDataSize));

            int64_t tmpJournalSize = 0;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetSaveDataJournalSize(&tmpJournalSize, saveDataId));
            EXPECT_EQ(tmpJournalSize, static_cast<int64_t>(SaveDataJournalSize));
        }
    }

    // 全セーブデータをマウント
    for (int i = 1; i <= SaveDataNum; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountSaveData(GetMountNameByIndex(i).c_str(), appId, GetUserIdByIndex(i)));
    }

    // 二重マウントはできない
    for (int i = 1; i <= SaveDataNum; i++)
    {
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetLocked, nn::fs::MountSaveData(GetDuplexMountNameByIndex(i).c_str(), appId, GetUserIdByIndex(i)));
    }

    // マウント中のセーブデータの拡張データの取得
    for (int i = 1; i <= SaveDataNum; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(FindSaveDataId(&saveDataId, appId, GetUserIdByIndex(i)));

        {
            nn::Bit64 tmpOwnerId = 0;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetSaveDataOwnerId(&tmpOwnerId, saveDataId));
            EXPECT_EQ(tmpOwnerId, GetOwnerIdByIndex(i));

            uint32_t tmpFlags = 0;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetSaveDataFlags(&tmpFlags, saveDataId));
            EXPECT_EQ(tmpFlags, GetFlagsByIndex(i));

            nn::time::PosixTime saveDataTimeStamp = { -1 };
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetSaveDataTimeStamp(&saveDataTimeStamp, saveDataId));

            {
                uint32_t flags = tmpFlags + 1024;
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::SetSaveDataFlags(saveDataId, nn::fs::SaveDataSpaceId::User, flags));
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetSaveDataFlags(&tmpFlags, saveDataId));
                EXPECT_EQ(tmpFlags, flags);

                NNT_EXPECT_RESULT_SUCCESS(nn::fs::SetSaveDataFlags(saveDataId, nn::fs::SaveDataSpaceId::User, GetFlagsByIndex(i)));
            }

            // SetFlags でタイムスタンプは更新されない
            {
                nn::time::PosixTime  tmpTimeStamp = { -1 };
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetSaveDataTimeStamp(&tmpTimeStamp, saveDataId));

                EXPECT_EQ(saveDataTimeStamp.value, tmpTimeStamp.value);
            }

            // SetTimeStamp
            {
                nn::time::PosixTime  tmpTimeStamp = { -1 };
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::SetSaveDataTimeStamp(nn::fs::SaveDataSpaceId::User, saveDataId, tmpTimeStamp));
                tmpTimeStamp.value = 0;
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetSaveDataTimeStamp(&tmpTimeStamp, saveDataId));
                EXPECT_EQ(tmpTimeStamp.value, -1);
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::SetSaveDataTimeStamp(nn::fs::SaveDataSpaceId::User, saveDataId, saveDataTimeStamp));
            }

            int64_t tmpAvailableSize = 0;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetSaveDataAvailableSize(&tmpAvailableSize, saveDataId));
            EXPECT_EQ(tmpAvailableSize, static_cast<int64_t>(SaveDataSize));

            int64_t tmpJournalSize = 0;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetSaveDataJournalSize(&tmpJournalSize, saveDataId));
            EXPECT_EQ(tmpJournalSize, static_cast<int64_t>(SaveDataJournalSize));
        }
    }

    // 全セーブデータをアンマウントして削除
    for (int i = 1; i <= SaveDataNum; i++)
    {
        nn::fs::Unmount(GetMountNameByIndex(i).c_str());

        NNT_EXPECT_RESULT_SUCCESS(FindSaveDataId(&saveDataId, appId, GetUserIdByIndex(i)));

        // アンマウント後のセーブデータの拡張データの取得
        {
            nn::Bit64 tmpOwnerId = 0;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetSaveDataOwnerId(&tmpOwnerId, saveDataId));
            EXPECT_EQ(tmpOwnerId, GetOwnerIdByIndex(i));

            uint32_t tmpFlags = 0;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetSaveDataFlags(&tmpFlags, saveDataId));
            EXPECT_EQ(tmpFlags, GetFlagsByIndex(i));

            nn::time::PosixTime  tmpTimeStamp = { -1 };
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetSaveDataTimeStamp(&tmpTimeStamp, saveDataId));
            EXPECT_NE(tmpTimeStamp.value, 0);

            {
                uint32_t flags = tmpFlags + 1024;
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::SetSaveDataFlags(saveDataId, nn::fs::SaveDataSpaceId::User, flags));
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetSaveDataFlags(&tmpFlags, saveDataId));
                EXPECT_EQ(tmpFlags, flags);
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::SetSaveDataFlags(saveDataId, nn::fs::SaveDataSpaceId::User, GetFlagsByIndex(i)));
            }

            int64_t tmpAvailableSize = 0;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetSaveDataAvailableSize(&tmpAvailableSize, saveDataId));
            EXPECT_EQ(tmpAvailableSize, static_cast<int64_t>(SaveDataSize));

            int64_t tmpJournalSize = 0;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetSaveDataJournalSize(&tmpJournalSize, saveDataId));
            EXPECT_EQ(tmpJournalSize, static_cast<int64_t>(SaveDataJournalSize));
        }

        NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteSaveData(saveDataId));

        // 存在しないセーブデータの拡張データ取得は失敗する
        {
            nn::Bit64 tmpOwnerId = 0;
            NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::GetSaveDataOwnerId(&tmpOwnerId, saveDataId));

            uint32_t tmpFlags = 0;
            NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::GetSaveDataFlags(&tmpFlags, saveDataId));
            NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::SetSaveDataFlags(saveDataId, nn::fs::SaveDataSpaceId::User, tmpFlags));

            nn::time::PosixTime  tmpTimeStamp = { -1 };
            NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::GetSaveDataTimeStamp(&tmpTimeStamp, saveDataId));
            NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::SetSaveDataTimeStamp(nn::fs::SaveDataSpaceId::User, saveDataId, tmpTimeStamp));

            int64_t tmpAvailableSize = 0;
            NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::GetSaveDataAvailableSize(&tmpAvailableSize, saveDataId));

            int64_t tmpJournalSize = 0;
            NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::GetSaveDataJournalSize(&tmpJournalSize, saveDataId));
        }
    }
} // NOLINT(impl/function_size)



TEST(SaveDataExtraData, CommitId)
{
    const size_t SaveDataSize = 1024 * 1024;
    const size_t SaveDataJournalSize = 1024 * 1024;

    nnt::fs::util::DeleteAllTestSaveData();

    nn::ncm::ApplicationId appId = { GetSaveDataApplicationId() };
    auto userId = nnt::fs::util::TestUserId0;

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSaveData(appId, userId, appId.value, SaveDataSize, SaveDataJournalSize, 0));
    SaveDataInfo info;
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::FindSaveDataByApplicationId(&info, appId, userId));

    SaveDataExtraData extraDataCreate;
    NNT_EXPECT_RESULT_SUCCESS(detail::ReadSaveDataFileSystemExtraData(&extraDataCreate, info.saveDataSpaceId, info.saveDataId));
    EXPECT_NE(0, extraDataCreate.commitId);

    SaveDataExtraData extraDataCommit1;

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountSaveData("save", appId, userId));

    //（空コミットでは更新されないこと）
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::Commit("save"));
    NNT_EXPECT_RESULT_SUCCESS(detail::ReadSaveDataFileSystemExtraData(&extraDataCommit1, info.saveDataSpaceId, info.saveDataId));
    // EXPECT_EQ(extraDataCreate.commitId, extraDataCommit1.commitId); // 現在更新される

    // コミットで更新されること
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CreateFileWith8BitCount("save:/test", 32, 1));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::Commit("save"));
    NNT_EXPECT_RESULT_SUCCESS(detail::ReadSaveDataFileSystemExtraData(&extraDataCommit1, info.saveDataSpaceId, info.saveDataId));
    EXPECT_NE(extraDataCreate.commitId, extraDataCommit1.commitId);
    EXPECT_NE(0,                          extraDataCommit1.commitId);

    // コミットごとに異なること
    SaveDataExtraData extraDataCommit2;
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CreateFileWith8BitCount("save:/test2", 32, 1));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::Commit("save"));
    NNT_EXPECT_RESULT_SUCCESS(detail::ReadSaveDataFileSystemExtraData(&extraDataCommit2, info.saveDataSpaceId, info.saveDataId));
    EXPECT_NE(extraDataCreate.commitId,  extraDataCommit2.commitId);
    EXPECT_NE(extraDataCommit1.commitId, extraDataCommit2.commitId);
    EXPECT_NE(0,                           extraDataCommit2.commitId);

    // Set で任意の値に設定し、Get で取得できること
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::SetSaveDataCommitId(info.saveDataSpaceId, info.saveDataId, extraDataCommit1.commitId));
    SaveDataCommitId tmpCommitId = 0;
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetSaveDataCommitId(&tmpCommitId, info.saveDataSpaceId, info.saveDataId));
    EXPECT_EQ(tmpCommitId, extraDataCommit1.commitId);

    Unmount("save");
    nnt::fs::util::DeleteAllTestSaveData();
}


// GetSaveDataCommitId で取れる値のテスト
TEST(CloudBackup, GetSaveDataCommitId)
{
    nnt::fs::util::DeleteAllTestSaveData();

    SaveDataInfo info;
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CreateAndFindSaveData(&info));

    SaveDataExtraData extraDataCreate;
    NNT_EXPECT_RESULT_SUCCESS(detail::ReadSaveDataFileSystemExtraData(&extraDataCreate, info.saveDataSpaceId, info.saveDataId));

    // 通常は extraData.commitId と一致すること
    {
        SaveDataCommitId commitId;
        NNT_EXPECT_RESULT_SUCCESS(GetSaveDataCommitId(&commitId, info.saveDataSpaceId, info.saveDataId));
        EXPECT_EQ(extraDataCreate.commitId, commitId);
    }


    // extraData.commitId が 0 埋めの場合（既存セーブデータ）は別の値（セーブデータのヘッダのハッシュ）が取得できること
    {
        SaveDataExtraData extraDataHashZero = extraDataCreate;
        extraDataHashZero.commitId = 0;
        SaveDataExtraData extraDataHashMask;
        memset(&extraDataHashMask, 0, sizeof(extraDataHashMask));
        memset(&extraDataHashMask.commitId, 0xFF, sizeof(extraDataHashMask.commitId));
        NNT_EXPECT_RESULT_SUCCESS(detail::WriteSaveDataFileSystemExtraData(info.saveDataSpaceId, info.saveDataId, extraDataHashZero, extraDataHashMask));

        {
            // extraData.commitId == 0 になっていること
            SaveDataExtraData extraData;
            NNT_EXPECT_RESULT_SUCCESS(detail::ReadSaveDataFileSystemExtraData(&extraData, info.saveDataSpaceId, info.saveDataId));
            EXPECT_EQ(0, extraData.commitId);
        }

        {
            // GetSaveDataCommitId 経由では別の値
            SaveDataCommitId commitId;
            NNT_EXPECT_RESULT_SUCCESS(GetSaveDataCommitId(&commitId, info.saveDataSpaceId, info.saveDataId));
            EXPECT_NE(0, commitId);
        }
    }

    // コミット後は非零が取得でき一致すること
    {
        SaveDataExtraData extraData;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountSaveData("save", info.applicationId, info.saveDataUserId));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CreateFileWith8BitCount("save:/test", 32, 1));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::Commit("save"));

        NNT_EXPECT_RESULT_SUCCESS(detail::ReadSaveDataFileSystemExtraData(&extraData, info.saveDataSpaceId, info.saveDataId));
        EXPECT_NE(extraDataCreate.commitId, extraData.commitId);
        EXPECT_NE(0,                          extraData.commitId);

        SaveDataCommitId commitId;
        NNT_EXPECT_RESULT_SUCCESS(GetSaveDataCommitId(&commitId, info.saveDataSpaceId, info.saveDataId));
        EXPECT_EQ(extraData.commitId, commitId);

        Unmount("save");
    }

    nnt::fs::util::DeleteAllTestSaveData();
}


// マウント中でも GetSaveDataCommitId() できること
TEST(SaveDataExtraData, GetSaveDataCommitIdOnMount)
{
    nnt::fs::util::DeleteAllTestSaveData();

    SaveDataInfo info;
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CreateAndFindSaveData(&info));

    // 通常
    {
        NNT_EXPECT_RESULT_SUCCESS(MountSaveData("save", info.applicationId, info.saveDataUserId));
        SaveDataExtraData extraDataCreate;
        NNT_EXPECT_RESULT_SUCCESS(detail::ReadSaveDataFileSystemExtraData(&extraDataCreate, info.saveDataSpaceId, info.saveDataId));
        Unmount("save");
    }

    // extraData.commitId が 0 埋めの場合
    {
        SaveDataExtraData extraDataCreate;
        NNT_EXPECT_RESULT_SUCCESS(detail::ReadSaveDataFileSystemExtraData(&extraDataCreate, info.saveDataSpaceId, info.saveDataId));
        SaveDataExtraData extraDataHashZero = extraDataCreate;
        extraDataHashZero.commitId = 0;
        SaveDataExtraData extraDataHashMask;
        memset(&extraDataHashMask, 0, sizeof(extraDataHashMask));
        memset(&extraDataHashMask.commitId, 0xFF, sizeof(extraDataHashMask.commitId));
        NNT_EXPECT_RESULT_SUCCESS(detail::WriteSaveDataFileSystemExtraData(info.saveDataSpaceId, info.saveDataId, extraDataHashZero, extraDataHashMask));

        {
            NNT_EXPECT_RESULT_SUCCESS(MountSaveData("save", info.applicationId, info.saveDataUserId));

            {
                // extraData.commitId == 0 になっていること
                SaveDataExtraData extraData;
                NNT_EXPECT_RESULT_SUCCESS(detail::ReadSaveDataFileSystemExtraData(&extraData, info.saveDataSpaceId, info.saveDataId));
                EXPECT_EQ(0, extraData.commitId);
            }

            {
                // GetSaveDataCommitId 経由では別の値
                SaveDataCommitId commitId;
                NNT_EXPECT_RESULT_SUCCESS(GetSaveDataCommitId(&commitId, info.saveDataSpaceId, info.saveDataId));
                EXPECT_NE(0, commitId);
            }

            Unmount("save");
        }
    }

    nnt::fs::util::DeleteAllTestSaveData();
}


TEST(SaveDataExtraData, ProperSystemOnNormalMode)
{
    // 通常起動では ProperSystem 指定が使用できないこと
    const SystemSaveDataId Id = 0x8000000000000003;

    NNT_EXPECT_RESULT_FAILURE(ResultInvalidArgument, CreateSystemSaveData(SaveDataSpaceId::ProperSystem, Id, GetUserIdByIndex(0), GetOwnerIdByIndex(0), 0x100000, 0x100000, 0));
    NNT_EXPECT_RESULT_FAILURE(ResultInvalidArgument, MountSystemSaveData("invalid", SaveDataSpaceId::ProperSystem, Id));

    NNT_EXPECT_RESULT_FAILURE(ResultInvalidArgument, DeleteSaveData(SaveDataSpaceId::ProperSystem, Id));

    std::unique_ptr<SaveDataIterator> iter;
    NNT_EXPECT_RESULT_FAILURE(ResultInvalidArgument, OpenSaveDataIterator(&iter, SaveDataSpaceId::ProperSystem));

    uint32_t flags;
    NNT_EXPECT_RESULT_FAILURE(ResultInvalidArgument, GetSaveDataFlags(&flags, SaveDataSpaceId::ProperSystem, Id));

    int64_t size;
    NNT_EXPECT_RESULT_FAILURE(ResultInvalidArgument, GetSaveDataAvailableSize(&size, SaveDataSpaceId::ProperSystem, Id));
    NNT_EXPECT_RESULT_FAILURE(ResultInvalidArgument, GetSaveDataJournalSize(&size, SaveDataSpaceId::ProperSystem, Id));

    nn::Bit64 ownerId;
    NNT_EXPECT_RESULT_FAILURE(ResultInvalidArgument, GetSaveDataOwnerId(&ownerId, SaveDataSpaceId::ProperSystem, Id));

    nn::time::PosixTime time;
    NNT_EXPECT_RESULT_FAILURE(ResultInvalidArgument, GetSaveDataTimeStamp(&time, SaveDataSpaceId::ProperSystem, Id));
}

// SaveDataFileSystem のキャッシュ機能との兼ね合いをチェックする
TEST(SaveDataExtraData, ModifyCachedSaveDataFileSystem)
{
    const size_t SaveDataSize = 1024 * 1024;
    const size_t SaveDataJournalSize = 1024 * 1024;

    nn::ncm::ApplicationId appId = { GetSaveDataApplicationId() };

    nn::fs::SaveDataId saveDataId = 0;

    auto result = FindSaveDataId(&saveDataId, appId, GetUserIdByIndex(1));
    if (result.IsSuccess())
    {
        (void)nn::fs::DeleteSaveData(saveDataId);
    }

    NNT_EXPECT_RESULT_SUCCESS(
        nn::fs::CreateSaveData(
            appId,
            GetUserIdByIndex(1),
            GetOwnerIdByIndex(1),
            SaveDataSize,
            SaveDataJournalSize,
            GetFlagsByIndex(1)
        )
    );
    FindSaveDataId(&saveDataId, appId, GetUserIdByIndex(1));

    uint32_t SaveDataFlag1 = 0x5555;
    uint32_t SaveDataFlag2 = 0xAAAA;
    uint32_t SaveDataFlag3 = 0xCCCC;

    // マウントし、拡張データを書き込み、セーブデータを更新する
    NNT_EXPECT_RESULT_SUCCESS(
        nn::fs::MountSaveData(GetMountNameByIndex(1).c_str(), appId, GetUserIdByIndex(1))
    );
    NNT_EXPECT_RESULT_SUCCESS(
        nn::fs::SetSaveDataFlags(saveDataId, nn::fs::SaveDataSpaceId::User, SaveDataFlag1)
    );
    NNT_EXPECT_RESULT_SUCCESS(
        nn::fs::CreateFile((GetMountNameByIndex(1) + ":/file.txt").c_str(), 0x1000 )
    );

    // 上書きして Unmount する
    NNT_EXPECT_RESULT_SUCCESS(
        nn::fs::SetSaveDataFlags(saveDataId, nn::fs::SaveDataSpaceId::User, SaveDataFlag2)
    );
    nn::fs::Unmount(GetMountNameByIndex(1).c_str());

    // Unmount された状態でさらに上書きする
    NNT_EXPECT_RESULT_SUCCESS(
        nn::fs::SetSaveDataFlags(saveDataId, nn::fs::SaveDataSpaceId::User, SaveDataFlag3)
    );

    // 再度マウントし、この時点での拡張データの内容が想定通りかどうかチェックする
    NNT_EXPECT_RESULT_SUCCESS(
        nn::fs::MountSaveData(GetMountNameByIndex(1).c_str(), appId, GetUserIdByIndex(1))
    );

    nn::fs::Unmount(GetMountNameByIndex(1).c_str());

    uint32_t saveDataFlag = 0;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::fs::GetSaveDataFlags(&saveDataFlag, saveDataId)
    );
    ASSERT_EQ(SaveDataFlag3, saveDataFlag);

    (void)nn::fs::DeleteSaveData(saveDataId);
}



TEST(SaveDataExtraData, WriteSaveDataFileSystemExtraDataMask)
{
    nnt::fs::util::DeleteAllTestSaveData();

    const SystemSaveDataId Id = 0x8000000000004000;
    NNT_ASSERT_RESULT_SUCCESS(CreateSystemSaveData(Id, GetOwnerIdByIndex(0), 0x100000, 0x100000, 0));

    // 従来のマスク無し WriteSaveDataFileSystemExtraData() は flag のみマスク指定扱い
    {
        nn::fs::SaveDataExtraData extraDataBefore;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::detail::ReadSaveDataFileSystemExtraData(&extraDataBefore, SaveDataSpaceId::System, Id));

        nn::fs::SaveDataExtraData extraDataRequest;
        nnt::fs::util::FillBufferWithRandomValue(&extraDataRequest, sizeof(extraDataRequest));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::detail::WriteSaveDataFileSystemExtraData(SaveDataSpaceId::System, Id, extraDataRequest));

        nn::fs::SaveDataExtraData extraDataAfter;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::detail::ReadSaveDataFileSystemExtraData(&extraDataAfter, SaveDataSpaceId::System, Id));

        // flag は書き換わる
        EXPECT_NE(extraDataBefore.flags,  extraDataAfter.flags);
        EXPECT_EQ(extraDataRequest.flags, extraDataAfter.flags);

        // 他のフィールドは before のまま
        extraDataAfter.flags = extraDataBefore.flags;
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(&extraDataBefore, &extraDataAfter, sizeof(SaveDataExtraData));
    }

    // マスク指定で指定箇所のみが書き変わること
    {
        nn::fs::SaveDataExtraData extraDataAllF;
        memset(&extraDataAllF, 0xFF, sizeof(SaveDataExtraData));

        nn::fs::SaveDataExtraData extraDataAllA;
        memset(&extraDataAllA, 0xAA, sizeof(SaveDataExtraData));

        nn::fs::SaveDataExtraData extraDataAll0;
        memset(&extraDataAll0, 0x00, sizeof(SaveDataExtraData));

        nn::fs::SaveDataExtraData extraDataRequest;
        nn::fs::SaveDataExtraData extraDataMask;
        nn::fs::SaveDataExtraData extraDataAfter;

        // ランダム埋め
        nnt::fs::util::FillBufferWithRandomValue(&extraDataRequest, sizeof(extraDataRequest));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::detail::WriteSaveDataFileSystemExtraData(SaveDataSpaceId::System, Id, extraDataRequest, extraDataAllF));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::detail::ReadSaveDataFileSystemExtraData(&extraDataAfter, SaveDataSpaceId::System, Id));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(&extraDataRequest, &extraDataAfter, sizeof(SaveDataExtraData));

        // mask 0 指定: 効果無し
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::detail::WriteSaveDataFileSystemExtraData(SaveDataSpaceId::System, Id, extraDataAllF, extraDataAll0));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::detail::ReadSaveDataFileSystemExtraData(&extraDataAfter, SaveDataSpaceId::System, Id));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(&extraDataRequest, &extraDataAfter, sizeof(SaveDataExtraData));


        // mask FF 指定: 全書き換え
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::detail::WriteSaveDataFileSystemExtraData(SaveDataSpaceId::System, Id, extraDataAllF, extraDataAllF)); // all F
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::detail::ReadSaveDataFileSystemExtraData(&extraDataAfter, SaveDataSpaceId::System, Id));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(&extraDataAllF, &extraDataAfter, sizeof(SaveDataExtraData));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::detail::WriteSaveDataFileSystemExtraData(SaveDataSpaceId::System, Id, extraDataAllA, extraDataAllF)); // all A
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::detail::ReadSaveDataFileSystemExtraData(&extraDataAfter, SaveDataSpaceId::System, Id));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(&extraDataAllA, &extraDataAfter, sizeof(SaveDataExtraData));

        // allA -> mask
        extraDataRequest = extraDataAllF;
        const nn::ncm::ProgramId programId = { 0x5555555544444444ULL };
        extraDataRequest.attribute.programId = programId;
        extraDataRequest.reserved[sizeof(extraDataRequest.reserved) - 1] = 0x33;

        extraDataMask = extraDataAll0;
        extraDataMask.attribute.programId.value = 0xFFFFFFFFFFFFFFFFULL;
        extraDataMask.reserved[sizeof(extraDataMask.reserved) - 1] = 0xFF;

        NNT_EXPECT_RESULT_SUCCESS(nn::fs::detail::WriteSaveDataFileSystemExtraData(SaveDataSpaceId::System, Id, extraDataRequest, extraDataMask));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::detail::ReadSaveDataFileSystemExtraData(&extraDataAfter, SaveDataSpaceId::System, Id));

        // マスク指定箇所のみ書き換わる
        EXPECT_EQ(programId, extraDataAfter.attribute.programId);
        EXPECT_EQ(0x33,      extraDataAfter.reserved[sizeof(extraDataAfter.reserved) - 1]);

        // それ以外は維持 = all A
        extraDataAfter.attribute.programId.value = 0xAAAAAAAAAAAAAAAAULL;
        extraDataAfter.reserved[sizeof(extraDataAfter.reserved) - 1] = 0xAA;
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(&extraDataAllA, &extraDataAfter, sizeof(SaveDataExtraData));
    }

    nnt::fs::util::DeleteAllTestSaveData();
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nn::fs::SetEnabledAutoAbort(false);
    nnt::fs::util::ResetAllocateCount();
    nn::time::Initialize();

    auto result = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(result);
}
