﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_SystemSaveData.h>

using namespace nn;

namespace {

const int MaxPathLength = 256;
const char HostSaveDataPath[] = {"D:/TempSave"};

const int BufferSize = 8 * 1024 * 1024;

}

void CopyFile(const char* srcFile, const char* destFile)
{
    nn::fs::FileHandle srcHandle;
    nn::fs::FileHandle destHandle;
    int64_t fileSize = 0;

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&srcHandle, srcFile, nn::fs::OpenMode_Read));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, srcHandle));

    nn::fs::CreateFile(destFile, fileSize);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&destHandle, destFile, nn::fs::OpenMode_Write));

    auto buffer = nnt::fs::util::AllocateBuffer(BufferSize);

    int64_t offset = 0;
    while (offset < fileSize)
    {
        int readSize = ((fileSize - offset) > BufferSize) ? BufferSize : (fileSize - offset);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(srcHandle, offset, buffer.get(), readSize));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(destHandle, offset, buffer.get(), readSize, nn::fs::WriteOption::MakeValue(0)));
        offset += readSize;
    }
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(destHandle));
    nn::fs::CloseFile(destHandle);
    nn::fs::CloseFile(srcHandle);
}

void CopyDir(const char* srcDir, const char* destDir)
{
    nn::fs::DirectoryHandle srcHandle;
    char srcPath[256];
    char destPath[256];
    nn::fs::DirectoryEntry entryBuffer;

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(&srcHandle, srcDir, nn::fs::OpenDirectoryMode_Directory));
    while (1)
    {
        int64_t outValue = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadDirectory(&outValue, &entryBuffer, srcHandle, 1));
        if (outValue == 0)
        {
            break;
        }

        nn::util::SNPrintf(srcPath, sizeof(srcPath), "%s/%s", srcDir, entryBuffer.name);
        nn::util::SNPrintf(destPath, sizeof(destPath), "%s/%s", destDir, entryBuffer.name);

        nn::fs::CreateDirectory(destPath);
        CopyDir(srcPath, destPath);
    }
    nn::fs::CloseDirectory(srcHandle);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(&srcHandle, srcDir, nn::fs::OpenDirectoryMode_File));
    while (1)
    {
        int64_t outValue = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadDirectory(&outValue, &entryBuffer, srcHandle, 1));
        if (outValue == 0)
        {
            break;
        }

        nn::util::SNPrintf(srcPath, sizeof(srcPath), "%s/%s", srcDir, entryBuffer.name);
        nn::util::SNPrintf(destPath, sizeof(destPath), "%s/%s", destDir, entryBuffer.name);

        CopyFile(srcPath, destPath);
    }
    nn::fs::CloseDirectory(srcHandle);
}

bool ExistsDirectory(const char* path)
{
    nn::fs::DirectoryEntryType outValue;
    auto result = nn::fs::GetEntryType(&outValue, path);
    if (result.IsFailure())
    {
        return false;
    }
    return true;
}

void CleanSaveDataCache()
{
    /* SaveDataFs のキャッシュ追い出し */
    nn::fs::MountSystemSaveData("save", 0x8000000000000000);
    nn::fs::Unmount("save");
}

void RestoreRawSaveDataCore(const char* srcBasePath, nn::fs::RawSaveDataInfo* pRawInfo, nn::fs::SaveDataInfo* pInfo)
{
    CleanSaveDataCache();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis(nn::fs::BisPartitionId::User, nullptr));

    char srcPath[MaxPathLength];
    char dstPath[MaxPathLength];
    nn::util::SNPrintf(srcPath, sizeof(srcPath), "%ssave/%016llx", srcBasePath, pRawInfo->id);
    nn::util::SNPrintf(dstPath, sizeof(dstPath), "%s:/save/%016llx", nn::fs::GetBisMountName(nn::fs::BisPartitionId::User), pInfo->saveDataId);
//    NN_LOG("CopyFile : %s -> %s\n", srcPath, dstPath);

    nn::fs::DeleteFile(dstPath);
    CopyFile(srcPath, dstPath);

    nn::util::SNPrintf(srcPath, sizeof(srcPath), "%ssaveMeta/%016llx", srcBasePath, pRawInfo->id);
    if (ExistsDirectory(srcPath))
    {
        nn::util::SNPrintf(dstPath, sizeof(dstPath), "%s:/saveMeta/%016llx", nn::fs::GetBisMountName(nn::fs::BisPartitionId::User), pInfo->saveDataId);
//        NN_LOG("CopyDir : %s -> %s\n", srcPath, dstPath);

        nn::fs::DeleteDirectoryRecursively(dstPath);
        nn::fs::CreateDirectory(dstPath);
        CopyDir(srcPath, dstPath);
    }
    nn::fs::Unmount(nn::fs::GetBisMountName(nn::fs::BisPartitionId::User));

    nn::fs::UpdateSaveDataMacForDebug(static_cast<uint8_t>(pInfo->saveDataSpaceId), pInfo->saveDataId);
}

void RestoreRawSaveData(const char* srcBaseDir)
{
    char infoFile[MaxPathLength];
    nn::util::SNPrintf(infoFile, sizeof(infoFile), "%s%s", srcBaseDir, nn::fs::RawSaveDataInfoFileName);

    nn::fs::RawSaveDataInfo saveDataInfo;
    nn::fs::FileHandle fileHandle;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, infoFile, nn::fs::OpenMode_Read));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, 0, &saveDataInfo, sizeof(saveDataInfo)));
    nn::fs::CloseFile(fileHandle);

    if ((saveDataInfo.type == nn::fs::SaveDataType::Account) || (saveDataInfo.type == nn::fs::SaveDataType::Device))
    {
        nnt::fs::util::Vector<nn::fs::SaveDataInfo> infoArray;
        nnt::fs::util::FindSaveData(&infoArray, nn::fs::SaveDataSpaceId::User,
            [=](const nn::fs::SaveDataInfo& info) { return (info.saveDataType == saveDataInfo.type) && (info.applicationId == saveDataInfo.applicationId) && (info.saveDataUserId == saveDataInfo.userId); }
        );
        if (infoArray.size() > 0)
        {
            NN_LOG("Already exists SaveData (Type=%02x, AppliId=%016llx, UserId=%016llx%016llx)\n", saveDataInfo.type, saveDataInfo.applicationId.value, saveDataInfo.userId._data[0], saveDataInfo.userId._data[1]);
            return;
        }

        if (saveDataInfo.type == nn::fs::SaveDataType::Account)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(saveDataInfo.applicationId, saveDataInfo.userId, saveDataInfo.ownerId, saveDataInfo.availableSize, saveDataInfo.journalSize, saveDataInfo.flags));
        }
        else if (saveDataInfo.type == nn::fs::SaveDataType::Device)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDeviceSaveData(saveDataInfo.applicationId, saveDataInfo.ownerId, saveDataInfo.availableSize, saveDataInfo.journalSize, saveDataInfo.flags));
        }

        nnt::fs::util::FindSaveData(&infoArray, nn::fs::SaveDataSpaceId::User,
            [=](const nn::fs::SaveDataInfo& info) { return (info.saveDataType == saveDataInfo.type) && (info.applicationId == saveDataInfo.applicationId) && (info.saveDataUserId == saveDataInfo.userId); }
        );
        if (infoArray.size() == 1)
        {
            RestoreRawSaveDataCore(srcBaseDir, &saveDataInfo, &infoArray[0]);
#if 0 // Mount Check
            if (saveDataInfo.type == nn::fs::SaveDataType::Account)
            {
                nn::fs::MountSaveData("save", saveDataInfo.applicationId, saveDataInfo.userId);
            }
            else if (saveDataInfo.type == nn::fs::SaveDataType::Device)
            {
                nn::fs::MountDeviceSaveData("save", saveDataInfo.applicationId);
            }
            nnt::fs::util::DumpDirectoryRecursive("save:/");
            nn::fs::Unmount("save");
#endif
        }
        else
        {
             NN_LOG("Find %d SaveData (Type=%02x, AppliId=%016llx, UserId=%016llx%016llx)\n", infoArray.size(), saveDataInfo.type, saveDataInfo.applicationId.value, saveDataInfo.userId._data[0], saveDataInfo.userId._data[1]);
        }
    }
}

void SearchRawSaveDataDirectory(const char* srcBaseDir)
{
    nn::fs::DirectoryHandle srcHandle;
    nn::fs::DirectoryEntry entryBuffer;

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(&srcHandle, srcBaseDir, nn::fs::OpenDirectoryMode_Directory));
    while (1)
    {
        int64_t outValue = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadDirectory(&outValue, &entryBuffer, srcHandle, 1));
        if (outValue == 0)
        {
            break;
        }
        if (entryBuffer.name[0] == '#')
        {
            continue;
        }

        char srcDir[MaxPathLength];
        nn::util::SNPrintf(srcDir, sizeof(srcDir), "%s%s/", srcBaseDir, entryBuffer.name);
        SearchRawSaveDataDirectory(srcDir);
    }
    nn::fs::CloseDirectory(srcHandle);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(&srcHandle, srcBaseDir, nn::fs::OpenDirectoryMode_File));
    while (1)
    {
        int64_t outValue = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadDirectory(&outValue, &entryBuffer, srcHandle, 1));
        if (outValue == 0)
        {
            break;
        }
        if (std::strncmp(entryBuffer.name, nn::fs::RawSaveDataInfoFileName, MaxPathLength) != 0)
        {
            continue;
        }

        RestoreRawSaveData(srcBaseDir);
    }
    nn::fs::CloseDirectory(srcHandle);
}

void TestSaveData(const char* dstMountName)
{
    char dstDir[MaxPathLength];
    nn::util::SNPrintf(dstDir, sizeof(dstDir), "%s:/", dstMountName);
    SearchRawSaveDataDirectory(dstDir);
}

TEST(SaveData, TestSaveDataRestore)
{
//    std::mt19937 random(nnt::fs::util::GetRandomSeed());

    const char* MountName = "host";
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost(MountName, HostSaveDataPath));

    TestSaveData(MountName);

    nn::fs::Unmount(MountName);
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();

    auto result = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(result);
}
