﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SystemSaveData.h>

namespace nn { namespace fs {
    Result MountSaveDataInternalStorage(const char* name, SaveDataSpaceId spaceId, SaveDataId saveDataId) NN_NOEXCEPT;
}}

using namespace nn;

/* 既存のセーブデータをダンプ */
/*
 * HostSaveDataPath 以下にディレクトリ名が <saveDataId> のディレクトリを作成する。
 * <saveDataId> ディレクトリ下に、ディレクトリ名 "0" のディレクトリ名を作成。（既にあれば数値をインクリメント）
 * <saveDataId>/<Count> ディレクトリ下にセーブデータイメージファイルを
 * "AllocationTableControlArea", "AllocationTableMeta", "AllocationTableData", "Raw" の名前で作成する。
 */

namespace {

const int MaxPathLength = 256;
const char HostSaveDataPath[] = {"D:/TempSave"};

const int BufferSize = 8 * 1024 * 1024;

}

void CopyFile(const char* srcFile, const char* destFile)
{
    nn::fs::FileHandle srcHandle;
    nn::fs::FileHandle destHandle;
    int64_t fileSize = 0;

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&srcHandle, srcFile, nn::fs::OpenMode_Read));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, srcHandle));

    nn::fs::CreateFile(destFile, fileSize);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&destHandle, destFile, nn::fs::OpenMode_Write));

    auto buffer = nnt::fs::util::AllocateBuffer(BufferSize);

    int64_t offset = 0;
    while (offset < fileSize)
    {
        int readSize = ((fileSize - offset) > BufferSize) ? BufferSize : (fileSize - offset);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(srcHandle, offset, buffer.get(), readSize));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(destHandle, offset, buffer.get(), readSize, nn::fs::WriteOption::MakeValue(0)));
        offset += readSize;
    }
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(destHandle));
    nn::fs::CloseFile(destHandle);
    nn::fs::CloseFile(srcHandle);
}

bool ExistsDirectory(const char* path)
{
    nn::fs::DirectoryHandle outValue;
    auto result = nn::fs::OpenDirectory(&outValue, path, nn::fs::OpenDirectoryMode_All);
    if (result.IsFailure())
    {
        return false;
    }
    nn::fs::CloseDirectory(outValue);
    return true;
}

void CleanSaveDataCache()
{
    /* SaveDataFs のキャッシュ追い出し */
    nn::fs::MountSystemSaveData("save", 0x8000000000000000);
    nn::fs::Unmount("save");
}

void DumpSaveDataRaw(char* dstBasePath, nn::fs::SaveDataId saveDataId)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis(nn::fs::BisPartitionId::User, nullptr));

    char srcPath[MaxPathLength];
    char dstPath[MaxPathLength];
    nn::util::SNPrintf(srcPath, sizeof(srcPath), "%s:/save/%016llx", nn::fs::GetBisMountName(nn::fs::BisPartitionId::User), saveDataId);
    nn::util::SNPrintf(dstPath, sizeof(dstPath), "%s/Raw", dstBasePath);
    CopyFile(srcPath, dstPath);

    nn::fs::Unmount(nn::fs::GetBisMountName(nn::fs::BisPartitionId::User));
}

void DumpSaveDataInternalStorage(char* dstBasePath, nn::fs::SaveDataSpaceId spaceId, nn::fs::SaveDataId saveDataId)
{
    const char* Paths[] =
    {
        "AllocationTableControlArea",
        "AllocationTableMeta",
        "AllocationTableData",
    };
    const char* MountName = "internal";

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataInternalStorage(MountName, spaceId, saveDataId));

    for (auto entryName : Paths)
    {
        char srcPath[MaxPathLength];
        char dstPath[MaxPathLength];
        nn::util::SNPrintf(srcPath, sizeof(srcPath), "%s:/%s", MountName, entryName);
        nn::util::SNPrintf(dstPath, sizeof(dstPath), "%s/%s", dstBasePath, entryName);

        CopyFile(srcPath, dstPath);
    }
    nn::fs::Unmount(MountName);
    CleanSaveDataCache();
}

void DumpSaveDataImage(const char* dstBaseDir)
{
    nnt::fs::util::Vector<nn::fs::SaveDataInfo> infoArray;
    nnt::fs::util::FindSaveData(&infoArray, nn::fs::SaveDataSpaceId::User,
        [](const nn::fs::SaveDataInfo& info) { return true; }
    );

    for (auto info : infoArray)
    {
        char dstDir[MaxPathLength];
        nn::util::SNPrintf(dstDir, sizeof(dstDir), "%s/%016llx", dstBaseDir, info.saveDataId);
        nn::fs::CreateDirectory(dstDir);

        int count = 0;
        do
        {
            nn::util::SNPrintf(dstDir, sizeof(dstDir), "%s/%016llx/%d", dstBaseDir, info.saveDataId, count);
            count++;
        } while (ExistsDirectory(dstDir));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory(dstDir));

        DumpSaveDataRaw(dstDir, info.saveDataId);
        DumpSaveDataInternalStorage(dstDir, info.saveDataSpaceId, info.saveDataId);
    }
}

void TestSaveData(const char* dstMountName)
{
    char dstDir[MaxPathLength];
    nn::util::SNPrintf(dstDir, sizeof(dstDir), "%s:", dstMountName);
    DumpSaveDataImage(dstDir);
}

TEST(SaveData, TestSaveDataDump)
{
//    std::mt19937 random(nnt::fs::util::GetRandomSeed());

    const char* MountName = "host";
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost(MountName, HostSaveDataPath));

    TestSaveData(MountName);

    nn::fs::Unmount(MountName);
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();

    auto result = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(result);
}
