﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/crypto/crypto_Sha256Generator.h>

using namespace nn;

/* ダンプされたセーブデータのブロックサイズ毎の差分量を表示 */
/*
 * HostSaveDataPath 以下にディレクトリ名が <saveDataId> のディレクトリがあるか探す。
 * "#"から始まるディレクトリはスキップする。
 * <saveDataId>/<Count> ディレクトリ下のセーブデータイメージファイルのハッシュファイルの
 * "AllocationTableControlArea.hash", "AllocationTableMeta.hash", "AllocationTableData.hash", "Raw.hash"
 * それぞれについて、直前の <Count> ディレクトリ下のイメージファイル との差分を表示する。
 */

namespace {

const int MaxPathLength = 256;
const char HostSaveDataPath[] = {"D:/TempSave"};
const size_t HashBlockSize = 64 * 1024;

const int BlockSizeArray[] =
{
    64 * 1024,
    128 * 1024,
    256 * 1024,
    512 * 1024,
    1024 * 1024,
    2 * 1024 * 1024,
    4 * 1024 * 1024,
    8 * 1024 * 1024,
    16 * 1024 * 1024,
    32 * 1024 * 1024,
    64 * 1024 * 1024,
    128 * 1024 * 1024,
    256 * 1024 * 1024,
    512 * 1024 * 1024,
};

}

bool ExistsDirectory(const char* path)
{
    nn::fs::DirectoryHandle outValue;
    auto result = nn::fs::OpenDirectory(&outValue, path, nn::fs::OpenDirectoryMode_All);
    if (result.IsFailure())
    {
        return false;
    }
    nn::fs::CloseDirectory(outValue);
    return true;
}

int64_t CompareHash(const char* buffer1, int bufferSize1, int64_t fileSize1, const char* buffer2, int bufferSize2, int64_t fileSize2, int blockSize)
{
    int64_t diffSize = 0;
    int restSize = std::min(bufferSize1, bufferSize2);
    int compareSize = (blockSize / HashBlockSize) * nn::crypto::Sha256Generator::HashSize;

    while (restSize > compareSize)
    {
        if (std::memcmp(buffer1, buffer2, compareSize) != 0)
        {
            diffSize += blockSize;
        }
        buffer1 += compareSize;
        buffer2 += compareSize;
        restSize -= compareSize;
    }

    int fileTailSize = (std::min(fileSize1, fileSize2) % blockSize) == 0 ? blockSize : std::min(fileSize1, fileSize2) % blockSize;

    if ((restSize < compareSize) && (fileSize1 != fileSize2))
    {
        diffSize += fileTailSize;
    }
    else
    {
        compareSize = restSize;
        if (std::memcmp(buffer1, buffer2, compareSize) != 0)
        {
            diffSize += fileTailSize;
        }
    }
    diffSize += std::max(fileSize1, fileSize2) - std::min(fileSize1, fileSize2);

    return diffSize;
}

nn::Result CompareFile(const char* suffix, const char* filePath1, const char* filePath2)
{
    nn::fs::FileHandle handle1;
    int64_t fileSize1 = 0;
    NN_RESULT_DO(nn::fs::OpenFile(&handle1, filePath1, nn::fs::OpenMode_Read));
    nn::fs::GetFileSize(&fileSize1, handle1);
    CloseFile(handle1);

    char hashFilePath1[MaxPathLength];
    nn::util::SNPrintf(hashFilePath1, sizeof(hashFilePath1), "%s.hash", filePath1);
    int64_t hashFileSize1 = 0;
    NN_RESULT_DO(nn::fs::OpenFile(&handle1, hashFilePath1, nn::fs::OpenMode_Read));
    nn::fs::GetFileSize(&hashFileSize1, handle1);
    auto buffer1 = nnt::fs::util::AllocateBuffer(hashFileSize1);
    NN_RESULT_DO(nn::fs::ReadFile(handle1, 0, buffer1.get(), hashFileSize1));
    CloseFile(handle1);

    nn::fs::FileHandle handle2;
    int64_t fileSize2 = 0;
    NN_RESULT_DO(nn::fs::OpenFile(&handle2, filePath2, nn::fs::OpenMode_Read));
    nn::fs::GetFileSize(&fileSize2, handle2);
    CloseFile(handle2);

    char hashFilePath2[MaxPathLength];
    nn::util::SNPrintf(hashFilePath2, sizeof(hashFilePath2), "%s.hash", filePath2);
    int64_t hashFileSize2 = 0;
    NN_RESULT_DO(nn::fs::OpenFile(&handle2, hashFilePath2, nn::fs::OpenMode_Read));
    nn::fs::GetFileSize(&hashFileSize2, handle2);
    auto buffer2 = nnt::fs::util::AllocateBuffer(hashFileSize2);
    NN_RESULT_DO(nn::fs::ReadFile(handle2, 0, buffer2.get(), hashFileSize2));
    CloseFile(handle2);

    for (auto blockSize : BlockSizeArray)
    {
        int64_t diffSize;
        diffSize = CompareHash(buffer1.get(), hashFileSize1, fileSize1, buffer2.get(), hashFileSize2, fileSize2, blockSize);

        NN_LOG("%s%s blk=%d : %lld : %lld\n", filePath2, suffix, blockSize, diffSize, std::max(fileSize1, fileSize2));
    }
    NN_RESULT_SUCCESS;
}

void CompareSaveDataHash(const char* srcBaseDir)
{
    int count = 0;
    while (1)
    {
        char srcDir[MaxPathLength];
        nn::util::SNPrintf(srcDir, sizeof(srcDir), "%s%d", srcBaseDir, count);
        if (!ExistsDirectory(srcDir))
        {
            break;
        }
        count++;
    }

    for (int i = 1; i < count; i++)
    {
        const char* Paths[] =
        {
            "AllocationTableControlArea",
            "AllocationTableMeta",
            "AllocationTableData",
            "Raw",
        };
        for (auto entryName : Paths)
        {
            char prevFilePath[MaxPathLength];
            char filePath[MaxPathLength];
            nn::util::SNPrintf(prevFilePath, sizeof(prevFilePath), "%s%d/%s", srcBaseDir, i - 1, entryName);
            nn::util::SNPrintf(filePath, sizeof(filePath), "%s%d/%s", srcBaseDir, i, entryName);

            CompareFile("", prevFilePath, filePath);

            nn::util::SNPrintf(prevFilePath, sizeof(prevFilePath), "%s0/%s", srcBaseDir, entryName);
            CompareFile("-0", prevFilePath, filePath);
        }
    }
}

void SearchHashDirectory(const char* srcBaseDir)
{
    nn::fs::DirectoryHandle srcHandle;
    nn::fs::DirectoryEntry entryBuffer;

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(&srcHandle, srcBaseDir, nn::fs::OpenDirectoryMode_Directory));
    while (1)
    {
        int64_t outValue = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadDirectory(&outValue, &entryBuffer, srcHandle, 1));
        if (outValue == 0)
        {
            break;
        }

        if (entryBuffer.name[0] == '#')
        {
            continue;
        }

        bool isSaveData = true;
        int i;
        for (i = 0; i < strlen(entryBuffer.name); i++)
        {
            if ((entryBuffer.name[i] < '0' && entryBuffer.name[i] > '9') && (entryBuffer.name[i] < 'a' && entryBuffer.name[i] > 'f') && (entryBuffer.name[i] < 'A' && entryBuffer.name[i] > 'F'))
            {
                isSaveData = false;
                break;
            }
        }
        if (i != 16)
        {
            isSaveData = false;
        }

        char srcDir[MaxPathLength];
        nn::util::SNPrintf(srcDir, sizeof(srcDir), "%s%s/", srcBaseDir, entryBuffer.name);
        if (!isSaveData)
        {
            SearchHashDirectory(srcDir);
            continue;
        }

        CompareSaveDataHash(srcDir);
    }
    nn::fs::CloseDirectory(srcHandle);
}

TEST(SaveData, TestSaveDataDiff)
{
    const char* MountName = "host";
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost(MountName, HostSaveDataPath));

    char srcBaseDir[256];
    nn::util::SNPrintf(srcBaseDir, sizeof(srcBaseDir), "%s:/", MountName);

    SearchHashDirectory(srcBaseDir);

    nn::fs::Unmount(MountName);
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();

    auto result = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(result);
}
