﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdio>
#include <map>

#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_Registrar.h>

#include <nn/os/os_ThreadLocalStorage.h>
#include <nn/os/os_MemoryHeap.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>

#include "fsa/fs_FileSystemAccessor.h"
#include "shim/fs_PathBasedFileDataCache.h"

using namespace nn;
using namespace fs;

namespace
{
    os::TlsSlot g_tlsSlot;

    void TlsDestructFunction(uintptr_t value) NN_NOEXCEPT
    {
        NN_UNUSED(value);
    }

    struct TestBaseFileExpectedContext
    {
        bool isReadTouched;
        bool isWriteTouched;

        bool isEnabledArgumentCheck;

        int64_t expectedReadOffset;
        const void* expectedReadBuffer;
        size_t expectedReadSize;
        fs::ReadOption expectedReadOption;

        int64_t expectedWriteOffset;
        const void* expectedWriteBuffer;
        size_t expectedWriteSize;
        fs::WriteOption expectedWriteOption;

        void Reset()
        {
            isReadTouched = false;
            isWriteTouched = false;
            isEnabledArgumentCheck = false;
        }

        void SetExpectedArgument(int64_t readOffset, const void* readBuffer, size_t readSize, fs::ReadOption& readOption, int64_t writeOffset, const void* writeBuffer, size_t writeSize, fs::WriteOption& writeOption) NN_NOEXCEPT
        {
            isEnabledArgumentCheck = true;
            expectedReadOffset = readOffset;
            expectedReadBuffer = readBuffer;
            expectedReadSize = readSize;
            expectedReadOption = readOption;

            expectedWriteOffset = writeOffset;
            expectedWriteBuffer = writeBuffer;
            expectedWriteSize = writeSize;
            expectedWriteOption = writeOption;
        }
    };

    void SetTestBaseFileExpectedContext(TestBaseFileExpectedContext* context) NN_NOEXCEPT
    {
        os::SetTlsValue(g_tlsSlot, reinterpret_cast<uintptr_t>(context));
    }

    TestBaseFileExpectedContext* GetTestBaseFileExpectedContext() NN_NOEXCEPT
    {
        return reinterpret_cast<TestBaseFileExpectedContext*>(os::GetTlsValue(g_tlsSlot));
    }

    class ScopedContextChecker
    {
    public:
        ScopedContextChecker(TestBaseFileExpectedContext* context, bool isReadTouched, bool isWriteTouched) NN_NOEXCEPT
            : m_pContext(context)
            , m_ExpectedReadTouched(isReadTouched)
            , m_ExpectedWriteTouched(isWriteTouched)
        {
            m_pContext->Reset();
        }

        ScopedContextChecker(TestBaseFileExpectedContext* context, bool isReadTouched, bool isWriteTouched, int64_t readOffset, const void* readBuffer, size_t readSize, fs::ReadOption& readOption, int64_t writeOffset, const void* writeBuffer, size_t writeSize, fs::WriteOption& writeOption) NN_NOEXCEPT
            : m_pContext(context)
            , m_ExpectedReadTouched(isReadTouched)
            , m_ExpectedWriteTouched(isWriteTouched)
        {
            m_pContext->Reset();
            m_pContext->SetExpectedArgument(readOffset, readBuffer, readSize, readOption, writeOffset, writeBuffer, writeSize, writeOption);
        }

        ~ScopedContextChecker() NN_NOEXCEPT
        {
            EXPECT_TRUE(m_pContext->isReadTouched == m_ExpectedReadTouched);
            EXPECT_TRUE(m_pContext->isWriteTouched == m_ExpectedWriteTouched);
        }

    private:
        TestBaseFileExpectedContext* m_pContext;
        bool m_ExpectedReadTouched;
        bool m_ExpectedWriteTouched;
    };

    class TestBaseFile : public fs::fsa::IFile, public fs::detail::Newable
    {
    public:
        TestBaseFile(void* pBuffer, size_t bufferSize) NN_NOEXCEPT
            : m_pBuffer(pBuffer)
            , m_BufferSize(bufferSize)
            , m_FileSize(static_cast<int64_t>(bufferSize))
        {
        }

        virtual Result DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE
        {
            auto context = GetTestBaseFileExpectedContext();

            if (context->isEnabledArgumentCheck)
            {
                EXPECT_TRUE(offset == context->expectedReadOffset);
                EXPECT_TRUE(buffer == context->expectedReadBuffer);
                EXPECT_TRUE(size == context->expectedReadSize);
                EXPECT_TRUE(option.reserved == context->expectedReadOption.reserved);
            }

            NN_ABORT_UNLESS(m_BufferSize == static_cast<size_t>(m_FileSize));

            int64_t restSize = std::max(static_cast<int64_t>(m_FileSize) - offset, static_cast<int64_t>(0));
            size_t readSize = std::min(size, static_cast<size_t>(restSize));
            if (restSize == 0)
            {
                *outValue = 0;
                context->isReadTouched = true;
                NN_RESULT_SUCCESS;
            }

            std::memcpy(buffer, reinterpret_cast<char*>(m_pBuffer) + offset, readSize);

            *outValue = readSize;
            context->isReadTouched = true;
            NN_RESULT_SUCCESS;
        }

        virtual Result DoWrite(int64_t offset, const void* buffer, size_t size, const fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE
        {
            auto context = GetTestBaseFileExpectedContext();

            if (context->isEnabledArgumentCheck)
            {
                EXPECT_TRUE(offset == context->expectedWriteOffset);
                EXPECT_TRUE(buffer == context->expectedWriteBuffer);
                EXPECT_TRUE(size == context->expectedWriteSize);
                EXPECT_TRUE(option.flags == context->expectedWriteOption.flags);
            }

            NN_ABORT_UNLESS(m_BufferSize == static_cast<size_t>(m_FileSize));

            if (size == 0)
            {
                context->isWriteTouched = true;
                NN_RESULT_SUCCESS;
            }

            std::memcpy(reinterpret_cast<char*>(m_pBuffer) + offset, buffer, size);

            context->isWriteTouched = true;
            NN_RESULT_SUCCESS;
        }

        virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_SUCCESS;
        }

        virtual Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            m_FileSize = size;
            NN_RESULT_SUCCESS;
        }

        virtual Result DoGetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *outValue = m_FileSize;
            NN_RESULT_SUCCESS;
        }

        virtual Result DoOperateRange(void* outBuffer, size_t outBufferSize, fs::OperationId operationId, int64_t offset, int64_t size, const void* inBuffer, size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(outBuffer);
            NN_UNUSED(outBufferSize);
            NN_UNUSED(operationId);
            NN_UNUSED(offset);
            NN_UNUSED(size);
            NN_UNUSED(inBuffer);
            NN_UNUSED(inBufferSize);
            NN_RESULT_SUCCESS;
        }

    private:
        void* m_pBuffer;
        size_t m_BufferSize;
        int64_t m_FileSize;

    };

    struct TestBaseFileTableEntry
    {
        const char* path;
        void* pBuffer;
        size_t bufferSize;
    };

    class TestBaseFileSystem : public fs::fsa::IFileSystem, public fs::detail::Newable
    {
    public:
        TestBaseFileSystem(const TestBaseFileTableEntry* table, int tableLength) NN_NOEXCEPT
            : m_Table(table)
            , m_TableLength(tableLength)
        {
        }

        virtual Result DoCreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(path);
            NN_UNUSED(size);
            NN_UNUSED(option);
            if (m_EmulatedResult)
            {
                return *m_EmulatedResult;
            }
            NN_RESULT_SUCCESS;
        }

        virtual Result DoDeleteFile(const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(path);
            return fs::ResultNotImplemented();
        }

        virtual Result DoCreateDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(path);
            return fs::ResultNotImplemented();
        }

        virtual Result DoDeleteDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(path);
            return fs::ResultNotImplemented();
        }

        virtual Result DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(path);
            return fs::ResultNotImplemented();
        }

        virtual Result DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(path);
            return fs::ResultNotImplemented();
        }

        virtual Result DoRenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(currentPath);
            NN_UNUSED(newPath);
            if (m_EmulatedResult)
            {
                return *m_EmulatedResult;
            }
            NN_RESULT_SUCCESS;
        }

        virtual Result DoRenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(currentPath);
            NN_UNUSED(newPath);
            if (m_EmulatedResult)
            {
                return *m_EmulatedResult;
            }
            NN_RESULT_SUCCESS;
        }

        virtual Result DoGetEntryType(fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(outValue);
            NN_UNUSED(path);
            return fs::ResultNotImplemented();
        }

        virtual Result DoOpenFile(std::unique_ptr<fs::fsa::IFile>* outValue, const char* path, fs::OpenMode mode) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(mode);

            if (m_EmulatedResult)
            {
                return *m_EmulatedResult;
            }

            for (int i = 0; i < m_TableLength; i++)
            {
                if (util::Strncmp(m_Table[i].path, path, fs::EntryNameLengthMax) == 0)
                {
                    std::unique_ptr<fs::fsa::IFile> pFile(new TestBaseFile(m_Table[i].pBuffer, m_Table[i].bufferSize));
                    NN_ABORT_UNLESS_NOT_NULL(pFile);
                    *outValue = std::move(pFile);
                    NN_RESULT_SUCCESS;
                }
            }

            return fs::ResultPathNotFound();
        }

        virtual Result DoOpenDirectory(std::unique_ptr<fs::fsa::IDirectory>* outValue, const char* path, fs::OpenDirectoryMode mode) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(outValue);
            NN_UNUSED(path);
            NN_UNUSED(mode);
            return fs::ResultNotImplemented();
        }

        virtual Result DoCommit() NN_NOEXCEPT NN_OVERRIDE
        {
            return fs::ResultNotImplemented();
        }

        void SetEmulatedResult(Result result)
        {
            m_EmulatedResult = result;
        }

        void UnsetEmulatedResult()
        {
            m_EmulatedResult = util::nullopt;
        }

    private:
        const TestBaseFileTableEntry* m_Table;
        int m_TableLength;
        util::optional<Result> m_EmulatedResult = util::nullopt;
    };

    const size_t BufferSize = 1 * 1024 * 1024;
    const size_t BufferCount = 16;
    char g_Buffer[BufferCount][BufferSize];

    TestBaseFileTableEntry g_Table[] =
    {
        { "/0", g_Buffer[0], BufferSize },
        { "/1", g_Buffer[1], BufferSize },
        { "/hoge/../0", g_Buffer[0], BufferSize },
    };
    const size_t g_TableLength = sizeof(g_Table) / sizeof(g_Table[0]);

    class PathBasedFileDataCache : public ::testing::Test
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            for (int i = 0; i < g_TableLength; i++)
            {
                std::memset(g_Buffer[i], 0, BufferSize);
            }

            std::unique_ptr<TestBaseFileSystem> testFs(new TestBaseFileSystem(g_Table, g_TableLength));
            ASSERT_TRUE(testFs != nullptr);

            m_pFs = testFs.get();
            fs::fsa::Register("test", std::move(testFs), nullptr, false, true, false);
            NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE("test");

            m_Context.Reset();
            SetTestBaseFileExpectedContext(&m_Context);
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            fs::fsa::Unregister("test");
        }

        void Remount() NN_NOEXCEPT
        {
            fs::fsa::Unregister("test");
            std::unique_ptr<TestBaseFileSystem> testFs(new TestBaseFileSystem(g_Table, g_TableLength));
            ASSERT_TRUE(testFs != nullptr);
            m_pFs = testFs.get();
            fs::fsa::Register("test", std::move(testFs), nullptr, false, true, false);
            NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE("test");
        }

        TestBaseFileSystem* GetTestBaseFileSystem() NN_NOEXCEPT
        {
            return m_pFs;
        }

        TestBaseFileExpectedContext* GetContext() NN_NOEXCEPT
        {
            return &m_Context;
        }

    private:
        TestBaseFileSystem* m_pFs;
        TestBaseFileExpectedContext m_Context;
    };

    struct TestParam
    {
        int64_t offset;
        size_t size;
    };

    class PathBasedFileDataCacheWithParam : public PathBasedFileDataCache, public ::testing::WithParamInterface<TestParam>
    {
    };

    class PathBasedFileDataCacheWithBool : public PathBasedFileDataCache, public ::testing::WithParamInterface<bool>
    {
    };

    void TestReadWriteArgument(TestBaseFileExpectedContext* context, TestParam& param, const void* randomBuffer, bool invalidate) NN_NOEXCEPT
    {
        fs::FileHandle file;
        NNT_EXPECT_RESULT_SUCCESS(fs::OpenFile(&file, "test:/0", fs::OpenMode::OpenMode_Read | fs::OpenMode::OpenMode_Write));

        size_t readSize = 0;
        auto readOption = fs::ReadOption::MakeValue(1);
        auto writeOption = fs::WriteOption::MakeValue(fs::WriteOptionFlag::WriteOptionFlag_Flush);

        if (param.size == 0)
        {
            ScopedContextChecker checker(context, false, false, param.offset, nullptr, param.size, readOption, param.offset, nullptr, param.size, writeOption);
            NNT_EXPECT_RESULT_SUCCESS(fs::ReadFile(&readSize, file, param.offset, nullptr, param.size, readOption));
            EXPECT_EQ(readSize, 0);
            NNT_EXPECT_RESULT_SUCCESS(fs::WriteFile(file, param.offset, nullptr, param.size, writeOption));
            NNT_EXPECT_RESULT_SUCCESS(fs::ReadFile(&readSize, file, param.offset, nullptr, param.size, readOption));
            EXPECT_EQ(readSize, 0);
        }
        else
        {
            auto readBuffer = nnt::fs::util::AllocateBuffer(BufferSize);
            auto writeBuffer = nnt::fs::util::AllocateBuffer(BufferSize);
            ASSERT_TRUE(readBuffer != nullptr && writeBuffer != nullptr);
            std::memset(readBuffer.get(), 0, BufferSize);
            nnt::fs::util::FillBufferWithRandomValue(writeBuffer.get(), BufferSize);

            int64_t restSize = std::max(static_cast<int64_t>(BufferSize) - param.offset, static_cast<int64_t>(0));
            size_t actualReadWriteSize = std::min(static_cast<size_t>(restSize), param.size);

            ScopedContextChecker checker(context, invalidate, actualReadWriteSize != 0 ? true : false, param.offset, readBuffer.get(), param.size, readOption, param.offset, writeBuffer.get(), actualReadWriteSize, writeOption);
            NNT_EXPECT_RESULT_SUCCESS(fs::ReadFile(&readSize, file, param.offset, readBuffer.get(), param.size, readOption));
            EXPECT_EQ(readSize, actualReadWriteSize);
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(reinterpret_cast<const char*>(randomBuffer) + param.offset, readBuffer.get(), actualReadWriteSize);
            NNT_EXPECT_RESULT_SUCCESS(fs::WriteFile(file, param.offset, writeBuffer.get(), actualReadWriteSize, writeOption));
            NNT_EXPECT_RESULT_SUCCESS(fs::ReadFile(&readSize, file, param.offset, readBuffer.get(), param.size, readOption));
            EXPECT_EQ(readSize, actualReadWriteSize);
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer.get(), readBuffer.get(), actualReadWriteSize);
        }
        fs::CloseFile(file);
    }

    Result CalculateHashDummy(fs::detail::FilePathHash* outValue, const char* path) NN_NOEXCEPT
    {
        // 常に衝突する
        NN_UNUSED(path);
        std::memset(outValue->data, 0, fs::detail::FilePathHashSize);
        NN_RESULT_SUCCESS;
    }

#if 0
    // Hash 探索関数
    void FindCollision(fs::detail::CalculateHashFunc hashFunc) NN_NOEXCEPT
    {
        auto compare = [] (const fs::detail::FilePathHash& hasha, const fs::detail::FilePathHash& hashb) -> bool
        {
            return std::memcmp(hasha.data, hashb.data, fs::detail::FilePathHashSize) >= 0;
        };

        std::map<fs::detail::FilePathHash, nnt::fs::util::String, decltype(compare)> map(compare);

        while (NN_STATIC_CONDITION(true))
        {
            auto entry = std::make_pair<fs::detail::FilePathHash, nnt::fs::util::String>({}, "");
            {
                nnt::fs::util::String targetPath("/");
                auto length = nnt::fs::util::GetRandomLength();
                if (length > 700)
                {
                    continue;
                }
                targetPath.append(nnt::fs::util::GenerateRandomLengthEntryName(length));
                entry.second.append(targetPath);
            }

            hashFunc(&(entry.first), entry.second.c_str());

            auto it = map.find(entry.first);
            if (it != map.end() && strncmp(entry.second.c_str(), it->second.c_str(), 700) != 0)
            {
                NN_LOG("found pair: %s - %s\n", entry.second.c_str(), it->second.c_str());
                break;
            }
            else
            {
                map.insert(std::move(entry));
                continue;
            }
        }
        return;
    }
#endif

}

TEST_F(PathBasedFileDataCache, Invalidate)
{
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::WriteFileWith32BitCount("test:/0", BufferSize, 0));
    EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));

    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::WriteFileWith32BitCount("test:/1", BufferSize, BufferSize));
    EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/1", BufferSize));

    auto buffer0 = nnt::fs::util::AllocateBuffer(BufferSize);
    auto buffer1 = nnt::fs::util::AllocateBuffer(BufferSize);
    ASSERT_TRUE(buffer0 != nullptr && buffer1 != nullptr);

    auto context = GetContext();

    {
        NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/1", buffer1.get(), BufferSize));
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/1", BufferSize));
    }

    auto enableCache = [&]() NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/0", buffer0.get(), BufferSize));
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    };

    // Unset
    enableCache();
    {
        fs::DisableIndividualFileDataCache("test:/0");
        ScopedContextChecker checker(context, true, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    {
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/1", BufferSize));
    }

    // Create
    enableCache();
    {
        GetTestBaseFileSystem()->SetEmulatedResult(fs::ResultUnexpected());
        NNT_EXPECT_RESULT_FAILURE(fs::ResultUnexpected, fs::CreateFile("test:/0", BufferSize));
        GetTestBaseFileSystem()->UnsetEmulatedResult();
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    {
        NNT_EXPECT_RESULT_SUCCESS(fs::CreateFile("test:/0", BufferSize));
        ScopedContextChecker checker(context, true, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    {
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/1", BufferSize));
    }

    // Rename
    enableCache();
    {
        GetTestBaseFileSystem()->SetEmulatedResult(fs::ResultUnexpected());
        NNT_EXPECT_RESULT_FAILURE(fs::ResultUnexpected, fs::RenameFile("test:/1", "test:/0"));
        GetTestBaseFileSystem()->UnsetEmulatedResult();
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    {
        NNT_EXPECT_RESULT_SUCCESS(fs::RenameFile("test:/1", "test:/0"));
        ScopedContextChecker checker(context, true, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    {
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/1", BufferSize));
    }

    fs::FileHandle file;

    // OpenFile (Append)
    enableCache();
    {
        GetTestBaseFileSystem()->SetEmulatedResult(fs::ResultUnexpected());
        NNT_EXPECT_RESULT_FAILURE(fs::ResultUnexpected, fs::OpenFile(&file, "test:/0", fs::OpenMode::OpenMode_Write | fs::OpenMode::OpenMode_AllowAppend));
        GetTestBaseFileSystem()->UnsetEmulatedResult();
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    {
        NNT_EXPECT_RESULT_SUCCESS(fs::OpenFile(&file, "test:/0", fs::OpenMode::OpenMode_Write | fs::OpenMode::OpenMode_AllowAppend));
        fs::CloseFile(file);
        ScopedContextChecker checker(context, true, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    {
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/1", BufferSize));
    }

    // SetFileSize
    enableCache();
    {
        NNT_EXPECT_RESULT_SUCCESS(fs::OpenFile(&file, "test:/0", fs::OpenMode::OpenMode_Write));
        fs::CloseFile(file);
        {
            ScopedContextChecker checker(context, false, false);
            EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
        }
        NNT_EXPECT_RESULT_SUCCESS(fs::OpenFile(&file, "test:/0", fs::OpenMode::OpenMode_Write));
        NNT_EXPECT_RESULT_FAILURE(fs::ResultOutOfRange, fs::SetFileSize(file, -1));
        fs::CloseFile(file);
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    {
        NNT_EXPECT_RESULT_SUCCESS(fs::OpenFile(&file, "test:/0", fs::OpenMode::OpenMode_Write));
        NNT_EXPECT_RESULT_SUCCESS(fs::SetFileSize(file, BufferSize * 2));
        fs::CloseFile(file);
        {
            ScopedContextChecker checker(context, true, false);
            EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
        }
    }
    {
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/1", BufferSize));
    }

    // RenameDir
    enableCache();
    {
        GetTestBaseFileSystem()->SetEmulatedResult(fs::ResultUnexpected());
        NNT_EXPECT_RESULT_FAILURE(fs::ResultUnexpected, fs::RenameDirectory("test:/2", "test:/3"));
        GetTestBaseFileSystem()->UnsetEmulatedResult();
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    {
        NNT_EXPECT_RESULT_SUCCESS(fs::RenameDirectory("test:/2", "test:/3"));
        ScopedContextChecker checker(context, true, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    {
        ScopedContextChecker checker(context, true, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/1", BufferSize));
    }


    // Unmount
    enableCache();
    {
        NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/1", buffer1.get(), BufferSize));
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/1", BufferSize));
    }
    {
        Remount();
        ScopedContextChecker checker(context, true, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    {
        ScopedContextChecker checker(context, true, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/1", BufferSize));
    }
} // NOLINT(impl/function_size)

TEST_F(PathBasedFileDataCache, PreConditionTargetInUse)
{
    auto buffer = nnt::fs::util::AllocateBuffer(BufferSize);
    ASSERT_TRUE(buffer != nullptr);

    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/0", buffer.get(), BufferSize));
    // Hash 衝突
    NNT_EXPECT_RESULT_FAILURE(fs::ResultIndividualFileDataCacheAlreadyEnabled, fs::EnableIndividualFileDataCache("test:/0", buffer.get(), BufferSize));
    fs::DisableIndividualFileDataCache("test:/0");
    // 2 重クローズは OK
    fs::DisableIndividualFileDataCache("test:/0");

    // Open 時エラーのハンドリング
    GetTestBaseFileSystem()->SetEmulatedResult(fs::ResultTargetLocked());
    NNT_EXPECT_RESULT_FAILURE(fs::ResultTargetLocked, fs::EnableIndividualFileDataCache("test:/0", buffer.get(), BufferSize));
    GetTestBaseFileSystem()->UnsetEmulatedResult();
}

TEST_F(PathBasedFileDataCache, PreCondition)
{
    auto buffer = nnt::fs::util::AllocateBuffer(BufferSize);
    ASSERT_TRUE(buffer != nullptr);

    NNT_EXPECT_RESULT_FAILURE(fs::ResultNullptrArgument, fs::EnableIndividualFileDataCache("test:/0", nullptr, BufferSize));
    NNT_EXPECT_RESULT_FAILURE(fs::ResultInvalidSize, fs::EnableIndividualFileDataCache("test:/0", buffer.get(), BufferSize - 1));
    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/0", buffer.get(), BufferSize + 1));
    fs::DisableIndividualFileDataCache("test:/0");
}

TEST_F(PathBasedFileDataCache, BufferAddressAlignment)
{
    auto buffer = nnt::fs::util::AllocateBuffer(BufferSize * 2);
    ASSERT_TRUE(buffer != nullptr);

    // 0～64までのオフセット用MAX値
    const size_t OffsetSize = 64;

    auto randomBuffer = nnt::fs::util::AllocateBuffer(BufferSize);
    auto readBuffer = nnt::fs::util::AllocateBuffer(BufferSize);
    ASSERT_TRUE(randomBuffer != nullptr && readBuffer != nullptr);

    auto context = GetContext();

    for (size_t changePos = 0; changePos <= OffsetSize; ++changePos)
    {
        nnt::fs::util::InvalidateVariable(reinterpret_cast<uint8_t*>(buffer.get()), BufferSize * 2);
        NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/0", buffer.get() + changePos, BufferSize));
        fs::FileHandle file;
        {
            nnt::fs::util::FillBufferWithRandomValue(randomBuffer.get(), BufferSize);
            NNT_EXPECT_RESULT_SUCCESS(fs::OpenFile(&file, "test:/0", fs::OpenMode::OpenMode_Read | fs::OpenMode::OpenMode_Write));

            ScopedContextChecker checker(context, false, true);

            NNT_EXPECT_RESULT_SUCCESS(fs::WriteFile(file, 0, randomBuffer.get(), BufferSize, fs::WriteOption::MakeValue(fs::WriteOptionFlag::WriteOptionFlag_Flush)));
            size_t readSize = 0;
            NNT_EXPECT_RESULT_SUCCESS(fs::ReadFile(&readSize, file, 0, readBuffer.get(), BufferSize, fs::ReadOption::MakeValue(0)));
            EXPECT_EQ(readSize, BufferSize);
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(randomBuffer.get(), readBuffer.get(), readSize);

            EXPECT_TRUE(nnt::fs::util::IsValidInitializedBound(reinterpret_cast<uint8_t*>(buffer.get()), BufferSize * 2, changePos, BufferSize));
        }
        fs::CloseFile(file);
        fs::DisableIndividualFileDataCache("test:/0");
    }
}

TEST_F(PathBasedFileDataCache, ReadWritePreCondition)
{
    auto buffer = nnt::fs::util::AllocateBuffer(BufferSize * 2);
    ASSERT_TRUE(buffer != nullptr);

    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::WriteFileWith32BitCount("test:/0", BufferSize, 0));

    // BufferSize をファイルサイズより余分に渡してテストする
    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/0", buffer.get(), BufferSize * 2));

    auto context = GetContext();

    {
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }

    auto rwBuffer = nnt::fs::util::AllocateBuffer(BufferSize);
    ASSERT_TRUE(rwBuffer != nullptr);

    fs::FileHandle file;
    size_t readSize = 0;

    // OpenMode, NullPtr, OutOfRange
    NNT_EXPECT_RESULT_SUCCESS(fs::OpenFile(&file, "test:/0", fs::OpenMode::OpenMode_Write));
    NNT_EXPECT_RESULT_FAILURE(fs::ResultInvalidOperationForOpenMode, fs::ReadFile(&readSize, file, 0, rwBuffer.get(), BufferSize, fs::ReadOption::MakeValue(0)));
    NNT_EXPECT_RESULT_FAILURE(fs::ResultNullptrArgument, fs::WriteFile(file, 0, nullptr, BufferSize, fs::WriteOption::MakeValue(fs::WriteOptionFlag::WriteOptionFlag_Flush)));
    fs::CloseFile(file);
    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/0", buffer.get(), BufferSize * 2));

    NNT_EXPECT_RESULT_SUCCESS(fs::OpenFile(&file, "test:/0", fs::OpenMode::OpenMode_Write));
    NNT_EXPECT_RESULT_FAILURE(fs::ResultOutOfRange, fs::WriteFile(file, -1, rwBuffer.get(), BufferSize, fs::WriteOption::MakeValue(fs::WriteOptionFlag::WriteOptionFlag_Flush)));
    fs::CloseFile(file);
    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/0", buffer.get(), BufferSize * 2));

    NNT_EXPECT_RESULT_SUCCESS(fs::OpenFile(&file, "test:/0", fs::OpenMode::OpenMode_Write));
    NNT_EXPECT_RESULT_FAILURE(fs::ResultOutOfRange, fs::WriteFile(file, BufferSize + 1, rwBuffer.get(), 1, fs::WriteOption::MakeValue(fs::WriteOptionFlag::WriteOptionFlag_Flush)));
    fs::CloseFile(file);
    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/0", buffer.get(), BufferSize * 2));

    NNT_EXPECT_RESULT_SUCCESS(fs::OpenFile(&file, "test:/0", fs::OpenMode::OpenMode_Write));
    NNT_EXPECT_RESULT_FAILURE(fs::ResultOutOfRange, fs::WriteFile(file, 0, rwBuffer.get(), BufferSize + 1, fs::WriteOption::MakeValue(fs::WriteOptionFlag::WriteOptionFlag_Flush)));
    fs::CloseFile(file);
    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/0", buffer.get(), BufferSize * 2));

    NNT_EXPECT_RESULT_SUCCESS(fs::OpenFile(&file, "test:/0", fs::OpenMode::OpenMode_Read));
    NNT_EXPECT_RESULT_FAILURE(fs::ResultInvalidOperationForOpenMode, fs::WriteFile(file, 0, rwBuffer.get(), BufferSize, fs::WriteOption::MakeValue(fs::WriteOptionFlag::WriteOptionFlag_Flush)));
    fs::CloseFile(file);
    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/0", buffer.get(), BufferSize * 2));

    NNT_EXPECT_RESULT_SUCCESS(fs::OpenFile(&file, "test:/0", fs::OpenMode::OpenMode_Read));
    NNT_EXPECT_RESULT_FAILURE(fs::ResultNullptrArgument, fs::ReadFile(&readSize, file, 0, nullptr, BufferSize, fs::ReadOption::MakeValue(0)));
    NNT_EXPECT_RESULT_FAILURE(fs::ResultNullptrArgument, fs::ReadFile(nullptr, file, 0, rwBuffer.get(), BufferSize, fs::ReadOption::MakeValue(0)));
    NNT_EXPECT_RESULT_FAILURE(fs::ResultOutOfRange, fs::ReadFile(&readSize, file, -1, rwBuffer.get(), BufferSize, fs::ReadOption::MakeValue(0)));
    NNT_EXPECT_RESULT_FAILURE(fs::ResultOutOfRange, fs::ReadFile(&readSize, file, BufferSize + 1, rwBuffer.get(), 1, fs::ReadOption::MakeValue(0)));
    fs::CloseFile(file);
}

TEST_P(PathBasedFileDataCacheWithParam, ReadWriteArgument)
{
    auto buffer = nnt::fs::util::AllocateBuffer(BufferSize);
    ASSERT_TRUE(buffer != nullptr);

    auto randomBuffer = nnt::fs::util::AllocateBuffer(BufferSize);
    ASSERT_TRUE(randomBuffer != nullptr);

    nnt::fs::util::FillBufferWithRandomValue(randomBuffer.get(), BufferSize);
    {
        fs::FileHandle file;
        NNT_EXPECT_RESULT_SUCCESS(fs::OpenFile(&file, "test:/0", fs::OpenMode::OpenMode_Write));
        NNT_EXPECT_RESULT_SUCCESS(fs::WriteFile(file, 0, randomBuffer.get(), BufferSize, fs::WriteOption::MakeValue(fs::WriteOptionFlag::WriteOptionFlag_Flush)));
        fs::CloseFile(file);
    }

    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/0", buffer.get(), BufferSize));

    auto context = GetContext();
    auto param = GetParam();

    TestReadWriteArgument(context, param, randomBuffer.get(), false);
}

TEST_P(PathBasedFileDataCacheWithParam, ReadWriteArgumentAfterInvalidate)
{
    auto buffer = nnt::fs::util::AllocateBuffer(BufferSize);
    ASSERT_TRUE(buffer != nullptr);

    auto randomBuffer = nnt::fs::util::AllocateBuffer(BufferSize);
    ASSERT_TRUE(randomBuffer != nullptr);

    nnt::fs::util::FillBufferWithRandomValue(randomBuffer.get(), BufferSize);
    {
        fs::FileHandle file;
        NNT_EXPECT_RESULT_SUCCESS(fs::OpenFile(&file, "test:/0", fs::OpenMode::OpenMode_Write));
        NNT_EXPECT_RESULT_SUCCESS(fs::WriteFile(file, 0, randomBuffer.get(), BufferSize, fs::WriteOption::MakeValue(fs::WriteOptionFlag::WriteOptionFlag_Flush)));
        fs::CloseFile(file);
    }

    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/0", buffer.get(), BufferSize));
    fs::DisableIndividualFileDataCache("test:/0");

    auto context = GetContext();
    auto param = GetParam();

    TestReadWriteArgument(context, param, randomBuffer.get(), true);
}

TestParam offsetAndSizeParam[] = {
    { -1, 0 },
    { 0, BufferSize - 1 },
    { 0, BufferSize },
    { 0, BufferSize + 1 },
    { 1, BufferSize - 1 },
    { 1, BufferSize },
    { 1, BufferSize + 1 },
    { BufferSize / 2 , BufferSize / 2 - 1 },
    { BufferSize / 2 , BufferSize / 2 },
    { BufferSize / 2 , BufferSize / 2 + 1 },
    { BufferSize / 2 - 1, BufferSize / 2 },
    { BufferSize / 2 , BufferSize / 2 },
    { BufferSize / 2 + 1, BufferSize / 2 },
    { BufferSize - 1, 1 },
    { BufferSize - 1, 2 },
    { BufferSize, 1 },
    { BufferSize, 0 },
    { BufferSize + 1, 0 },
};

INSTANTIATE_TEST_CASE_P(WithVariousOffsetAnrSize, PathBasedFileDataCacheWithParam, ::testing::ValuesIn(offsetAndSizeParam));

TEST_P(PathBasedFileDataCacheWithBool, ReadWriteArgument)
{
    bool invalidate = GetParam();

    auto buffer = nnt::fs::util::AllocateBuffer(BufferSize);
    ASSERT_TRUE(buffer != nullptr);

    auto randomBuffer = nnt::fs::util::AllocateBuffer(BufferSize);
    ASSERT_TRUE(randomBuffer != nullptr);

    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/0", buffer.get(), BufferSize));
    if (invalidate)
    {
        fs::DisableIndividualFileDataCache("test:/0");
    }

    auto context = GetContext();
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    static const int LoopCount = 256;
    TestParam param;

    for( int i = 0; i < LoopCount; ++i )
    {
        param.size = static_cast<size_t>(std::uniform_int_distribution<int64_t>(1, BufferSize)(mt));
        param.offset = std::uniform_int_distribution<int64_t>(0, BufferSize - static_cast<int64_t>(param.size))(mt);

        nnt::fs::util::FillBufferWithRandomValue(randomBuffer.get(), BufferSize);
        {
            context->Reset();
            fs::FileHandle file;
            NNT_EXPECT_RESULT_SUCCESS(fs::OpenFile(&file, "test:/0", fs::OpenMode::OpenMode_Write));
            NNT_EXPECT_RESULT_SUCCESS(fs::WriteFile(file, 0, randomBuffer.get(), BufferSize, fs::WriteOption::MakeValue(fs::WriteOptionFlag::WriteOptionFlag_Flush)));
            fs::CloseFile(file);
        }
        TestReadWriteArgument(context, param, randomBuffer.get(), invalidate);
    }
}

INSTANTIATE_TEST_CASE_P(WithRandomOffsetAnrSize, PathBasedFileDataCacheWithBool, ::testing::Bool());

TEST_F(PathBasedFileDataCache, HashCollision)
{
    auto buffer0 = nnt::fs::util::AllocateBuffer(BufferSize);
    auto buffer1 = nnt::fs::util::AllocateBuffer(BufferSize);
    ASSERT_TRUE(buffer0 != nullptr && buffer1 != nullptr);

    fs::detail::SetCalculateHashFunc(CalculateHashDummy);

    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::WriteFileWith32BitCount("test:/0", BufferSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::WriteFileWith32BitCount("test:/1", BufferSize, BufferSize));

    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/0", buffer0.get(), BufferSize));
    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/1", buffer1.get(), BufferSize));

    auto context = GetContext();

    // 別ファイルでハッシュが衝突しても、データを取り違えない
    {
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    {
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/1", BufferSize));
    }

    // 各種 Invalidate でも取り違えない
    {
        fs::DisableIndividualFileDataCache("test:/1");
        ScopedContextChecker checker(context, true, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/1", BufferSize));
    }
    {
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/1", buffer1.get(), BufferSize));

    {
        NNT_EXPECT_RESULT_SUCCESS(fs::CreateFile("test:/1", BufferSize));
        ScopedContextChecker checker(context, true, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/1", BufferSize));
    }
    {
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/1", buffer1.get(), BufferSize));

    {
        NNT_EXPECT_RESULT_SUCCESS(fs::RenameFile("test:/0", "test:/1"));
        ScopedContextChecker checker(context, true, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/1", BufferSize));
    }
    {
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/1", buffer1.get(), BufferSize));

    fs::FileHandle file;
    {
        NNT_EXPECT_RESULT_SUCCESS(fs::OpenFile(&file, "test:/1", fs::OpenMode::OpenMode_Write | fs::OpenMode::OpenMode_AllowAppend));
        fs::CloseFile(file);
        ScopedContextChecker checker(context, true, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/1", BufferSize));
    }
    {
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/1", buffer1.get(), BufferSize));

    {
        NNT_EXPECT_RESULT_SUCCESS(fs::OpenFile(&file, "test:/1", fs::OpenMode::OpenMode_Write));
        NNT_EXPECT_RESULT_SUCCESS(fs::SetFileSize(file, BufferSize * 2));
        fs::CloseFile(file);
        ScopedContextChecker checker(context, true, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/1", BufferSize));
    }
    {
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/1", buffer1.get(), BufferSize));

    fs::detail::SetDefaultCalculateHashFunc();
}

TEST_F(PathBasedFileDataCache, PathNormalization)
{
    auto buffer = nnt::fs::util::AllocateBuffer(BufferSize);
    ASSERT_TRUE(buffer != nullptr);

    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::WriteFileWith32BitCount("test:/0", BufferSize, 0));

    const char* testFileName = "test:/hoge/../0";

    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/0", buffer.get(), BufferSize));
    NNT_EXPECT_RESULT_FAILURE(fs::ResultIndividualFileDataCacheAlreadyEnabled, fs::EnableIndividualFileDataCache(testFileName, buffer.get(), BufferSize));

    auto context = GetContext();

    // 同じファイルを指すパスで開いてもキャッシュが効く
    {
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount(testFileName, 0));
    }

    // 同じファイルを指すパスでも Invalidate される
    {
        fs::DisableIndividualFileDataCache(testFileName);
        ScopedContextChecker checker(context, true, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/0", buffer.get(), BufferSize));

    {
        NNT_EXPECT_RESULT_SUCCESS(fs::CreateFile(testFileName, BufferSize));
        ScopedContextChecker checker(context, true, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/0", buffer.get(), BufferSize));

    {
        NNT_EXPECT_RESULT_SUCCESS(fs::RenameFile("test:/1", testFileName));
        ScopedContextChecker checker(context, true, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/0", buffer.get(), BufferSize));

    fs::FileHandle file;
    {
        NNT_EXPECT_RESULT_SUCCESS(fs::OpenFile(&file, testFileName, fs::OpenMode::OpenMode_Write | fs::OpenMode::OpenMode_AllowAppend));
        fs::CloseFile(file);
        ScopedContextChecker checker(context, true, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/0", buffer.get(), BufferSize));

    {
        NNT_EXPECT_RESULT_SUCCESS(fs::OpenFile(&file, testFileName, fs::OpenMode::OpenMode_Write));
        NNT_EXPECT_RESULT_SUCCESS(fs::SetFileSize(file, BufferSize * 2));
        fs::CloseFile(file);
        ScopedContextChecker checker(context, true, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/0", buffer.get(), BufferSize));
}

TEST_F(PathBasedFileDataCache, OtherFileSystem)
{
    std::unique_ptr<TestBaseFileSystem> testFs(new TestBaseFileSystem(g_Table, g_TableLength));
    ASSERT_TRUE(testFs != nullptr);
    fs::fsa::Register("test2", std::move(testFs), nullptr, false, true, false);
    NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE("test2");

    auto buffer0 = nnt::fs::util::AllocateBuffer(BufferSize);
    auto buffer1 = nnt::fs::util::AllocateBuffer(BufferSize);
    ASSERT_TRUE(buffer0 != nullptr && buffer1 != nullptr);

    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::WriteFileWith32BitCount("test:/0", BufferSize, 0));
    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test:/0", buffer0.get(), BufferSize));

    auto context = GetContext();
    {
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    {
        ScopedContextChecker checker(context, true, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test2:/0", 0));
    }

    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test2:/0", buffer0.get(), BufferSize));
    fs::DisableIndividualFileDataCache("test:/0");

    {
        ScopedContextChecker checker(context, true, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test:/0", 0));
    }
    {
        ScopedContextChecker checker(context, false, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test2:/0", 0));
    }

    fs::fsa::Unregister("test2");
}

TEST_F(PathBasedFileDataCache, UnsupportedFileSystem)
{
    std::unique_ptr<TestBaseFileSystem> testFs(new TestBaseFileSystem(g_Table, g_TableLength));
    ASSERT_TRUE(testFs != nullptr);
    fs::fsa::Register("test2", std::move(testFs));
    NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE("test2");

    auto buffer = nnt::fs::util::AllocateBuffer(BufferSize);
    ASSERT_TRUE(buffer != nullptr);

    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::WriteFileWith32BitCount("test2:/0", BufferSize, 0));

    NNT_EXPECT_RESULT_SUCCESS(fs::EnableIndividualFileDataCache("test2:/0", buffer.get(), BufferSize));
    auto context = GetContext();
    {
        ScopedContextChecker checker(context, true, false);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("test2:/0", 0));
    }
    fs::DisableIndividualFileDataCache("test2:/0");

    fs::fsa::Unregister("test2");
}

extern "C" void nnMain()
{
    int     argc = os::GetHostArgc();
    char**  argv = os::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    fs::SetEnabledAutoAbort(false);
    nnt::fs::util::ResetAllocateCount();

    NN_ABORT_UNLESS_RESULT_SUCCESS(os::AllocateTlsSlot(&g_tlsSlot,TlsDestructFunction));

    auto ret = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(ret);
}
