﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <memory>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/fs/fs_SubStorage.h>
#include <nn/fs/fs_FileStorage.h>

#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
#include <nn/fat/fat_FatFileSystem.h>
#endif

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

#if !defined(NN_BUILD_CONFIG_OS_WIN32)

TEST(OpenStorageTest, OpenBisPartition)
{
    // Bis の System パーティションを開く
    std::unique_ptr<nn::fs::IStorage> storage;
    NNT_ASSERT_RESULT_SUCCESS(OpenBisPartition(&storage, BisPartitionId::User));

    // FAT フォーマットであることを確認
    auto cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
    std::unique_ptr<nn::fat::FatFileSystem> fatFs(new nn::fat::FatFileSystem());
    std::unique_ptr<char[]> cacheBuffer(new char[cacheBufferSize]);
    NNT_ASSERT_RESULT_SUCCESS(fatFs->Initialize(storage.get(), cacheBuffer.get(), cacheBufferSize));
    NNT_ASSERT_RESULT_SUCCESS(fatFs->Mount());

    NNT_ASSERT_RESULT_SUCCESS(fsa::Register("@System", std::move(fatFs)));

    DirectoryHandle dirHandle;
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenDirectory(&dirHandle, "@System:/", OpenDirectoryMode_All));
    while(NN_STATIC_CONDITION(true))
    {
        int64_t readNum = 0;
        const int entryBufferNum = 3;
        DirectoryEntry dirEntry[entryBufferNum];
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadDirectory(&readNum, dirEntry, dirHandle, entryBufferNum));
        for(int i=0; i<readNum; i++)
        {
            // Check results
            NN_LOG("entry: <%s> %s %lld\n", dirEntry[i].name, dirEntry[i].directoryEntryType == DirectoryEntryType_File ? "File" : "Dir", dirEntry[i].fileSize);
        }
        if( readNum == 0 )
        {
            break;
        }
    }
    nn::fs::CloseDirectory(dirHandle);
    fsa::Unregister("@System");
}

class OpenBisPartitionTest : public ::testing::TestWithParam<BisPartitionId>
{
};

// Bis の BootPartition1, BootPartition2, UserDataRoot を開いて先頭 1 KB のデータを表示する
TEST_P(OpenBisPartitionTest, Read)
{
    std::unique_ptr<nn::fs::IStorage> storage;
    auto bisPartitionId = GetParam();
    NN_LOG("bisPartitionId: %d\n", bisPartitionId);
    NNT_ASSERT_RESULT_SUCCESS(OpenBisPartition(&storage, bisPartitionId));

    const size_t BufferSize = 1024;
    static char buffer[BufferSize];
    std::memset(buffer, 0, sizeof(buffer));

    NNT_ASSERT_RESULT_SUCCESS(storage->Read(0, buffer, BufferSize));

    for (int i = 0; i < static_cast<int>(BufferSize); i++)
    {
        NN_LOG("%02x", buffer[i]);
        if (i % 32 == 31)
        {
            NN_LOG("\n");
        }
    }
    NN_LOG("\n");
}

BisPartitionId rootIds[] = {
    BisPartitionId::BootPartition1Root,
    BisPartitionId::BootPartition2Root,
    BisPartitionId::UserDataRoot
};

BisPartitionId systemPartitionIds[] = {
    BisPartitionId::System,
    BisPartitionId::SystemProperEncryption,
};

INSTANTIATE_TEST_CASE_P(WithRootIds,
                        OpenBisPartitionTest,
                        ::testing::ValuesIn(rootIds));

INSTANTIATE_TEST_CASE_P(WithSystemIds,
                        OpenBisPartitionTest,
                        ::testing::ValuesIn(systemPartitionIds));


#endif


namespace {


class OpenGameCardPartitionTest : public nnt::fs::util::CheckGlobalNewDeleteFlagTestFixture
{
protected:

    virtual void SetUp()
    {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
        MountHostRoot();

        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&xciHandle, "d:/vol/test.xci", OpenMode_Read | OpenMode_Write));
        int64_t xciSize;
        NNT_ASSERT_RESULT_SUCCESS(GetFileSize(&xciSize, xciHandle));
        std::shared_ptr<IStorage> xciStorage(new FileHandleStorage(xciHandle));
        xciBodyStorage.reset(new SubStorage(std::move(xciStorage), 4 * 1024, xciSize - 4 * 1024)); // skip keyarea

        nn::gc::SetStorage(xciBodyStorage.get());
        nn::gc::Attach();
#endif

    }

    virtual void TearDown()
    {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
        NNT_ASSERT_RESULT_SUCCESS(FlushFile(xciHandle));
        CloseFile(xciHandle);
        UnmountHostRoot();
#endif

    }

    FileHandle xciHandle;
    std::shared_ptr<IStorage> xciBodyStorage;

};

}

// TODO: 整理
TEST_F(OpenGameCardPartitionTest, Read_Normal)
{
    const int CardHeaderSize = 0x78 * 512;
    const size_t BufferSize = 1024;
    {
        GameCardHandle handle;
        NNT_ASSERT_RESULT_SUCCESS(GetGameCardHandle(&handle));
        std::unique_ptr<nn::fs::IStorage> storage;
        NNT_ASSERT_RESULT_SUCCESS(OpenGameCardPartition(&storage, handle, GameCardPartitionRaw::NormalReadOnly));

        char buffer[BufferSize];
        InvalidateVariable(buffer, BufferSize);
        NNT_ASSERT_RESULT_SUCCESS(storage->Read(CardHeaderSize, buffer, BufferSize));

        DumpBuffer(buffer, BufferSize);
    }
}

TEST_F(OpenGameCardPartitionTest, Read_Secure)
{
    const size_t BufferSize = 1024;
    {
        GameCardHandle handle;
        NNT_ASSERT_RESULT_SUCCESS(GetGameCardHandle(&handle));
        std::unique_ptr<nn::fs::IStorage> storage;
        NNT_ASSERT_RESULT_SUCCESS(OpenGameCardPartition(&storage, handle, GameCardPartitionRaw::SecureReadOnly));

        char buffer[BufferSize];
        InvalidateVariable(buffer, BufferSize);
        NNT_ASSERT_RESULT_SUCCESS(storage->Read(0, buffer, BufferSize));

        DumpBuffer(buffer, BufferSize);
    }
}


TEST_F(OpenGameCardPartitionTest, WriteRead_Normal)
{
    const int KeyAreaSize = 0x8 * 512;
    const int CardHeaderSize = 0x78 * 512;
    const size_t BufferSize = 1024;

    {
        GameCardHandle handle;
        NNT_ASSERT_RESULT_SUCCESS(GetGameCardHandle(&handle));
        std::unique_ptr<nn::fs::IStorage> storage;
        NNT_ASSERT_RESULT_SUCCESS(OpenGameCardPartition(&storage, handle, GameCardPartitionRaw::RootWriteOnly));

        char buffer[BufferSize];
        FillBufferWith32BitCount(buffer, BufferSize, 0);
        // 0x8 * 512 (KeyArea) + 0x78 * 512 (Card Header + etc) 飛ばして書きこむ
        NNT_ASSERT_RESULT_SUCCESS(storage->Write(KeyAreaSize + CardHeaderSize, buffer, BufferSize));
    }

    {
        GameCardHandle handle;
        NNT_ASSERT_RESULT_SUCCESS(GetGameCardHandle(&handle));
        std::unique_ptr<nn::fs::IStorage> storage;
        NNT_ASSERT_RESULT_SUCCESS(OpenGameCardPartition(&storage, handle, GameCardPartitionRaw::NormalReadOnly));

        char buffer[BufferSize];
        InvalidateVariable(buffer, BufferSize);
        NNT_ASSERT_RESULT_SUCCESS(storage->Read(CardHeaderSize, buffer, BufferSize));

        DumpBuffer(buffer, BufferSize);
    }
}

TEST_F(OpenGameCardPartitionTest, WriteRead_Secure)
{
    const int KeyAreaSize = 0x8 * 512;
    const int SecureAreaOffset = 0x80000 * 512;
    const size_t BufferSize = 1024;

    {
        GameCardHandle handle;
        NNT_ASSERT_RESULT_SUCCESS(GetGameCardHandle(&handle));
        std::unique_ptr<nn::fs::IStorage> storage;
        NNT_ASSERT_RESULT_SUCCESS(OpenGameCardPartition(&storage, handle, GameCardPartitionRaw::RootWriteOnly));

        char buffer[BufferSize];
        FillBufferWith32BitCount(buffer, BufferSize, 0xAB0000);
        //! 0x8 * 512 (KeyArea) + 0x80000 * 512 (NormalArea) 飛ばして書きこむ
        NNT_ASSERT_RESULT_SUCCESS(storage->Write(KeyAreaSize + SecureAreaOffset, buffer, BufferSize));
    }

    {
        GameCardHandle handle;
        NNT_ASSERT_RESULT_SUCCESS(GetGameCardHandle(&handle));
        std::unique_ptr<nn::fs::IStorage> storage;
        NNT_ASSERT_RESULT_SUCCESS(OpenGameCardPartition(&storage, handle, GameCardPartitionRaw::SecureReadOnly));

        char buffer[BufferSize];
        InvalidateVariable(buffer, BufferSize);
        NNT_ASSERT_RESULT_SUCCESS(storage->Read(0, buffer, BufferSize));

        DumpBuffer(buffer, BufferSize);
    }
}


TEST_F(OpenGameCardPartitionTest, GetSize)
{
    int64_t normalAreaSize;
    int64_t secureAreaSize;
    int64_t writableAreaSize;
    const int64_t KeyAreaSize = 0x1000;

    GameCardHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(GetGameCardHandle(&handle));

    {
        std::unique_ptr<nn::fs::IStorage> storage;
        NNT_ASSERT_RESULT_SUCCESS(OpenGameCardPartition(&storage, handle, GameCardPartitionRaw::NormalReadOnly));
        NNT_EXPECT_RESULT_SUCCESS(storage->GetSize(&normalAreaSize));
        NN_LOG("normal area size: %lld\n", normalAreaSize);
    }

    {
        std::unique_ptr<nn::fs::IStorage> storage;
        NNT_ASSERT_RESULT_SUCCESS(OpenGameCardPartition(&storage, handle, GameCardPartitionRaw::SecureReadOnly));
        NNT_EXPECT_RESULT_SUCCESS(storage->GetSize(&secureAreaSize));
        NN_LOG("secure area size: %lld\n", secureAreaSize);
    }

    {
        std::unique_ptr<nn::fs::IStorage> storage;
        NNT_ASSERT_RESULT_SUCCESS(OpenGameCardPartition(&storage, handle, GameCardPartitionRaw::RootWriteOnly));
        NNT_EXPECT_RESULT_SUCCESS(storage->GetSize(&writableAreaSize));
        NN_LOG("writable area size: %lld\n", writableAreaSize);
    }

    EXPECT_EQ(writableAreaSize, KeyAreaSize + normalAreaSize + secureAreaSize);
}


#if !defined(NN_BUILD_CONFIG_OS_WIN32)

class RaceConditionFileAndStorageOperation : public ::testing::Test
{
public:
    struct FileOperationParams
    {
        String pathPrefix;
        char* sourceBuffer;
        char* buffer;
    };

    struct StorageOperationParams
    {
        IStorage* storage;
        char* sourceBuffer;
        char* buffer;
    };

    static void FileOperationThreadFunction(void* arg)
    {
        const int loopNum = 500;
        static const int BufferSize = 512;
        FileOperationParams* params = static_cast<FileOperationParams*>(arg);
        for (int i = 0; i < loopNum; i++)
        {
            FileHandle file;
            String fileName = params->pathPrefix + "RaceConditionTest.dat";
            std::memset(params->buffer, 0, BufferSize);
            DeleteFile(fileName.c_str());
            NNT_ASSERT_RESULT_SUCCESS(CreateFile(fileName.c_str(), BufferSize));
            NNT_ASSERT_RESULT_SUCCESS(OpenFile(&file, fileName.c_str(), OpenMode_Read | OpenMode_Write));
            NNT_ASSERT_RESULT_SUCCESS(WriteFile(file, 0, params->sourceBuffer, BufferSize, WriteOption()));
            NNT_ASSERT_RESULT_SUCCESS(FlushFile(file));
            size_t readSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(ReadFile(&readSize, file, 0, params->buffer, BufferSize, ReadOption()));
            EXPECT_EQ(static_cast<size_t>(BufferSize), readSize);
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(params->buffer, params->sourceBuffer, BufferSize);
            CloseFile(file);
        }
    }

    static void StorageOperationThreadFunction(void* arg)
    {
        const int loopNum = 500;
        static const int BufferSize = 512;
        StorageOperationParams* params = static_cast<StorageOperationParams*>(arg);
        for (int i = 0; i < loopNum; i++)
        {
#if 1 // 自動テスト用
            NNT_ASSERT_RESULT_SUCCESS(params->storage->Read(0, params->buffer, BufferSize));
#else
            std::memset(params->buffer, 0, BufferSize);
            NNT_ASSERT_RESULT_SUCCESS(params->storage->Write(0, params->sourceBuffer, BufferSize));
            NNT_ASSERT_RESULT_SUCCESS(params->storage->Read(0, params->buffer, BufferSize));
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(params->buffer, params->sourceBuffer, BufferSize);
#endif
        }
    }
};

// BootPartition2, Bis, SdCard の FAT のデータを並列読み書きする
TEST_F(RaceConditionFileAndStorageOperation, WriteAndRead)
{
    static const int ThreadCount = 3;
    const size_t StackSize = 64 * 1024;
    static NN_ALIGNAS(4096) uint8_t stack[ThreadCount][StackSize];

    static const int BufferSize = 512;
    static char sourceBuffer[BufferSize];
    for (int i = 0; i < BufferSize; i++)
    {
        sourceBuffer[i] = static_cast<char>(i);
    }

    nn::os::ThreadType threads[ThreadCount];
    FileOperationParams fileParams[2];
    StorageOperationParams storageParam;
    static char buffer[ThreadCount][BufferSize];

    std::unique_ptr<IStorage> storage;
    NNT_ASSERT_RESULT_SUCCESS(OpenBisPartition(&storage, BisPartitionId::BootPartition2Root));
    storageParam.storage = storage.get();
    storageParam.sourceBuffer = sourceBuffer;
    storageParam.buffer = buffer[0];
    NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(&threads[0], StorageOperationThreadFunction, &storageParam, stack[0], StackSize, nn::os::DefaultThreadPriority));
    nn::os::StartThread(&threads[0]);

    NNT_ASSERT_RESULT_SUCCESS(MountBis(BisPartitionId::System, nullptr));
    fileParams[0].pathPrefix = String("@System:/");
    fileParams[0].sourceBuffer = sourceBuffer;
    fileParams[0].buffer = buffer[1];
    NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(&threads[1], FileOperationThreadFunction, &fileParams[0], stack[1], StackSize, nn::os::DefaultThreadPriority));
    nn::os::StartThread(&threads[1]);

    NNT_ASSERT_RESULT_SUCCESS(MountSdCardForDebug("Sdcard"));
    fileParams[1].pathPrefix = String("Sdcard:/");
    fileParams[1].sourceBuffer = sourceBuffer;
    fileParams[1].buffer = buffer[2];
    NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(&threads[2], FileOperationThreadFunction, &fileParams[1], stack[2], StackSize, nn::os::DefaultThreadPriority));
    nn::os::StartThread(&threads[2]);

    for (auto& t : threads)
    {
        nn::os::WaitThread(&t);
        nn::os::DestroyThread(&t);
    }

    Unmount("@System");
    Unmount("Sdcard");
}
#else
TEST(OpenStorageTest, OpenBisPartition)
{
    // Bis の System パーティションを開く
    std::unique_ptr<nn::fs::IStorage> storage;
    NNT_EXPECT_RESULT_FAILURE(ResultPartitionNotFound, OpenBisPartition(&storage, BisPartitionId::System));
}
#endif



#if defined(NN_BUILD_CONFIG_OS_WIN32)
// For GameCardManager.cpp
extern "C" void nninitStartup()
{
    NN_LOG("malloc disabled by %s %d\n", __FUNCTION__, __LINE__);
    const size_t MemoryHeapSize = 1024 * 1024 * 1024;
    nn::os::SetMemoryHeapSize( MemoryHeapSize );
}
#endif


extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    nnt::fs::util::ResetAllocateCount();

    SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nn::fs::SetEnabledAutoAbort(false);

    auto testResult = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
