﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataWithApplicationId.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForApplications.h>

const char* MountName = "testsave";
const char* ReadTestFileName = "readtest.bin";
const char* ReadTestFileData = "ReadOK!!";
const char* WriteTestFileName = "writetest.bin";
const nn::ncm::ApplicationId ApplicationId = {0x0100000000010000};
const nn::ncm::ApplicationId OtherApplicationIdAccess = {0x0100000000010001};
const nn::ncm::ApplicationId OtherApplicationIdNotAccess = {0x0100000000010002};

class TestMountSaveData : public ::testing::Test
{
public:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        // アカウントライブラリを初期化
        nn::account::Initialize();

        nn::Result result;
        g_AccountUid = nn::account::InvalidUid;
        int userCount = 0;
        result = nn::account::ListAllUsers(&userCount, &g_AccountUid, 1);
        NN_ASSERT(result.IsSuccess() && userCount > 0);

        m_UserId._data[0] = g_AccountUid._data[0];
        m_UserId._data[1] = g_AccountUid._data[1];
    }
    static void TearDownTestCase() NN_NOEXCEPT
    {
    }

    static nn::account::Uid g_AccountUid;
    static nn::fs::UserId m_UserId;
};

nn::account::Uid TestMountSaveData::g_AccountUid;
nn::fs::UserId TestMountSaveData::m_UserId;

void CheckMountSaveDataRead()
{
    nn::fs::FileHandle handle;
    char buffer[sizeof(ReadTestFileData)];
    char path[256];
    nn::util::SNPrintf(path, sizeof(path), "%s:/%s", MountName, ReadTestFileName);

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(handle, 0, buffer, sizeof(buffer)));
    nn::fs::CloseFile(handle);

    EXPECT_EQ(0, std::memcmp(buffer, ReadTestFileData, sizeof(buffer)));
}

void CheckMountSaveDataWrite()
{
    char path[256];
    nn::util::SNPrintf(path, sizeof(path), "%s:/%s", MountName, WriteTestFileName);

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile(path, 256));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData(MountName));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile(path));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData(MountName));
}

//!< @brief 自アプリIDのSaveDataをマウント
TEST_F(TestMountSaveData, MountSaveDataReadOnly_SelfSaveData)
{
    nn::Result result = nn::fs::MountSaveDataReadOnly(MountName, ApplicationId, m_UserId);
    NNT_EXPECT_RESULT_SUCCESS(result);
    if (result.IsSuccess())
    {
        CheckMountSaveDataRead();
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData(MountName));
        nn::fs::Unmount(MountName);
    }
}

//!< @brief 他アプリID(SaveDataOwnerIdsに記載あり)のSaveDataをマウント
TEST_F(TestMountSaveData, MountSaveDataReadOnly_OwnerSaveData)
{
    nn::Result result = nn::fs::MountSaveDataReadOnly(MountName, OtherApplicationIdAccess, m_UserId);
    NNT_EXPECT_RESULT_SUCCESS(result);
    if (result.IsSuccess())
    {
        CheckMountSaveDataRead();
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData(MountName));
        nn::fs::Unmount(MountName);
    }
}

//!< @brief 他アプリID(SaveDataOwnerIdsに記載なし)のSaveDataをマウント
TEST_F(TestMountSaveData, MountSaveDataReadOnly_OtherSaveData)
{
    nn::Result result = nn::fs::MountSaveDataReadOnly(MountName, OtherApplicationIdNotAccess, m_UserId);
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPermissionDenied, result);
    if (result.IsSuccess())
    {
        nn::fs::Unmount(MountName);
    }
}

//!< @brief 自アプリIDのSaveDataをマウント
TEST_F(TestMountSaveData, MountSaveData_SelfSaveData)
{
    nn::Result result = nn::fs::MountSaveData(MountName, ApplicationId, m_UserId);
    NNT_EXPECT_RESULT_SUCCESS(result);
    if (result.IsSuccess())
    {
        CheckMountSaveDataRead();
        CheckMountSaveDataWrite();
        nn::fs::Unmount(MountName);
    }
}

//!< @brief 他アプリID(SaveDataOwnerIdsに記載あり)のSaveDataをマウント
TEST_F(TestMountSaveData, MountSaveData_OwnerSaveData)
{
    nn::Result result = nn::fs::MountSaveData(MountName, OtherApplicationIdAccess, m_UserId);
    NNT_EXPECT_RESULT_SUCCESS(result);
    if (result.IsSuccess())
    {
        CheckMountSaveDataRead();
        CheckMountSaveDataWrite();
        nn::fs::Unmount(MountName);
    }
}

//!< @brief 他アプリID(SaveDataOwnerIdsに記載あり)のSaveDataをマウント
TEST_F(TestMountSaveData, MountSaveData_OwnerSaveData_from_User)
{
    nn::ApplicationId applicationId;
    applicationId.value = OtherApplicationIdAccess.value;

    nn::account::Uid uid;
    uid._data[0] = m_UserId._data[0];
    uid._data[1] = m_UserId._data[1];

    nn::Result result = nn::fs::MountSaveData(MountName, applicationId, uid);
    NNT_EXPECT_RESULT_SUCCESS(result);
    if (result.IsSuccess())
    {
        CheckMountSaveDataRead();
        CheckMountSaveDataWrite();
        nn::fs::Unmount(MountName);
    }
}

//!< @brief 他アプリID(SaveDataOwnerIdsに記載なし)のSaveDataをマウント
TEST_F(TestMountSaveData, MountSaveData_OtherSaveData)
{
    nn::Result result = nn::fs::MountSaveData(MountName, OtherApplicationIdNotAccess, m_UserId);
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPermissionDenied, result);
    if (result.IsSuccess())
    {
        nn::fs::Unmount(MountName);
    }
}

namespace
{
    const nnt::fs::util::MountTestAttribute GetAttribute() NN_NOEXCEPT
    {
        nnt::fs::util::MountTestAttribute attribute = {};
        attribute.setUp = TestMountSaveData::SetUpTestCase;
        return attribute;
    }

    // FsLibMountTest の SaveData にもテストあり

    // Result MountSaveData(const char* name, const nn::account::Uid& user)
    nn::Result MountSaveDataType3ForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nnt::fs::util::CreateAndMountSaveData(mountName, TestMountSaveData::g_AccountUid));
        NN_RESULT_SUCCESS;
    }

    // Result MountSaveData(const char* name, nn::ApplicationId applicationId, const nn::account::Uid& user)
    nn::Result MountSaveDataType4ForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        static nn::ApplicationId ApplicationId = { OtherApplicationIdAccess.value };
        NN_RESULT_DO(nnt::fs::util::CreateAndMountSaveData(mountName, ApplicationId, TestMountSaveData::g_AccountUid));
        NN_RESULT_SUCCESS;
    }

    // Result MountSaveDataReadOnly(const char* name, const nn::ApplicationId applicationId, const nn::account::Uid& user)
    nn::Result MountSaveDataReadOnlyType2ForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        static nn::ApplicationId ApplicationId = { OtherApplicationIdAccess.value };
        NN_RESULT_DO(nnt::fs::util::CreateAndMountSaveDataReadOnly(mountName, ApplicationId, TestMountSaveData::g_AccountUid));
        NN_RESULT_SUCCESS;
    }

    const nnt::fs::util::MountTestParameter MountTestParameters[] = {
        { "MountSaveDataType3", MountSaveDataType3ForMountNameTest, nullptr, GetAttribute },
        { "MountSaveDataType4", MountSaveDataType4ForMountNameTest, nullptr, GetAttribute },
        { "MountSaveDataReadOnlyType2", MountSaveDataReadOnlyType2ForMountNameTest, nullptr, GetAttribute },
    };
}

namespace nnt { namespace fs { namespace util {
    NNT_FS_INSTANTIATE_TEST_CASE_MOUNT(WithMountSaveData2, ::testing::ValuesIn(MountTestParameters));
}}}

extern "C" void nnMain()
{
    nnt::fs::util::LoadMountTest();

    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    nnt::fs::util::mock::InitializeForFsTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nn::fs::SetEnabledAutoAbort(false);
    nnt::fs::util::ResetAllocateCount();

    auto result = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(result);
}
