﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_RomPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

using namespace nn;
using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nnt::fs::util;

#include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>

namespace {

const nn::ApplicationId ProperApplicationId = { 0x0100000000010000 };
const nn::ApplicationId NonExistentApplicationId = { 0x0100000000010001 };

// HostFs 版 Rom かどうかを判定
bool IsEmulatedRom()
{
    size_t size;
    NN_ABORT_UNLESS_RESULT_SUCCESS(QueryMountRomCacheSize(&size));
    return size <= 32;
}

TEST(WithRom, CanMountRom)
{
    EXPECT_TRUE( CanMountRomForDebug() );

    EXPECT_FALSE(nnt::fs::util::CheckMemoryLeak());
}

TEST(WithRom, MountRom)
{
    size_t size;
    NNT_EXPECT_RESULT_SUCCESS(QueryMountRomCacheSize(&size));

    auto buffer = AllocateBuffer(size);
    ASSERT_NE(nullptr, buffer);

    const size_t WorkBufferSize = 64 * 1024 * 1024;
    auto workBuffer = AllocateBuffer(WorkBufferSize);

    NNT_ASSERT_RESULT_SUCCESS(MountRom("rom", buffer.get(), size));
    NNT_EXPECT_RESULT_SUCCESS(DumpDirectoryRecursive("rom:/", true, workBuffer.get(), WorkBufferSize));

    DirectoryEntryType type;
    NNT_EXPECT_RESULT_SUCCESS(GetEntryType(&type, "rom:/test/testdata"));
    EXPECT_EQ(DirectoryEntryType_File, type);

    Unmount("rom");

    EXPECT_FALSE(nnt::fs::util::CheckMemoryLeak());
}

#ifndef NN_BUILD_CONFIG_OS_WIN

TEST(WithRom, MountMultipleRom)
{
    const auto MountCountLimit = 10;

    bool isEmulatedRom = IsEmulatedRom();

    {
        size_t cacheSize = 0;
        NNT_EXPECT_RESULT_SUCCESS(QueryMountRomCacheSize(&cacheSize));

        const auto cacheBuffer = AllocateBuffer(cacheSize * (MountCountLimit + 1));
        ASSERT_NE(nullptr, cacheBuffer);

        auto mountCount = 0;

        NN_UTIL_SCOPE_EXIT
        {
            for( auto unmountCount = 0; unmountCount < mountCount; ++unmountCount )
            {
                char mountName[16] = {};
                util::SNPrintf(mountName, sizeof(mountName), "rom%d", unmountCount);
                fs::Unmount(mountName);
            }
        };

        // MountCountLimit だけマウントできる
        for( mountCount = 0; mountCount < MountCountLimit; ++mountCount )
        {
            char mountName[16] = {};
            util::SNPrintf(mountName, sizeof(mountName), "rom%d", mountCount);
            const auto currentCacheBuffer = cacheBuffer.get() + mountCount * cacheSize;
            NNT_ASSERT_RESULT_SUCCESS(MountRom(mountName, currentCacheBuffer, cacheSize));
        }

        if (!isEmulatedRom)
        {
            // MountCountLimit を超えてマウントできない
            {
                char mountName[16] = {};
                util::SNPrintf(mountName, sizeof(mountName), "rom%d", mountCount);
                const auto currentCacheBuffer = cacheBuffer.get() + mountCount * cacheSize;
                NNT_ASSERT_RESULT_FAILURE(
                    fs::ResultOpenCountLimit,
                    MountRom(mountName, currentCacheBuffer, cacheSize));
            }
        }
    }

    EXPECT_FALSE(nnt::fs::util::CheckMemoryLeak());
}

#endif

TEST(WithRom, QueryMountRomCacheSize)
{
    size_t size;
    NNT_EXPECT_RESULT_SUCCESS(QueryMountRomCacheSize(&size));
    NN_LOG("QueryMountRomCacheSize(): %zd\n", size);

    EXPECT_LE(static_cast<size_t>(0), size); // raw では 0 バイト
    EXPECT_LT(size, static_cast<size_t>(256)); // 単一ファイルの小さな RomFs では 116 バイト程度

    EXPECT_FALSE(nnt::fs::util::CheckMemoryLeak());
}

TEST(WithRom, RootParent)
{
    size_t size;
    NNT_ASSERT_RESULT_SUCCESS(QueryMountRomCacheSize(&size));

    auto buffer = AllocateBuffer(size);
    ASSERT_NE(nullptr, buffer);

    NNT_ASSERT_RESULT_SUCCESS(MountRom("rom", buffer.get(), size));

    nn::fs::FileHandle file;
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnobtainable,
        nn::fs::OpenFile(&file, "rom:/..", nn::fs::OpenMode_Read));
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnobtainable,
        nn::fs::OpenFile(&file, "rom:/../", nn::fs::OpenMode_Read));
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnobtainable,
        nn::fs::OpenFile(&file, "rom:/../foo", nn::fs::OpenMode_Read));
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnobtainable,
        nn::fs::OpenFile(&file, "rom:/../foo/", nn::fs::OpenMode_Read));

    Unmount("rom");
}

typedef CheckGlobalNewDeleteFlagTestFixture WithRomCheckGlobalNew;
TEST_F(WithRomCheckGlobalNew, MountRom)
{
    EXPECT_TRUE( CanMountRomForDebug() );

    size_t size;
    NNT_ASSERT_RESULT_SUCCESS(QueryMountRomCacheSize(&size));

    auto buffer = AllocateBuffer(size);
    ASSERT_NE(nullptr, buffer);

    NNT_ASSERT_RESULT_SUCCESS(MountRom("rom", buffer.get(), size));
    Unmount("rom");

    EXPECT_FALSE(nnt::fs::util::CheckMemoryLeak());
}

typedef CheckGlobalNewDeleteFlagTestFixture WithRomCheckGlobalNew;
TEST_F(WithRomCheckGlobalNew, MountProperRom)
{
#ifndef NN_BUILD_CONFIG_OS_WIN
    EXPECT_TRUE( CanMountRom(ProperApplicationId) );
#else
    EXPECT_FALSE( CanMountRom(ProperApplicationId) );
#endif

    size_t size;
    NNT_ASSERT_RESULT_SUCCESS(QueryMountRomCacheSize(&size, ProperApplicationId));

    auto buffer = AllocateBuffer(size);
    ASSERT_NE(nullptr, buffer);

#ifndef NN_BUILD_CONFIG_OS_WIN
    NNT_ASSERT_RESULT_SUCCESS(MountRom("rom", ProperApplicationId, buffer.get(), size));
    Unmount("rom");
#else
    NNT_ASSERT_RESULT_FAILURE(ResultNotImplemented, MountRom("rom", NonExistentApplicationId, buffer.get(), size));
#endif

    EXPECT_FALSE(nnt::fs::util::CheckMemoryLeak());
}

TEST(WithNonExistentRom, Basic)
{
    EXPECT_FALSE( CanMountRom(NonExistentApplicationId) );

    size_t size;
    NNT_ASSERT_RESULT_SUCCESS(QueryMountRomCacheSize(&size, NonExistentApplicationId));

    auto buffer = AllocateBuffer(size);
    ASSERT_NE(nullptr, buffer);

#ifndef NN_BUILD_CONFIG_OS_WIN
    NNT_ASSERT_RESULT_FAILURE(ResultProgramInfoNotFound, MountRom("rom", NonExistentApplicationId, buffer.get(), size));
#else
    NNT_ASSERT_RESULT_FAILURE(ResultNotImplemented, MountRom("rom", NonExistentApplicationId, buffer.get(), size));
#endif
}

}

namespace
{
    nn::Result MountRomForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        size_t size;
        NN_RESULT_DO(QueryMountRomCacheSize(&size));

        auto buffer = AllocateBuffer(size);
        NN_ABORT_UNLESS(buffer);

        NN_RESULT_DO(nn::fs::MountRom(mountName, buffer.get(), size));
        NN_RESULT_SUCCESS;
    }

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    nn::Result MountRomWithApplicationIdForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        size_t size;
        NN_RESULT_DO(QueryMountRomCacheSize(&size, ProperApplicationId));

        auto buffer = AllocateBuffer(size);
        NN_ABORT_UNLESS(buffer);

        NN_RESULT_DO(nn::fs::MountRom(mountName, ProperApplicationId, buffer.get(), size));
        NN_RESULT_SUCCESS;
    }
#endif // !defined(NN_BUILD_CONFIG_OS_WIN)

    const MountTestParameter MountTestParameters[] = {
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        { "MountRomWithApplicationId", MountRomWithApplicationIdForMountNameTest },
#endif // !defined(NN_BUILD_CONFIG_OS_WIN)
        { "MountRom", MountRomForMountNameTest }
    };
}

NNT_FS_INSTANTIATE_TEST_CASE_MOUNT(WithMountRom, ::testing::ValuesIn(MountTestParameters));

extern "C" void nnMain()
{
    nnt::fs::util::LoadMountTest();

    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    nnt::fs::util::mock::InitializeForFsTest(&argc, argv);

    SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nn::fs::SetEnabledAutoAbort(false);
    nnt::fs::util::ResetAllocateCount();

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
