﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>

#include <nn/nn_Result.h>
#include <nn/util/util_FormatString.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SaveData.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataForDebug.h>
#include <nn/fs/fs_SaveDataManageMent.h>
#include <nn/fs/fs_SaveDataManageMentPrivate.h>
#include <nn/fs/fs_CacheStorage.h>
#include <nn/fs/fs_DeviceSaveData.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_SdCardForDebug.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

namespace {

const int MaxNameLength = 64;
const int OpenHandleMax = 1024;

const int MountSaveLimit = 10;
const int OpenSaveLimit = 256;
const char* OpenSaveTestFileName = "TestOpen";

class MountSaveMulti : public ::testing::Test
{
public:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nnt::fs::util::DeleteAllTestSaveData();
        const int64_t Size = 1024 * 1024;
        const char* MountName = "test";
        char fileName[MaxNameLength];
        nn::util::SNPrintf(fileName, sizeof(fileName), "%s:/%s", MountName, OpenSaveTestFileName);

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateTemporaryStorage(nnt::fs::util::ApplicationId, 0, Size, 0));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountTemporaryStorage(MountName));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(fileName, 1024));
        nn::fs::Unmount(MountName);

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateCacheStorage(nnt::fs::util::ApplicationId, nn::fs::SaveDataSpaceId::SdUser, 0, Size, Size, 0));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountCacheStorage(MountName));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(fileName, 1024));
        nn::fs::Commit(MountName);
        nn::fs::Unmount(MountName);

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDeviceSaveData(nnt::fs::util::ApplicationId, 0, Size, Size, 0));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountDeviceSaveData(MountName));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(fileName, 1024));
        nn::fs::Commit(MountName);
        nn::fs::Unmount(MountName);

        for (int i = 0; i <= MountSaveLimit + 1; i++)
        {
            nn::fs::UserId userId = {{0, static_cast<nn::Bit64>(i)}};
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::ApplicationId, userId, 0, Size, Size, 0));
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(MountName, userId));
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(fileName, 1024));
            nn::fs::Commit(MountName);
            nn::fs::Unmount(MountName);
        }
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
        nnt::fs::util::DeleteAllTestSaveData();
    }
};

void TryMount(nn::Result expectResult)
{
    const char* MountName = "test";
    nn::fs::UserId userId = {{0, MountSaveLimit + 1}};

    auto check = [=](nn::Result result) {
        EXPECT_EQ(expectResult.GetInnerValueForDebug(), result.GetInnerValueForDebug());
        if (result.IsSuccess())
        {
            nn::fs::Unmount(MountName);
        }
    };

    check(nn::fs::MountTemporaryStorage(MountName));
    check(nn::fs::MountCacheStorage(MountName));
    check(nn::fs::MountDeviceSaveData(MountName));
    check(nn::fs::MountSaveData(MountName, userId));
    check(nn::fs::MountSaveDataReadOnly(MountName, nnt::fs::util::ApplicationId, userId));
    check(nn::fs::MountSaveDataForDebug(MountName));
}

void MountTestSave(int count)
{
    char mountName[MaxNameLength];
    for (int i = 0; i < count; i++)
    {
        nn::fs::UserId userId = {{0, static_cast<nn::Bit64>(i + 1)}};
        nn::util::SNPrintf(mountName, sizeof(mountName), "save%d", i);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(mountName, userId));
    }
}

void UnmountTestSave(int count)
{
    char mountName[MaxNameLength];
    for (int i = 0; i < count; i++)
    {
        nn::util::SNPrintf(mountName, sizeof(mountName), "save%d", i);
        nn::fs::Unmount(mountName);
    }
}

TEST_F(MountSaveMulti, Mount10)
{
    MountTestSave(MountSaveLimit - 1);
    TryMount(nn::ResultSuccess());
    UnmountTestSave(MountSaveLimit - 1);
}

TEST_F(MountSaveMulti, Mount11)
{
    MountTestSave(MountSaveLimit);
    TryMount(nn::fs::ResultOpenCountLimit());
    UnmountTestSave(MountSaveLimit);
}

void TryOpen(nn::Result expectResult)
{
    const char* MountName = "test";
    nn::fs::UserId userId = {{0, MountSaveLimit + 1}};

    auto check = [=](nn::Result result) {
        NNT_ASSERT_RESULT_SUCCESS(result);

        char fileName[MaxNameLength];
        nn::util::SNPrintf(fileName, sizeof(fileName), "%s:/%s", MountName, OpenSaveTestFileName);
        nn::fs::FileHandle handle;
        auto openResult = nn::fs::OpenFile(&handle, fileName, nn::fs::OpenMode_Read);
        EXPECT_EQ(expectResult.GetInnerValueForDebug(), openResult.GetInnerValueForDebug());
        if (openResult.IsSuccess())
        {
           nn::fs::CloseFile(handle);
        }
        nn::fs::Unmount(MountName);
    };

    check(nn::fs::MountTemporaryStorage(MountName));
    check(nn::fs::MountCacheStorage(MountName));
    check(nn::fs::MountDeviceSaveData(MountName));
    check(nn::fs::MountSaveData(MountName, userId));
    check(nn::fs::MountSaveDataReadOnly(MountName, nnt::fs::util::ApplicationId, userId));
    check(nn::fs::MountSaveDataForDebug(MountName));
}

void OpenTestSave(nn::fs::FileHandle handle[], int count)
{
    const char* MountName = "dummy";
    nn::fs::UserId userId = {{0, 1}};
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(MountName, userId));

    char fileName[MaxNameLength];
    nn::util::SNPrintf(fileName, sizeof(fileName), "%s:/%s", MountName, OpenSaveTestFileName);
    for (int i = 0; i < count; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handle[i], fileName, nn::fs::OpenMode_Read));
    }
}

void CloseTestSave(nn::fs::FileHandle handle[], int count)
{
    for (int i = 0; i < count; i++)
    {
        nn::fs::CloseFile(handle[i]);
    }
    nn::fs::Unmount("dummy");
}

TEST_F(MountSaveMulti, Open255)
{
    nn::fs::FileHandle handle[OpenSaveLimit - 1];
    OpenTestSave(handle, OpenSaveLimit - 1);
    TryOpen(nn::ResultSuccess());
    CloseTestSave(handle, OpenSaveLimit - 1);
}

TEST_F(MountSaveMulti, Open256)
{
    nn::fs::FileHandle handle[OpenSaveLimit];
    OpenTestSave(handle, OpenSaveLimit);
    TryOpen(nn::fs::ResultOpenCountLimit());
    CloseTestSave(handle, OpenSaveLimit);
}

class MountOpenLimitTest : public ::testing::Test
{
public:
    explicit MountOpenLimitTest(const char* prefix) NN_NOEXCEPT
        : m_MountCount(0), m_OpenCount(0)
    {
        std::strncpy(m_MountNamePrefix, prefix, MaxNameLength);
    }

    virtual ~MountOpenLimitTest() NN_NOEXCEPT
    {
        Cleanup();
    }

    void Cleanup()
    {
        while (m_OpenCount > 0)
        {
            CloseTestFile();
        }
        while (m_MountCount > 0)
        {
            Unmount();
        }
    }

    nn::Result Mount() NN_NOEXCEPT
    {
        char mountName[MaxNameLength];
        nn::util::SNPrintf(mountName, sizeof(mountName), "%s%d", m_MountNamePrefix, m_MountCount);
        auto result = MountImpl(mountName);
        if (result.IsSuccess())
        {
            m_MountCount++;
        }
        return result;
    }

    void Unmount() NN_NOEXCEPT
    {
        ASSERT_TRUE(m_MountCount > 0);
        char mountName[MaxNameLength];
        m_MountCount--;
        nn::util::SNPrintf(mountName, sizeof(mountName), "%s%d", m_MountNamePrefix, m_MountCount);
        UnmountImpl(mountName);
    }

    int GetMountCount()
    {
        return m_MountCount;
    }

    nn::Result OpenTestFile()
    {
        if (m_OpenCount >= OpenHandleMax)
        {
            return nn::fs::ResultNotImplemented();
        }
        if (m_OpenCount == 0)
        {
            CreateTestFile();
        }
        char filePath[MaxNameLength];
        nn::util::SNPrintf(filePath, sizeof(filePath), "%s0:/%s", m_MountNamePrefix, OpenSaveTestFileName);
        auto result = nn::fs::OpenFile(&m_Handle[m_OpenCount], filePath, nn::fs::OpenMode_Read);
        if (result.IsSuccess())
        {
            m_OpenCount++;
        }
        return result;
    }

    void CloseTestFile()
    {
        ASSERT_TRUE(m_OpenCount > 0);
        m_OpenCount--;
        nn::fs::CloseFile(m_Handle[m_OpenCount]);
        if (m_OpenCount == 0)
        {
            DeleteTestFile();
        }
    }

    int GetOpenCount()
    {
        return m_OpenCount;
    }

private:
    virtual nn::Result MountImpl(const char* mountName) NN_NOEXCEPT
    {
        return nn::fs::ResultNotImplemented();
    }

    virtual void UnmountImpl(const char* mountName) NN_NOEXCEPT
    {
        nn::fs::Unmount(mountName);
    }

    nn::Result CreateTestFile()
    {
        char filePath[MaxNameLength];
        nn::util::SNPrintf(filePath, sizeof(filePath), "%s0:/%s", m_MountNamePrefix, OpenSaveTestFileName);
        return nn::fs::CreateFile(filePath, 1024);
    }

    nn::Result DeleteTestFile()
    {
        char filePath[MaxNameLength];
        nn::util::SNPrintf(filePath, sizeof(filePath), "%s0:/%s", m_MountNamePrefix, OpenSaveTestFileName);
        return nn::fs::DeleteFile(filePath);
    }

private:
    char m_MountNamePrefix[MaxNameLength];
    int m_MountCount;
    int m_OpenCount;
    nn::fs::FileHandle m_Handle[OpenHandleMax];
};

class MountOpenLimitSdTest : public MountOpenLimitTest
{
public:
    MountOpenLimitSdTest() NN_NOEXCEPT
        : MountOpenLimitTest("sd")
    {
        Mount();
    }

private:
    virtual nn::Result MountImpl(const char* mountName) NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::fs::MountSdCardForDebug(mountName);
    }
};

class MountOpenLimitHostTest : public MountOpenLimitTest
{
public:
    MountOpenLimitHostTest() NN_NOEXCEPT
        : MountOpenLimitTest("host")
    {
        m_HostDir.Create();
        Mount();
    }

    ~MountOpenLimitHostTest() NN_NOEXCEPT
    {
        Cleanup();
        m_HostDir.Delete();
    }

private:
    virtual nn::Result MountImpl(const char* mountName) NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::fs::MountHost(mountName, m_HostDir.GetPath().c_str());
    }

    nnt::fs::util::TemporaryHostDirectory m_HostDir;
};

TEST_F(MountOpenLimitSdTest, MountLimitCount)
{
    nn::Result result;
    do {
        result = Mount();
    } while (result.IsSuccess());

    int limit = GetMountCount();
    NN_LOG("Limit=%d, Result=%08x\n", limit, result.GetInnerValueForDebug());
    EXPECT_LT(10, limit);
}

TEST_F(MountOpenLimitHostTest, MountLimitCount)
{
    nn::Result result;
    do {
        result = Mount();
    } while (result.IsSuccess());

    int limit = GetMountCount();
    NN_LOG("Limit=%d, Result=%08x\n", limit, result.GetInnerValueForDebug());
    EXPECT_LT(10, limit);
}

TEST_F(MountOpenLimitSdTest, OpenLimitCount)
{
    nn::Result result;
    do {
        result = OpenTestFile();
    } while (result.IsSuccess());

    int limit = GetOpenCount();
    NN_LOG("Limit=%d, Result=%08x\n", limit, result.GetInnerValueForDebug());
    EXPECT_LT(100, limit);
}

TEST_F(MountOpenLimitHostTest, OpenLimitCount)
{
    nn::Result result;
    do {
        result = OpenTestFile();
    } while (result.IsSuccess());

    int limit = GetOpenCount();
    NN_LOG("Limit=%d, Result=%08x\n", limit, result.GetInnerValueForDebug());
    EXPECT_LT(100, limit);
}

}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nn::fs::SetEnabledAutoAbort(false);
    nnt::fs::util::ResetAllocateCount();

    auto result = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(result);
}
