﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <algorithm>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/os.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_Optional.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/fs/fs_SystemData.h>
#include <nnt/nnt_Argument.h>
#include <nn/account.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_InstallApi.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentMetaKey.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/fs/fs_SubStorage.h>

#include <nn/time/time_PosixTime.h>

#include <nn/aoc.h>
#include <nn/fs/fs_AddOnContent.h>
#include <nn/fs/fs_AddOnContentPrivate.h>

#include <nnt/fsUtil/testFs_util_function.h>

using namespace nn::fs;

#include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>

TEST(Mount, MountAddOnContentNotFound)
{
    const nn::ApplicationId Application = {0x0000000000000000}; // 存在しない Aoc
    const nn::aoc::AddOnContentIndex Index = 1;

    char dummyBuffer[32];

    NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, MountAddOnContent("aoc", Application, Index, dummyBuffer, sizeof(dummyBuffer)));
}

namespace
{
    void GetArg(
        const char** pSourceDirectoryPath,
        const char** pNspDirectoryPath
    ) NN_NOEXCEPT
    {
        const int argc = nnt::GetHostArgc();
        char** argv = nnt::GetHostArgv();

        // google test の引数が argv から取り除かれて null になっているため
        // argc を実際の argv の個数に合わせる
        int argcRemoved = 0;
        for( int i = 0; i < argc && argv[i] != nullptr; ++i )
        {
            ++argcRemoved;
        }

        if( argcRemoved < 3 )
        {
            NN_LOG("Usage: *.exe <Source directory path> <Nsp directory path>\n");
            return;
        }

        if( pSourceDirectoryPath != nullptr )
        {
            *pSourceDirectoryPath = argv[1];
        }
        if( pNspDirectoryPath != nullptr )
        {
            *pNspDirectoryPath = argv[2];
        }
    }

    nn::Result InstallAddOnContent(
        const char* filePath,
        nn::ncm::StorageId storage
    ) NN_NOEXCEPT
    {
        NN_LOG("Install %s.\n", filePath);

        nn::fs::FileHandle file;
        NN_RESULT_DO(nn::fs::OpenFile(&file, filePath, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(file);
        };

        static char s_Buffer[16 * 1024 * 1024];
        nn::ns::ApplicationInstallTask task;
        NN_RESULT_DO(task.Initialize(file, storage, s_Buffer, sizeof(s_Buffer)));
        NN_RESULT_DO(task.Prepare());
        auto progress = task.GetProgress();
        NN_LOG("Prepared. progress.installedSize %lld, progress.totalSize %lld\n",
               progress.installedSize,
               progress.totalSize);

        for( int offset = 0; ; )
        {
            nn::ncm::ApplicationContentMetaKey key[16];
            const int count = sizeof(key) / sizeof(key[0]);
            int outCount;
            NN_RESULT_DO(task.ListApplicationContentMetaKey(&outCount, key, count, offset));
            for( int i = 0; i < outCount; i++ )
            {
                NN_LOG("  ApplicationId 0x%016llx, ContentMetaType %u, "
                       "ContentMetaId 0x%016llx, ContentMetaVersion %u\n",
                       key[i].applicationId.value,
                       static_cast<uint32_t>(key[i].key.type),
                       key[i].key.id,
                       key[i].key.version);
            }

            if( outCount < count )
            {
                break;
            }

            offset += outCount;
        }
        NN_RESULT_DO(task.Execute());
        progress = task.GetProgress();
        NN_LOG("Installed. progress.installedSize %lld, progress.totalSize %lld\n",
            progress.installedSize, progress.totalSize);
        NN_RESULT_DO(task.Commit());

        NN_RESULT_SUCCESS;
    }

    class AddOnContentTest : public testing::Test
    {
    protected:
        static const nn::Bit64 ApplicationIdValue = 0x1000000000003000;

        virtual void SetUp()
        {
            nn::ncm::Initialize();
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHostRoot());

            nn::ns::Initialize();

            // インストールする AddOnContent と同じ ID がすでにインストールされていれば削除する
            for( int offset = 0; ; )
            {
                static const int RecordBufferCount = 14; // 配列のサイズが 256byte を超えない数
                nn::ns::ApplicationRecord recordBuffer[RecordBufferCount];
                const int readCount
                    = nn::ns::ListApplicationRecord(recordBuffer, RecordBufferCount, offset);
                for( int i = 0; i < readCount; ++i )
                {
                    if( recordBuffer[i].id.value == ApplicationIdValue )
                    {
                        const nn::ncm::ApplicationId applicationId = {ApplicationIdValue};
                        NN_ABORT_UNLESS_RESULT_SUCCESS(
                            nn::ns::DeleteApplicationCompletely(applicationId));
                    }
                }
                if( readCount < RecordBufferCount )
                {
                    break;
                }
                offset += readCount;
            }

            const char* nspDirectoryPath;
            GetArg(nullptr, &nspDirectoryPath);

            char addOnContent1NspPath[256];
            std::sprintf(addOnContent1NspPath, "%s/AddOnContent1.nsp", nspDirectoryPath);
            NN_LOG("%s\n", addOnContent1NspPath);
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                InstallAddOnContent(addOnContent1NspPath, nn::ncm::StorageId::BuildInUser));

            char addOnContent2NspPath[256];
            std::sprintf(addOnContent2NspPath, "%s/AddOnContent2.nsp", nspDirectoryPath);
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                InstallAddOnContent(addOnContent2NspPath, nn::ncm::StorageId::BuildInUser));

        }

        virtual void TearDown()
        {
            const nn::ncm::ApplicationId applicationId = {ApplicationIdValue};
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ns::DeleteApplicationCompletely(applicationId));

            nn::ns::Finalize();
            nn::fs::UnmountHostRoot();
            nn::ncm::Finalize();
        }
    };
}

// @pre
//      - {0x1000000000003000, 0} が未インストール
//      - {0x1000000000003000, 1} が BisUser にインストール済み
//      - {0x1000000000003000, 2} が SD にインストール済み
//      // - {0x1000000000003000, 3} が Card に書き込み済み

TEST_F(AddOnContentTest, MountAddOnContent)
{
    const nn::ApplicationId applicationId = {ApplicationIdValue};

    char dummyBuffer[32];
    NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, MountAddOnContent("aoc", applicationId, 0, dummyBuffer, sizeof(dummyBuffer)));

    {
        size_t size;
        NNT_EXPECT_RESULT_SUCCESS(QueryMountAddOnContentCacheSize(&size, applicationId, 1));
        auto buffer = nnt::fs::util::AllocateBuffer(size);

        NNT_ASSERT_RESULT_SUCCESS(MountAddOnContent("aoc", applicationId, 1, buffer.get(), size));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("aoc:/"));
        Unmount("aoc");
    }

    {
        size_t size;
        NNT_EXPECT_RESULT_SUCCESS(QueryMountAddOnContentCacheSize(&size, applicationId, 1));
        auto buffer = nnt::fs::util::AllocateBuffer(size);

        NNT_EXPECT_RESULT_SUCCESS(MountAddOnContent("aoc", applicationId, 2, buffer.get(), size));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("aoc:/"));
        Unmount("aoc");
    }

    //NNT_EXPECT_RESULT_SUCCESS(MountAddOnContent("aoc", BaseId, 3));
    //NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("aoc:/"));
    //Unmount("aoc");
}

TEST_F(AddOnContentTest, MountAddOnContentVerify)
{
    const int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    // google test の引数が argv から取り除かれて null になっているため
    // argc を実際の argv の個数に合わせる
    int argcRemoved = 0;
    for( int i = 0; i < argc && argv[i] != nullptr; ++i )
    {
        ++argcRemoved;
    }

    if( argcRemoved < 2 )
    {
        NN_LOG("Usage: *.exe <source directory path>\n");
        return;
    }

    const nn::ApplicationId applicationId = { ApplicationIdValue };

    size_t size;
    NNT_EXPECT_RESULT_SUCCESS(QueryMountAddOnContentCacheSize(&size, applicationId, 1));
    auto buffer = nnt::fs::util::AllocateBuffer(size);

    bool verifyResult = false;
    NNT_ASSERT_RESULT_SUCCESS(
        nnt::fs::util::VerifyMount(
            &verifyResult,
            [&](const char* mountName) NN_NOEXCEPT
            {
                return MountAddOnContent(mountName, applicationId, 1, buffer.get(), size);
            },
            argv[1],
            "mount",
            "mount:/"
        )
    );

    ASSERT_TRUE(verifyResult);
}

TEST(Mount, MountSystemDataNotFound)
{
    static const nn::ncm::SystemDataId Id = {0x100000000000FFFF}; // 存在しないシステムデータ

    NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, MountSystemData("systemData", Id));
}

namespace
{
    nn::Result InstallSystemData(const char* filePath, nn::ncm::StorageId storage) NN_NOEXCEPT
    {
        NN_LOG("Install %s.\n", filePath);

        nn::fs::FileHandle file;
        NN_RESULT_DO(nn::fs::OpenFile(&file, filePath, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{nn::fs::CloseFile(file);};

        static char g_Buffer[16 * 1024 * 1024];
        nn::ncm::SubmissionPackageInstallTask task;
        NN_RESULT_DO(task.Initialize(file, storage, g_Buffer, sizeof(g_Buffer)));
        NN_RESULT_DO(task.Prepare());
        auto progress = task.GetProgress();
        NN_LOG("Prepared. progress.installedSize %lld, progress.totalSize %lld\n",
            progress.installedSize, progress.totalSize);
        NN_RESULT_DO(task.Execute());
        progress = task.GetProgress();
        NN_LOG("Installed. progress.installedSize %lld, progress.totalSize %lld\n",
            progress.installedSize, progress.totalSize);
        NN_RESULT_DO(task.Commit());

        NN_RESULT_SUCCESS;
    }

    nn::Result DeleteContent(nn::Bit64 id, nn::ncm::StorageId storageId)
    {
        nn::ncm::ContentMetaDatabase db;
        nn::ncm::ContentStorage storage;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ncm::OpenContentMetaDatabase(&db, storageId));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ncm::OpenContentStorage(&storage, storageId));
        nn::ncm::ContentManagerAccessor accessor(&db, &storage);

        return accessor.DeleteAll(id);
    }
}

class SystemDataTest : public testing::Test
{
protected:
    static const nn::Bit64 SystemDataIdValue = 0x1000000000002000;

    virtual void SetUp()
    {
        nn::ncm::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHostRoot());

        nn::ns::Initialize();

        DeleteContent(SystemDataIdValue, nn::ncm::StorageId::BuildInSystem);

        const char* nspDirectoryPath;
        GetArg(nullptr, &nspDirectoryPath);

        char nspPath[256];
        std::sprintf(nspPath, "%s/SystemData.nsp", nspDirectoryPath);
        NN_ABORT_UNLESS_RESULT_SUCCESS(InstallSystemData(nspPath, nn::ncm::StorageId::BuildInSystem));
    }

    virtual void TearDown()
    {
        DeleteContent(SystemDataIdValue, nn::ncm::StorageId::BuildInSystem);

        nn::ns::Finalize();
        nn::fs::UnmountHostRoot();
        nn::ncm::Finalize();
    }
};

TEST_F(SystemDataTest, MountSystemData)
{
    const nn::ncm::SystemDataId systemDataId = {SystemDataIdValue};
    NN_LOG("Mount system data <0x%16llx>.\n", systemDataId.value);
    NNT_ASSERT_RESULT_SUCCESS(MountSystemData("systemData", systemDataId));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("systemData:/"));
    Unmount("systemData");
}

TEST_F(SystemDataTest, MountSystemDataVerify)
{
    const int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    // google test の引数が argv から取り除かれて null になっているため
    // argc を実際の argv の個数に合わせる
    int argcRemoved = 0;
    for( int i = 0; i < argc && argv[i] != nullptr; ++i )
    {
        ++argcRemoved;
    }

    if( argcRemoved < 2 )
    {
        NN_LOG("Usage: *.exe <source directory path>\n");
        return;
    }

    bool verifyResult = false;
    NNT_ASSERT_RESULT_SUCCESS(
        nnt::fs::util::VerifyMount(
            &verifyResult,
            [](const char* mountName) NN_NOEXCEPT
            {
                const nn::ncm::SystemDataId systemDataId = {SystemDataIdValue};
                return MountSystemData(mountName, systemDataId);
            },
            argv[1],
            "mount",
            "mount:/"
        )
    );

    ASSERT_TRUE(verifyResult);
}

namespace
{
    class SystemDataTestForFsUtilMountTest : public SystemDataTest
    {
    public:
        static const auto SystemDataIdValue = SystemDataTest::SystemDataIdValue;

        SystemDataTestForFsUtilMountTest() NN_NOEXCEPT
            : m_Buffer(nullptr, nnt::fs::util::DeleterBuffer)
        {
        }

        virtual void TestBody() NN_NOEXCEPT NN_OVERRIDE
        {
        }

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            SystemDataTest::SetUp();
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            SystemDataTest::TearDown();
            m_Buffer.reset();
        }

        char* AllocateBuffer(size_t size) NN_NOEXCEPT
        {
            m_Buffer = nnt::fs::util::AllocateBuffer(size);
            return m_Buffer.get();
        }

    private:
        decltype(nnt::fs::util::AllocateBuffer(0)) m_Buffer;
    };

    class AddOnContentTestForFsUtilMountTest : public AddOnContentTest
    {
    public:
        static const auto ApplicationIdValue = AddOnContentTest::ApplicationIdValue;

        AddOnContentTestForFsUtilMountTest() NN_NOEXCEPT
            : m_Buffer(nullptr, nnt::fs::util::DeleterBuffer)
        {
        }

        virtual void TestBody() NN_NOEXCEPT NN_OVERRIDE
        {
        }

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            AddOnContentTest::SetUp();
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            AddOnContentTest::TearDown();
            m_Buffer.reset();
        }

        char* AllocateBuffer(size_t size) NN_NOEXCEPT
        {
            m_Buffer = nnt::fs::util::AllocateBuffer(size);
            return m_Buffer.get();
        }

    private:
        decltype(nnt::fs::util::AllocateBuffer(0)) m_Buffer;
    };

    SystemDataTestForFsUtilMountTest g_SystemDataTest;
    AddOnContentTestForFsUtilMountTest g_AddOnContentTest;

    void SetUpForSystemData() NN_NOEXCEPT
    {
        nnt::fs::util::SetStackTraceDumpOnGlobalNewDeleteCallEnabled(false);
        g_SystemDataTest.SetUp();
        nnt::fs::util::ResetGlobalNewDeleteFlag();
        nnt::fs::util::SetStackTraceDumpOnGlobalNewDeleteCallEnabled(true);
    }

    void TearDownForSystemData() NN_NOEXCEPT
    {
        g_SystemDataTest.TearDown();
    }

    const nnt::fs::util::MountTestAttribute GetAttributeForSystemData() NN_NOEXCEPT
    {
        nnt::fs::util::MountTestAttribute attribute = {};
        attribute.setUp = SetUpForSystemData;
        attribute.tearDowm = TearDownForSystemData;
        return attribute;
    }

    void SetUpForAddOnContent() NN_NOEXCEPT
    {
        nnt::fs::util::SetStackTraceDumpOnGlobalNewDeleteCallEnabled(false);
        g_AddOnContentTest.SetUp();
        nnt::fs::util::ResetGlobalNewDeleteFlag();
        nnt::fs::util::SetStackTraceDumpOnGlobalNewDeleteCallEnabled(true);
    }

    void TearDownForAddOnContent() NN_NOEXCEPT
    {
        g_AddOnContentTest.TearDown();
    }

    const nnt::fs::util::MountTestAttribute GetAttributeForAddOnContent() NN_NOEXCEPT
    {
        nnt::fs::util::MountTestAttribute attribute = {};
        attribute.setUp = SetUpForAddOnContent;
        attribute.tearDowm = TearDownForAddOnContent;
        return attribute;
    }

    // Result MountSystemData(const char* name, nn::ncm::SystemDataId dataId)
    nn::Result MountSystemDataType1ForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        const nn::ncm::SystemDataId systemDataId = { SystemDataTestForFsUtilMountTest::SystemDataIdValue };
        NN_RESULT_DO(nn::fs::MountSystemData(mountName, systemDataId));
        NN_RESULT_SUCCESS;
    }

    // Result MountSystemData(const char* name, nn::ncm::SystemDataId dataId, void* pFileSystemCacheBuffer, size_t fileSystemCacheBufferSize)
    nn::Result MountSystemDataType2ForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        const nn::ncm::SystemDataId systemDataId = { SystemDataTestForFsUtilMountTest::SystemDataIdValue };

        size_t size;
        NN_RESULT_DO(nn::fs::QueryMountSystemDataCacheSize(&size, systemDataId));
        auto pBuffer = g_SystemDataTest.AllocateBuffer(size);

        NN_RESULT_DO(nn::fs::MountSystemData(mountName, systemDataId, pBuffer, size));
        NN_RESULT_SUCCESS;
    }

    // Result MountAddOnContent(
    //     const char* name,
    //     nn::aoc::AddOnContentIndex targetIndex,
    //     void* pFileSystemCacheBuffer,
    //     size_t fileSystemCacheBufferSize
    // )
    // → testFs_FsLib_MountData2

    // Result MountAddOnContent( const char*                 name,
    //                           nn::ApplicationId           targetApplication,
    //                           nn::aoc::AddOnContentIndex  targetIndex,
    //                           void*                       pFileSystemCacheBuffer,
    //                           size_t                      fileSystemCacheBufferSize)
    nn::Result MountAddOnContentType2ForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        const nn::ApplicationId applicationId = { AddOnContentTestForFsUtilMountTest::ApplicationIdValue };

        size_t size;
        NN_RESULT_DO(nn::fs::QueryMountAddOnContentCacheSize(&size, applicationId, 1));
        auto pBuffer = g_AddOnContentTest.AllocateBuffer(size);

        NN_RESULT_DO(nn::fs::MountAddOnContent(mountName, applicationId, 1, pBuffer, size));
        NN_RESULT_SUCCESS;
    }

    const nnt::fs::util::MountTestParameter MountTestParameters[] = {
        { "MountSystemDataType1", MountSystemDataType1ForMountNameTest, nullptr, GetAttributeForSystemData },
        { "MountSystemDataType2", MountSystemDataType2ForMountNameTest, nullptr, GetAttributeForSystemData },
        { "MountAddOnContentType2", MountAddOnContentType2ForMountNameTest, nullptr, GetAttributeForAddOnContent }
    };
}

namespace nnt { namespace fs { namespace util {
    NNT_FS_INSTANTIATE_TEST_CASE_MOUNT(WithMountData, ::testing::ValuesIn(MountTestParameters));
}}}

extern "C" void nnMain()
{
    nnt::fs::util::LoadMountTest();

    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    nnt::fs::util::mock::InitializeForFsTest(&argc, argv);

    SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nn::fs::SetEnabledAutoAbort(false);
    nnt::fs::util::ResetAllocateCount();

    auto result = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(result);
}
