﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_ContentStorage.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>

#include "testFs_FsLib_Mount.h"

namespace
{
    void CreateSystemSaveDataWithFile(nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
    {
        const int FileSize = 32;

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(nn::fs::SaveDataSpaceId::SdSystem, saveDataId, nn::fs::InvalidUserId, 0, 48 * 1024, 48 * 1024, 0));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSystemSaveData("save", nn::fs::SaveDataSpaceId::SdSystem, saveDataId, nn::fs::InvalidUserId));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CreateFileWith32BitCount("save:/file", FileSize, 0));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData("save"));
        nn::fs::Unmount("save");
    }
}

TEST(MountSystemSaveData, SdCardEncryption)
{
    nn::fs::EncryptionSeed seed = { { 1 } };
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetSdCardEncryptionSeed(seed));

    nnt::fs::util::DeleteAllTestSaveData();

    nn::fs::SaveDataId saveId1 = 0x8000000000000003ULL;
    nn::fs::SaveDataId saveId2 = 0x8000000000000004ULL;
    CreateSystemSaveDataWithFile(saveId1);
    CreateSystemSaveDataWithFile(saveId2);

    // セーブバイナリを入れ替えて読めないこと
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("sd"));
#if defined(NN_BUILD_CONFIG_OS_WIN)
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::RenameDirectory("sd:/Nintendo/Save/8000000000000003", "sd:/Nintendo/Save/tmp"));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::RenameDirectory("sd:/Nintendo/Save/8000000000000004", "sd:/Nintendo/Save/8000000000000003"));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::RenameDirectory("sd:/Nintendo/Save/tmp", "sd:/Nintendo/Save/8000000000000004"));
#else
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::RenameFile("sd:/Nintendo/Save/8000000000000003", "sd:/Nintendo/Save/tmp"));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::RenameFile("sd:/Nintendo/Save/8000000000000004", "sd:/Nintendo/Save/8000000000000003"));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::RenameFile("sd:/Nintendo/Save/tmp", "sd:/Nintendo/Save/8000000000000004"));
#endif
    nn::fs::Unmount("sd");

    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultDataCorrupted, nn::fs::MountSystemSaveData("save", nn::fs::SaveDataSpaceId::SdSystem, saveId1, nn::fs::InvalidUserId));
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultDataCorrupted, nn::fs::MountSystemSaveData("save", nn::fs::SaveDataSpaceId::SdSystem, saveId2, nn::fs::InvalidUserId));

    nnt::fs::util::DeleteAllTestSaveData();
}

// 別 seed で別暗号化になること
TEST(MountSystemSaveData, SdCardEncryptionBySeed)
{
    nn::fs::EncryptionSeed seedA = { { 0xA } };
    nn::fs::EncryptionSeed seedB = { { 0xB } };
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetSdCardEncryptionSeed(seedA));

    nnt::fs::util::DeleteAllTestSaveData();

    nn::fs::SaveDataId saveId1 = 0x8000000000000003ULL;
    CreateSystemSaveDataWithFile(saveId1);

    // 別 seed
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetSdCardEncryptionSeed(seedB));

    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultDataCorrupted, MountSystemSaveData("save", nn::fs::SaveDataSpaceId::SdSystem, saveId1, nn::fs::InvalidUserId));

    nnt::fs::util::DeleteAllTestSaveData();

    // ゴミが残るため直接削除
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("sd"));
#if defined(NN_BUILD_CONFIG_OS_WIN)
    nn::fs::DeleteDirectoryRecursively("sd:/Nintendo/save/8000000000000003");
#else
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile("sd:/Nintendo/save/8000000000000003"));
#endif
    nn::fs::Unmount("sd");
}

// TODO: SetSdCardEncryptionSeed() の設定が残ってしまう対策
TEST(MountContentStorage, SdCardEncryption)
{
    nn::fs::EncryptionSeed seed = { { 1 } };
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetSdCardEncryptionSeed(seed));

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountContentStorage("content", nn::fs::ContentStorageId::SdCard));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("sd"));

    auto deleteFiles = []() {
        nn::fs::DeleteFile("content:/file0");
        nn::fs::DeleteFile("content:/file1");
        nn::fs::DeleteFile("content:/dir/file0");
        nn::fs::DeleteDirectory("content:/dir");
    };

    deleteFiles();

    const int SdEncryptionHeaderSize = 16 * 1024;
    const int FileSize = 8;

    const size_t BufferSize = 1024 * 1024;
    auto buffer = nnt::fs::util::AllocateBuffer(BufferSize);


    // ContentStorageId::SdCard = @SdCard:/Nintendo/contents に配置されたファイルは暗号化されていること
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CreateFileWith32BitCount("content:/file0", FileSize, 0));
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("content:/file0", 0));
        EXPECT_FALSE(nnt::fs::util::IsFilledWith32BitCount("sd:/Nintendo/Contents/file0", 0));
    }

    // 別パスで作成したファイルは別暗号化になっていること
    {
        nnt::fs::util::Hash hash[2];
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CreateFileWith32BitCount("content:/file1", FileSize, 0));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CalculateFileHash(&hash[0], "sd:/Nintendo/Contents/file0", buffer.get(), BufferSize, SdEncryptionHeaderSize, FileSize));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CalculateFileHash(&hash[1], "sd:/Nintendo/Contents/file1", buffer.get(), BufferSize, SdEncryptionHeaderSize, FileSize));
        EXPECT_TRUE(memcmp(&hash[0], &hash[1], sizeof(nnt::fs::util::Hash)) != 0);
    }

    // 別ディレクトリ同名ファイルも別暗号化になっていること
    {
        nnt::fs::util::Hash hash[2];
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateDirectory("content:/dir"));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CreateFileWith32BitCount("content:/dir/file0", FileSize, 0));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CalculateFileHash(&hash[0], "sd:/Nintendo/Contents/file0", buffer.get(), BufferSize, SdEncryptionHeaderSize, FileSize));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CalculateFileHash(&hash[1], "sd:/Nintendo/Contents/dir/file0", buffer.get(), BufferSize, SdEncryptionHeaderSize, FileSize));
        EXPECT_TRUE(memcmp(&hash[0], &hash[1], sizeof(nnt::fs::util::Hash)) != 0);
    }

    // teardown
    deleteFiles();
    nn::fs::Unmount("sd");
    nn::fs::Unmount("content");
}

TEST(MountContentStorage, SdCardEncryptionBySeed)
{
    nn::fs::EncryptionSeed seedA = { { 0xA } };
    nn::fs::EncryptionSeed seedB = { { 0xB } };
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetSdCardEncryptionSeed(seedA));

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountContentStorage("content", nn::fs::ContentStorageId::SdCard));

    auto deleteFiles = []() {
        nn::fs::DeleteFile("content:/file0");
    };

    deleteFiles();

    const int FileSize = 8;

    // seedA でファイルを作成
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CreateFileWith32BitCount("content:/file0", FileSize, 0));
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("content:/file0", 0));
    }

    nn::fs::Unmount("content");
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetSdCardEncryptionSeed(seedB));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountContentStorage("content", nn::fs::ContentStorageId::SdCard));

    // 別 seed でマウントした場合には暗号化が異なること
    {
        nn::fs::FileHandle handle;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultDataCorrupted, nn::fs::OpenFile(&handle, "content:/file0", nn::fs::OpenMode_Read));
    }

    // teardown
    deleteFiles();
    nn::fs::Unmount("content");
}

// seed の引継ぎで別本体でも開けること(手動テスト)
TEST(SdCardEncryption, PrepareMove)
{
    nn::fs::EncryptionSeed seedA = { { 0xA } };
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetSdCardEncryptionSeed(seedA));
    nnt::fs::util::DeleteAllTestSaveData();

    // ファイルを作成 @ ContentStorage
    {
        const int FileSize = 8;

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountContentStorage("content", nn::fs::ContentStorageId::SdCard));
        nn::fs::DeleteFile("content:/file0");
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CreateFileWith32BitCount("content:/file0", FileSize, 0));
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("content:/file0", 0));

        nn::fs::Unmount("content");
    }

    // セーブを作成
    {
        nnt::fs::util::DeleteAllTestSaveData();
        nn::fs::SaveDataId saveId1 = 0x8000000000000003ULL;
        CreateSystemSaveDataWithFile(saveId1);
    }
}

// @pre 別本体で SdCardEncryption.PrepareMove を実行した SD カードを挿入
TEST(SdCardEncryption, VerifyMove)
{
    nn::fs::EncryptionSeed seedA = { { 0xA } };
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetSdCardEncryptionSeed(seedA));

    // ContentStorage の確認
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountContentStorage("content", nn::fs::ContentStorageId::SdCard));
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("content:/file0", 0));
        nn::fs::DeleteFile("content:/file0");
        nn::fs::Unmount("content");
    }

    // セーブの確認
    {
        nn::fs::SaveDataId saveId1 = 0x8000000000000003ULL;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSystemSaveData("save", nn::fs::SaveDataSpaceId::SdSystem, saveId1, nn::fs::InvalidUserId));
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount("save:/file", 0));
        nn::fs::Unmount("save");

        nnt::fs::util::DeleteAllTestSaveData();
    }
}
