﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_SaveDataForDebug.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_ImageDirectory.h>

#include "testFs_FsLib_Mount.h"

TEST(MountVariousFsTest, PathCharacterLimitation)
{
    // TODO: FsApi テストのマウントが充実したらそちらに移動する

    const auto Test = []() NN_NOEXCEPT
    {
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("test");
        };

        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidCharacter,
            nn::fs::CreateFile("test:/foo<bar", 0));
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidCharacter,
            nn::fs::CreateFile("test:/foo>bar", 0));
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidCharacter,
            nn::fs::CreateFile("test:/foo:bar", 0));
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidCharacter,
            nn::fs::CreateFile("test:/foo*bar", 0));
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidCharacter,
            nn::fs::CreateFile("test:/foo?bar", 0));
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidCharacter,
            nn::fs::CreateFile("test:/foo|bar", 0));
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug("test"));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost("test", g_TestDirPath.GetPath().c_str()));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis("test", nn::fs::BisPartitionId::User));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::Nand));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::SdCard));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("test"));
    Test();
}

TEST(MountVariousFsTest, PathLengthLimitation)
{
    // TODO: FsApi テストのマウントが充実したらそちらに移動する

    const auto TestDifferentSizes = [](
        size_t filePathLengthMax,
        size_t directoryPathLengthMax,
        size_t createDirectoryPathLengthMax,
        size_t openDirectoryPathLengthMax,
        size_t deleteDirectoryPathLengthMax) NN_NOEXCEPT
    {
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("test");
        };

        nn::Result result;

        result = nn::fs::CreateDirectory("test:/PathLengthLimitation");
        if( !nn::fs::ResultPathAlreadyExists::Includes(result) )
        {
            NNT_ASSERT_RESULT_SUCCESS(result);
        }
        NN_UTIL_SCOPE_EXIT
        {
            (void)nn::fs::DeleteDirectoryRecursively("test:/PathLengthLimitation");
        };

        static char s_FilePath[nn::fs::EntryNameLengthMax + 1] = "test:/PathLengthLimitation/";
        static char s_DirectoryPath[nn::fs::EntryNameLengthMax + 1] = "test:/PathLengthLimitation/";
        static char s_CreateDirectoryPath[nn::fs::EntryNameLengthMax + 1] = "test:/PathLengthLimitation/";
        static char s_OpenDirectoryPath[nn::fs::EntryNameLengthMax + 1] = "test:/PathLengthLimitation/";
        static char s_DeleteDirectoryPath[nn::fs::EntryNameLengthMax + 1] = "test:/PathLengthLimitation/";

        const auto rootDirectoryPathLength = strlen("test:/PathLengthLimitation/");

        std::fill(
            s_FilePath + rootDirectoryPathLength,
            s_FilePath + rootDirectoryPathLength + filePathLengthMax + 1,
            'a');
        std::fill(
            s_DirectoryPath + rootDirectoryPathLength,
            s_DirectoryPath + rootDirectoryPathLength + directoryPathLengthMax + 1,
            'b');
        std::fill(
            s_CreateDirectoryPath + rootDirectoryPathLength,
            s_CreateDirectoryPath + rootDirectoryPathLength + createDirectoryPathLengthMax + 1,
            'c');
        std::fill(
            s_OpenDirectoryPath + rootDirectoryPathLength,
            s_OpenDirectoryPath + rootDirectoryPathLength + openDirectoryPathLengthMax + 1,
            'd');
        std::fill(
            s_DeleteDirectoryPath + rootDirectoryPathLength,
            s_DeleteDirectoryPath + rootDirectoryPathLength + deleteDirectoryPathLengthMax + 1,
            'e');
        s_FilePath[rootDirectoryPathLength + filePathLengthMax + 1] = '\0';
        s_DirectoryPath[rootDirectoryPathLength + directoryPathLengthMax + 1] = '\0';
        s_CreateDirectoryPath[rootDirectoryPathLength + createDirectoryPathLengthMax + 1] = '\0';
        s_OpenDirectoryPath[rootDirectoryPathLength + openDirectoryPathLengthMax + 1] = '\0';
        s_DeleteDirectoryPath[rootDirectoryPathLength + deleteDirectoryPathLengthMax + 1] = '\0';

        nn::fs::FileHandle file;
        nn::fs::DirectoryHandle directory;

        {
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultTooLongPath,
                nn::fs::CreateFile(s_FilePath, 1));
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultTooLongPath,
                result = nn::fs::OpenFile(&file, s_FilePath, nn::fs::OpenMode_Read));
            if( result.IsSuccess() )
            {
                nn::fs::CloseFile(file);
            }
            nn::fs::DirectoryEntryType entryType;
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultTooLongPath,
                nn::fs::GetEntryType(&entryType, s_FilePath));
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultTooLongPath,
                nn::fs::DeleteFile(s_FilePath));
        }

        {
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultTooLongPath,
                nn::fs::CreateDirectory(s_CreateDirectoryPath));
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultTooLongPath,
                result = nn::fs::OpenDirectory(
                    &directory,
                    s_OpenDirectoryPath,
                    nn::fs::OpenDirectoryMode_All));
            if( result.IsSuccess() )
            {
                nn::fs::CloseDirectory(directory);
            }
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultTooLongPath,
                nn::fs::DeleteDirectory(s_DeleteDirectoryPath));
        }

        s_FilePath[rootDirectoryPathLength + filePathLengthMax] = '\0';
        s_DirectoryPath[rootDirectoryPathLength + directoryPathLengthMax] = '\0';
        s_CreateDirectoryPath[rootDirectoryPathLength + createDirectoryPathLengthMax] = '\0';
        s_OpenDirectoryPath[rootDirectoryPathLength + openDirectoryPathLengthMax] = '\0';
        s_DeleteDirectoryPath[rootDirectoryPathLength + deleteDirectoryPathLengthMax] = '\0';

        {
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile(s_FilePath, 1));
            NNT_EXPECT_RESULT_SUCCESS(
                result = nn::fs::OpenFile(&file, s_FilePath, nn::fs::OpenMode_Read));
            if( result.IsSuccess() )
            {
                nn::fs::CloseFile(file);
            }
            nn::fs::DirectoryEntryType entryType;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetEntryType(&entryType, s_FilePath));
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile(s_FilePath));
        }

        {
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateDirectory(s_CreateDirectoryPath));
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::RenameDirectory(
                s_CreateDirectoryPath,
                s_OpenDirectoryPath));
            NNT_EXPECT_RESULT_SUCCESS(
                result = nn::fs::OpenDirectory(
                    &directory,
                    s_OpenDirectoryPath,
                    nn::fs::OpenDirectoryMode_All));
            if( result.IsSuccess() )
            {
                nn::fs::CloseDirectory(directory);
            }
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::RenameDirectory(
                s_OpenDirectoryPath,
                s_DirectoryPath));
            // 実機版の HOST FS では directoryPathLengthMax < deleteDirectoryPathLengthMax であり、
            // s_DeleteDirectoryPath へのリネームができないため、s_DirectoryPath の削除成功をテストする
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteDirectory(s_DirectoryPath));
        }

        s_FilePath[rootDirectoryPathLength] = '.';
        s_FilePath[rootDirectoryPathLength + 1] = '/';
        s_FilePath[rootDirectoryPathLength + filePathLengthMax] = 'A';
        s_FilePath[rootDirectoryPathLength + filePathLengthMax + 1] = 'A';
        s_FilePath[rootDirectoryPathLength + filePathLengthMax + 2] = 'A';
        s_FilePath[rootDirectoryPathLength + filePathLengthMax + 3] = '\0';
        s_DirectoryPath[rootDirectoryPathLength] = '.';
        s_DirectoryPath[rootDirectoryPathLength + 1] = '/';
        s_DirectoryPath[rootDirectoryPathLength + directoryPathLengthMax] = 'B';
        s_DirectoryPath[rootDirectoryPathLength + directoryPathLengthMax + 1] = 'B';
        s_DirectoryPath[rootDirectoryPathLength + directoryPathLengthMax + 2] = 'B';
        s_DirectoryPath[rootDirectoryPathLength + directoryPathLengthMax + 3] = '\0';
        s_CreateDirectoryPath[rootDirectoryPathLength] = '.';
        s_CreateDirectoryPath[rootDirectoryPathLength + 1] = '/';
        s_CreateDirectoryPath[rootDirectoryPathLength + createDirectoryPathLengthMax] = 'C';
        s_CreateDirectoryPath[rootDirectoryPathLength + createDirectoryPathLengthMax + 1] = 'C';
        s_CreateDirectoryPath[rootDirectoryPathLength + createDirectoryPathLengthMax + 2] = 'C';
        s_CreateDirectoryPath[rootDirectoryPathLength + createDirectoryPathLengthMax + 3] = '\0';
        s_OpenDirectoryPath[rootDirectoryPathLength] = '.';
        s_OpenDirectoryPath[rootDirectoryPathLength + 1] = '/';
        s_OpenDirectoryPath[rootDirectoryPathLength + openDirectoryPathLengthMax] = 'D';
        s_OpenDirectoryPath[rootDirectoryPathLength + openDirectoryPathLengthMax + 1] = 'D';
        s_OpenDirectoryPath[rootDirectoryPathLength + openDirectoryPathLengthMax + 2] = 'D';
        s_OpenDirectoryPath[rootDirectoryPathLength + openDirectoryPathLengthMax + 3] = '\0';
        s_DeleteDirectoryPath[rootDirectoryPathLength] = '.';
        s_DeleteDirectoryPath[rootDirectoryPathLength + 1] = '/';
        s_DeleteDirectoryPath[rootDirectoryPathLength + deleteDirectoryPathLengthMax] = 'E';
        s_DeleteDirectoryPath[rootDirectoryPathLength + deleteDirectoryPathLengthMax + 1] = 'E';
        s_DeleteDirectoryPath[rootDirectoryPathLength + deleteDirectoryPathLengthMax + 2] = 'E';
        s_DeleteDirectoryPath[rootDirectoryPathLength + deleteDirectoryPathLengthMax + 3] = '\0';

        const auto currentDirectoryPathLength = strlen("./");

        {
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultTooLongPath,
                nn::fs::CreateFile(s_FilePath, 1));
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultTooLongPath,
                result = nn::fs::OpenFile(&file, s_FilePath, nn::fs::OpenMode_Read));
            if( result.IsSuccess() )
            {
                nn::fs::CloseFile(file);
            }
            nn::fs::DirectoryEntryType entryType;
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultTooLongPath,
                nn::fs::GetEntryType(&entryType, s_FilePath));
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultTooLongPath,
                nn::fs::DeleteFile(s_FilePath));
        }

        {
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultTooLongPath,
                nn::fs::CreateDirectory(s_CreateDirectoryPath));
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultTooLongPath,
                result = nn::fs::OpenDirectory(
                    &directory,
                    s_OpenDirectoryPath,
                    nn::fs::OpenDirectoryMode_All));
            if( result.IsSuccess() )
            {
                nn::fs::CloseDirectory(directory);
            }
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultTooLongPath,
                nn::fs::DeleteDirectory(s_DeleteDirectoryPath));
        }

        const auto parentDirectoryPathLength
            = rootDirectoryPathLength + currentDirectoryPathLength;
        s_FilePath[parentDirectoryPathLength + filePathLengthMax] = '\0';
        s_DirectoryPath[parentDirectoryPathLength + directoryPathLengthMax] = '\0';
        s_CreateDirectoryPath[parentDirectoryPathLength + createDirectoryPathLengthMax] = '\0';
        s_OpenDirectoryPath[parentDirectoryPathLength + openDirectoryPathLengthMax] = '\0';
        s_DeleteDirectoryPath[parentDirectoryPathLength + deleteDirectoryPathLengthMax] = '\0';

        {
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile(s_FilePath, 1));
            NNT_EXPECT_RESULT_SUCCESS(
                result = nn::fs::OpenFile(&file, s_FilePath, nn::fs::OpenMode_Read));
            if( result.IsSuccess() )
            {
                nn::fs::CloseFile(file);
            }
            nn::fs::DirectoryEntryType entryType;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetEntryType(&entryType, s_FilePath));
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile(s_FilePath));
        }

        {
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateDirectory(s_CreateDirectoryPath));
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::RenameDirectory(s_CreateDirectoryPath, s_OpenDirectoryPath));
            NNT_EXPECT_RESULT_SUCCESS(
                result = nn::fs::OpenDirectory(&directory, s_OpenDirectoryPath, nn::fs::OpenDirectoryMode_All));
            if( result.IsSuccess() )
            {
                nn::fs::CloseDirectory(directory);
            }
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::RenameDirectory(s_OpenDirectoryPath, s_DirectoryPath));
            // 実機版の HOST FS では directoryPathLengthMax < deleteDirectoryPathLengthMax であり、
            // s_DeleteDirectoryPath へのリネームができないため、s_DirectoryPath の削除成功をテストする
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteDirectory(s_DirectoryPath));
        }
    };

    const auto Test = [&TestDifferentSizes](size_t lengthMax) NN_NOEXCEPT
    {
        TestDifferentSizes(lengthMax, lengthMax, lengthMax, lengthMax, lengthMax);
    };

#ifdef NN_BUILD_CONFIG_OS_WIN
    static const auto EntryLengthMax = 260 - 1;
    static const auto CreateDirectoryLengthMax = 248 - 1;
    static const auto OpenDirectoryLengthMax = 260 - 2 - 1;

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost("test", g_TestDirPath.GetPath().c_str()));
    TestDifferentSizes(
        EntryLengthMax - strlen(g_TestDirPath.GetPath().c_str()) - 1 - strlen("PathLengthLimitation/"),
        EntryLengthMax - strlen(g_TestDirPath.GetPath().c_str()) - 1 - strlen("PathLengthLimitation/"),
        CreateDirectoryLengthMax - strlen(g_TestDirPath.GetPath().c_str()) - 1 - strlen("PathLengthLimitation/"),
        OpenDirectoryLengthMax - strlen(g_TestDirPath.GetPath().c_str()) - 1 - strlen("PathLengthLimitation/"),
        EntryLengthMax - strlen(g_TestDirPath.GetPath().c_str()) - 1 - strlen("PathLengthLimitation/"));
#else
    static const auto SaveDataLengthMax = 64;
    static const auto HostFileLengthMax = 260 - 1;
    static const auto HostDirectoryLengthMax = 248 - 1;
    static const auto FatLengthMax = 260 - 1;

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug("test"));
    Test(SaveDataLengthMax);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost("test", g_TestDirPath.GetPath().c_str()));
    TestDifferentSizes(
        HostFileLengthMax - strlen(g_TestDirPath.GetPath().c_str()) - 1 - strlen("PathLengthLimitation/"),
        HostDirectoryLengthMax - strlen(g_TestDirPath.GetPath().c_str()) - 1 - strlen("PathLengthLimitation/"),
        HostDirectoryLengthMax - strlen(g_TestDirPath.GetPath().c_str()) - 1 - strlen("PathLengthLimitation/"),
        HostDirectoryLengthMax - strlen(g_TestDirPath.GetPath().c_str()) - 1 - strlen("PathLengthLimitation/"),
        HostFileLengthMax - strlen(g_TestDirPath.GetPath().c_str()) - 1 - strlen("PathLengthLimitation/"));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis("test", nn::fs::BisPartitionId::User));
    Test(FatLengthMax - strlen("X:/") - strlen("PathLengthLimitation/"));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::Nand));
    Test(FatLengthMax - strlen("X:/Album/") - strlen("PathLengthLimitation/"));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::SdCard));
    Test(FatLengthMax - strlen("X:/Nintendo/Album/") - strlen("PathLengthLimitation/"));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("test"));
    Test(FatLengthMax - strlen("X:/") - strlen("PathLengthLimitation/"));
#endif
} // NOLINT(impl/function_size)

TEST(MountVariousFsTest, PathLengthLimitationWideCharacter)
{
    // TODO: FsApi テストのマウントが充実したらそちらに移動する

    const auto Test = []() NN_NOEXCEPT
    {
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("test");
        };

        const char WideCharacter[] = {
            static_cast<char>(-16),
            static_cast<char>(-96),
            static_cast<char>(-128),
            static_cast<char>(-117),
            '\0'
        }; // (4 バイト文字)

        nnt::fs::util::String path = "test:/";

        while( path.size() <= nn::fs::EntryNameLengthMax )
        {
            path.append(WideCharacter);
        }

        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTooLongPath, nn::fs::CreateFile(path.c_str(), 0));
        (void)nn::fs::DeleteFile(path.c_str());
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug("test"));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost("test", g_TestDirPath.GetPath().c_str()));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis("test", nn::fs::BisPartitionId::User));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::Nand));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::SdCard));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("test"));
    Test();
}

TEST(MountVariousFsTest, CreateOrRenameToOpenedExistentEntry)
{
    // TODO: FsApi テストのマウントが充実したらそちらに移動する

    const auto Test = []() NN_NOEXCEPT
    {
        nn::fs::FileHandle file;
        nn::fs::DirectoryHandle directory;
        (void)nn::fs::CreateFile("test:/openedfile", 0);
        (void)nn::fs::CreateFile("test:/closedfile", 0);
        (void)nn::fs::CreateDirectory("test:/openeddir");
        (void)nn::fs::CreateDirectory("test:/closeddir");
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, "test:/openedfile", nn::fs::OpenMode_Read));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(&directory, "test:/openeddir", nn::fs::OpenDirectoryMode_All));

        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, nn::fs::CreateFile("test:/openedfile", 0));
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, nn::fs::CreateFile("test:/openeddir", 0));
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, nn::fs::CreateDirectory("test:/openedfile"));
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, nn::fs::CreateDirectory("test:/openeddir"));
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, nn::fs::RenameFile("test:/closedfile", "test:/openedfile"));
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, nn::fs::RenameFile("test:/closedfile", "test:/openeddir"));
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, nn::fs::RenameFile("test:/openedfile", "test:/closedfile"));
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, nn::fs::RenameFile("test:/openedfile", "test:/closeddir"));
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, nn::fs::RenameDirectory("test:/closeddir", "test:/openedfile"));
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, nn::fs::RenameDirectory("test:/closeddir", "test:/openeddir"));
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, nn::fs::RenameDirectory("test:/openeddir", "test:/closedfile"));
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, nn::fs::RenameDirectory("test:/openeddir", "test:/closeddir"));

        nn::fs::CloseFile(file);
        nn::fs::CloseDirectory(directory);
        nn::fs::Unmount("test");
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug("test"));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost("test", g_TestDirPath.GetPath().c_str()));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis("test", nn::fs::BisPartitionId::User));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::Nand));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::SdCard));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("test"));
    Test();
}

TEST(MountVariousFsTest, DeleteRoot)
{
    // TODO: FsApi テストのマウントが充実したらそちらに移動する

    const auto Test = []() NN_NOEXCEPT
    {
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("test");
        };

        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDirectoryUndeletable, nn::fs::DeleteDirectory("test:/"));
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDirectoryUndeletable, nn::fs::DeleteDirectory("test:/./"));
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug("test"));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost("test", g_TestDirPath.GetPath().c_str()));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis("test", nn::fs::BisPartitionId::User));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::Nand));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::SdCard));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("test"));
    Test();
}

TEST(MountVariousFsTest, DeleteDirectoryWithOpenedChildren)
{
    // TODO: FsApi テストのマウントが充実したらそちらに移動する

    const auto Test = []() NN_NOEXCEPT
    {
        nn::fs::FileHandle file;
        (void)nn::fs::CreateDirectory("test:/parentdir");
        (void)nn::fs::CreateFile("test:/parentdir/childfile", 0);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, "test:/parentdir/childfile", nn::fs::OpenMode_Read));

        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDirectoryNotEmpty, nn::fs::DeleteDirectory("test:/parentdir"));

        nn::fs::CloseFile(file);
        nn::fs::Unmount("test");
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug("test"));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost("test", g_TestDirPath.GetPath().c_str()));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis("test", nn::fs::BisPartitionId::User));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::Nand));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::SdCard));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("test"));
    Test();
}

TEST(MountVariousFsTest, FilioparentalRename)
{
    // TODO: FsApi テストのマウントが充実したらそちらに移動する

    const auto Test = []() NN_NOEXCEPT
    {
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("test");
        };

        (void)nn::fs::CreateDirectory("test:/parentdir");
        (void)nn::fs::CreateDirectory("test:/parentdir/childdir");
        (void)nn::fs::CreateFile("test:/parentdir/childfile", 0);

        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultDirectoryUnrenamable,
            nn::fs::RenameDirectory("test:/parentdir", "test:/parentdir/nonexistent"));
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultDirectoryUnrenamable,
            nn::fs::RenameDirectory("test:/parentdir", "test:/parentdir/childdir"));
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultDirectoryUnrenamable,
            nn::fs::RenameDirectory("test:/parentdir", "test:/parentdir/childfile"));
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultDirectoryUnrenamable,
            nn::fs::RenameDirectory("test:/parentdir/nonexistent", "test:/parentdir"));
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultDirectoryUnrenamable,
            nn::fs::RenameDirectory("test:/parentdir/childdir", "test:/parentdir"));
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultDirectoryUnrenamable,
            nn::fs::RenameDirectory("test:/parentdir/childfile", "test:/parentdir"));

        nn::fs::FileHandle file;
        nn::fs::DirectoryHandle directory;

        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(
                &directory,
                "test:/parentdir",
                nn::fs::OpenDirectoryMode_All));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseDirectory(directory);
            };

            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultDirectoryUnrenamable,
                nn::fs::RenameDirectory("test:/parentdir", "test:/parentdir/nonexistent"));
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultDirectoryUnrenamable,
                nn::fs::RenameDirectory("test:/parentdir", "test:/parentdir/childdir"));
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultDirectoryUnrenamable,
                nn::fs::RenameDirectory("test:/parentdir", "test:/parentdir/childfile"));
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultDirectoryUnrenamable,
                nn::fs::RenameDirectory("test:/parentdir/nonexistent", "test:/parentdir"));
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultDirectoryUnrenamable,
                nn::fs::RenameDirectory("test:/parentdir/childdir", "test:/parentdir"));
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultDirectoryUnrenamable,
                nn::fs::RenameDirectory("test:/parentdir/childfile", "test:/parentdir"));
        }

        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(
                &directory,
                "test:/parentdir/childdir",
                nn::fs::OpenDirectoryMode_All));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseDirectory(directory);
            };

            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultDirectoryUnrenamable,
                nn::fs::RenameDirectory("test:/parentdir", "test:/parentdir/childdir"));
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultDirectoryUnrenamable,
                nn::fs::RenameDirectory("test:/parentdir/childdir", "test:/parentdir"));
        }

        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(
                &file,
                "test:/parentdir/childfile",
                nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(file);
            };

            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultDirectoryUnrenamable,
                nn::fs::RenameDirectory("test:/parentdir", "test:/parentdir/childfile"));
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultDirectoryUnrenamable,
                nn::fs::RenameDirectory("test:/parentdir/childfile", "test:/parentdir"));
        }
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug("test"));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost("test", g_TestDirPath.GetPath().c_str()));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis("test", nn::fs::BisPartitionId::User));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::Nand));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::SdCard));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("test"));
    Test();
} // NOLINT(impl/function_size)

TEST(MountVariousFsTest, CreateRoot)
{
    // TODO: FsApi テストのマウントが充実したらそちらに移動する

    const auto Test = []() NN_NOEXCEPT
    {
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("test");
        };

        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, nn::fs::CreateDirectory("test:/"));
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, nn::fs::CreateDirectory("test:/./"));
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug("test"));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost("test", g_TestDirPath.GetPath().c_str()));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis("test", nn::fs::BisPartitionId::User));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::Nand));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::SdCard));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("test"));
    Test();
}

TEST(MountVariousFsTest, RootParent)
{
    // TODO: FsApi テストのマウントが充実したらそちらに移動する

    const auto Test = []() NN_NOEXCEPT
    {
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("test");
        };

        (void)nn::fs::CreateDirectory("test:/dir");

        nn::fs::FileHandle file;
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultDirectoryUnobtainable,
            nn::fs::OpenFile(&file, "test:/../foo", nn::fs::OpenMode_Read));
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultDirectoryUnobtainable,
            nn::fs::OpenFile(&file, "test:/dir/../../bar", nn::fs::OpenMode_Read));
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug("test"));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost("test", g_TestDirPath.GetPath().c_str()));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis("test", nn::fs::BisPartitionId::User));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::Nand));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::SdCard));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("test"));
    Test();
}

TEST(MountVariousFsTest, DoubleSeparator)
{
    // TODO: FsApi テストのマウントが充実したらそちらに移動する

    const auto Test = []() NN_NOEXCEPT
    {
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("test");
        };

        nn::fs::FileHandle file;
        const auto result = nn::fs::OpenFile(&file, "test://foo", nn::fs::OpenMode_Read);
        if( !nn::fs::ResultPathNotFound::Includes(result) )
        {
            NNT_ASSERT_RESULT_SUCCESS(result);
            nn::fs::CloseFile(file);
        }
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug("test"));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost("test", g_TestDirPath.GetPath().c_str()));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis("test", nn::fs::BisPartitionId::User));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::Nand));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::SdCard));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("test"));
    Test();
}

TEST(MountVariousFsTest, SelfRename)
{
    // TODO: FsApi テストのマウントが充実したらそちらに移動する

    const auto Test = []() NN_NOEXCEPT
    {
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("test");
        };

        (void)nn::fs::CreateFile("test:/file", 0);
        (void)nn::fs::CreateDirectory("test:/dir");

        NNT_EXPECT_RESULT_SUCCESS(nn::fs::RenameFile("test:/file", "test:/file"));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::RenameDirectory("test:/dir", "test:/dir"));

        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultPathNotFound,
            nn::fs::RenameFile("test:/dir", "test:/dir"));
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultPathNotFound,
            nn::fs::RenameFile("test:/nonexistent", "test:/nonexistent"));
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultPathNotFound,
            nn::fs::RenameDirectory("test:/file", "test:/file"));
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultPathNotFound,
            nn::fs::RenameDirectory("test:/nonexistent", "test:/nonexistent"));

        nn::fs::FileHandle file;
        nn::fs::DirectoryHandle directory;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::OpenFile(&file, "test:/file", nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(file);
        };
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::OpenDirectory(&directory, "test:/dir", nn::fs::OpenDirectoryMode_All));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseDirectory(directory);
        };

        NNT_EXPECT_RESULT_SUCCESS(nn::fs::RenameFile("test:/file", "test:/file"));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::RenameDirectory("test:/dir", "test:/dir"));

        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultPathNotFound,
            nn::fs::RenameFile("test:/dir", "test:/dir"));
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultPathNotFound,
            nn::fs::RenameFile("test:/nonexistent", "test:/nonexistent"));
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultPathNotFound,
            nn::fs::RenameDirectory("test:/file", "test:/file"));
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultPathNotFound,
            nn::fs::RenameDirectory("test:/nonexistent", "test:/nonexistent"));
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug("test"));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost("test", g_TestDirPath.GetPath().c_str()));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis("test", nn::fs::BisPartitionId::User));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::Nand));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::SdCard));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("test"));
    Test();
}

TEST(MountVariousFsTest, BackslashIncludingPath)
{
    // TODO: FsApi テストのマウントが充実したらそちらに移動する

    const auto Test = []() NN_NOEXCEPT
    {
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("test");
        };

        {
            const auto result = nn::fs::CreateDirectory("test:\\dir\\");
            if( !nn::fs::ResultPathAlreadyExists::Includes(result) )
            {
                NNT_ASSERT_RESULT_SUCCESS(result);
            }
        }

        NN_UTIL_SCOPE_EXIT
        {
            (void)nn::fs::DeleteDirectoryRecursively("test:/dir");
        };

        {
            const auto result = nn::fs::CreateFile("test:\\dir\\file", 0);
            if( !nn::fs::ResultPathAlreadyExists::Includes(result) )
            {
                NNT_ASSERT_RESULT_SUCCESS(result);
            }
        }

        {
            nn::fs::FileHandle file;
            const auto result = nn::fs::OpenFile(&file, "test:\\dir\\file", nn::fs::OpenMode_Read);
            NNT_EXPECT_RESULT_SUCCESS(result);
            if( result.IsSuccess() )
            {
                nn::fs::CloseFile(file);
            }
        }

        NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile("test:\\dir\\file"));

        nn::fs::DirectoryEntryType entryType;
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultPathNotFound,
            nn::fs::GetEntryType(&entryType, "test:\\dir\\file"));
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug("test"));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost("test", g_TestDirPath.GetPath().c_str()));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis("test", nn::fs::BisPartitionId::User));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::Nand));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory("test", nn::fs::ImageDirectoryId::SdCard));
    Test();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("test"));
    Test();
}

#ifdef NN_BUILD_CONFIG_HARDWARE_NX
namespace
{
class AfterExpandFileErrorTest : public ::testing::Test
{
public:
    AfterExpandFileErrorTest():
    m_TestFileSize(0),
    m_IsSetupFinished(false),
    m_WriteBuffer(new char[MaxFileSize + 1]),
    m_ReadBuffer(new char[MaxFileSize])
{
    if( m_WriteBuffer != nullptr )
    {
        std::iota(m_WriteBuffer.get(), m_WriteBuffer.get() + MaxFileSize + 1, '\x0');
    }
}

protected:
    static const char MountName[];

protected:
    // 空き容量を減らします。
    // parentDirectoryPath 内に、サイズが MaxFileSize であるファイルを限界まで作成する
    void FillFreeSpace() NN_NOEXCEPT
    {
        const size_t MaxFilePathLength = 50;
        size_t directoryPathLength = std::strlen(DirectoryPath);
        EXPECT_GT(MaxFilePathLength, directoryPathLength);
        char dummyFilePath[MaxFilePathLength] = {};
        nn::util::SNPrintf(dummyFilePath, MaxFilePathLength, DirectoryPath);
        if( dummyFilePath[directoryPathLength - 1] != '/' )
        {
            dummyFilePath[directoryPathLength] = '/';
            ++directoryPathLength;
        }
        const char* DummyFileName = "dummy%03d";
        const size_t DummyFileNameLength = std::strlen(DummyFileName);
        EXPECT_GT(MaxFilePathLength, directoryPathLength + DummyFileNameLength);

        (void)nn::fs::DeleteDirectoryRecursively(DirectoryPath);
        NNT_EXPECT_RESULT_SUCCESS(
            nn::fs::CreateDirectory(DirectoryPath)
        );
        for( int fileId = 0; fileId < 1000; ++fileId )
        {
            nn::util::SNPrintf(dummyFilePath + directoryPathLength, DummyFileNameLength, DummyFileName, fileId);
            nn::Result result = nn::fs::CreateFile(dummyFilePath, MaxFileSize);
            if( result.IsFailure() )
            {
                NNT_EXPECT_RESULT_FAILURE(
                    nn::fs::ResultUsableSpaceNotEnough,
                    result
                );
                // ダミーファイルを作った後の空き容量のうちの半分のサイズのファイルを作成する
                int64_t fileSize = 0;
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetFreeSpaceSize(&fileSize, TestFilePath));
                m_TestFileSize = static_cast<size_t>(fileSize / 2);

                m_IsSetupFinished = true;
                return;
            }
        }

        // ダミーファイルが 1000 個以上作成できるほど空き容量がある場合は想定していません
        m_IsSetupFinished = false;
    }

// ダミーファイルを作成し、空き領域を減らします。
    void DeleteAllDummyFiles() const NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteDirectoryRecursively(DirectoryPath));
    }

    bool IsSetupFinished() const NN_NOEXCEPT
    {
        return m_IsSetupFinished && (m_WriteBuffer != nullptr) && (m_ReadBuffer != nullptr);
    }

    void DoTest() NN_NOEXCEPT
    {
        // WriteFile で空き領域以上にファイルを拡張します
        // 書き込みと同時に FlushFile を行います
        {
            // 空き容量のうち半分のサイズでファイルを作成する
            CreateTestFile();
            NN_UTIL_SCOPE_EXIT
            {
                DeleteTestFile();
            };

            // 作成したファイルを空き容量以上まで拡張する書き込みを行う
            {
                nn::fs::FileHandle fileHandle;
                NNT_EXPECT_RESULT_SUCCESS(
                    nn::fs::OpenFile(
                        &fileHandle,
                        TestFilePath,
                        static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend)
                    )
                );
                NN_UTIL_SCOPE_EXIT
                {
                    nn::fs::CloseFile(fileHandle);
                };
                NNT_EXPECT_RESULT_FAILURE(
                    nn::fs::ResultUsableSpaceNotEnough,
                    nn::fs::WriteFile(
                        fileHandle,
                        0,
                        m_WriteBuffer.get() + 1,
                        MaxFileSize,
                        nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)
                    )
                );
                CheckFileAfterWriteNoClose(fileHandle);
            }
            CheckFileAfterWriteReOpen();
        }

        // WriteFile で空き領域以上にファイルを拡張します
        // 書き込みと同時に FlushFile を行いません
        {
            // 空き容量のうち半分のサイズでファイルを作成する
            CreateTestFile();
            NN_UTIL_SCOPE_EXIT
            {
                DeleteTestFile();
            };

            // 作成したファイルを空き容量以上まで拡張する書き込みを行う
            {
                nn::fs::FileHandle fileHandle;
                NNT_EXPECT_RESULT_SUCCESS(
                    nn::fs::OpenFile(
                        &fileHandle,
                        TestFilePath,
                        static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend)
                    )
                );
                NN_UTIL_SCOPE_EXIT
                {
                    nn::fs::CloseFile(fileHandle);
                };
                NNT_EXPECT_RESULT_FAILURE(
                    nn::fs::ResultUsableSpaceNotEnough,
                    nn::fs::WriteFile(
                        fileHandle,
                        0,
                        m_WriteBuffer.get() + 1,
                        MaxFileSize,
                        nn::fs::WriteOption::MakeValue(0)
                    )
                );
                CheckFileAfterWriteNoClose(fileHandle);
            }
            CheckFileAfterWriteReOpen();
        }

        // SetFileSize で空き領域以上にファイルを拡張します
        {
            // 空き容量のうち半分のサイズでファイルを作成する
            CreateTestFile();
            NN_UTIL_SCOPE_EXIT
            {
                DeleteTestFile();
            };

            // 作成したファイルを空き容量以上まで拡張する書き込みを行う
            {
                nn::fs::FileHandle fileHandle;
                NNT_EXPECT_RESULT_SUCCESS(
                    nn::fs::OpenFile(
                        &fileHandle,
                        TestFilePath,
                        static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend)
                    )
                );
                NN_UTIL_SCOPE_EXIT
                {
                    nn::fs::CloseFile(fileHandle);
                };
                NNT_EXPECT_RESULT_FAILURE(
                    nn::fs::ResultUsableSpaceNotEnough,
                    nn::fs::SetFileSize(
                        fileHandle,
                        MaxFileSize
                    )
                );
                CheckFileAfterSetSizeNoClose(fileHandle);
            }
            CheckFileAfterSetSizeReOpen();
        }
    } // NOLINT(impl/function_size)

private:
    static const char DirectoryPath[];
    static const char TestFilePath[];
    static const size_t MaxFileSize;

private:
    void CreateTestFile() NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile(TestFilePath, m_TestFileSize));
        nn::fs::FileHandle fileHandle;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, TestFilePath, nn::fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(fileHandle);
        };

        NNT_EXPECT_RESULT_SUCCESS(
            nn::fs::WriteFile(
                fileHandle,
                0,
                m_WriteBuffer.get(),
                m_TestFileSize,
                nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)
            )
        );
    }

    void DeleteTestFile() NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile(TestFilePath));
    }

    void CheckFileAfterWriteNoClose(const nn::fs::FileHandle& fileHandle) NN_NOEXCEPT
    {
        size_t readSize = 0;

        int64_t fileSize = 0;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, fileHandle));
        size_t newFileSize = static_cast<size_t>(fileSize);
        if( m_TestFileSize == newFileSize )
        {
            // ファイルサイズが変化していない場合は、内容が変更されていないことを確認する
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(&readSize, fileHandle, 0, m_ReadBuffer.get(), m_TestFileSize));
            EXPECT_EQ(0, memcmp(m_ReadBuffer.get(), m_WriteBuffer.get(), m_TestFileSize));
            EXPECT_EQ(m_TestFileSize, readSize);
        }
        else
        {
            // ファイルサイズが変化している場合は、内容が拡張時の書き込みのものになっていることを確認する
            EXPECT_LT(m_TestFileSize, newFileSize);
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(&readSize, fileHandle, 0, m_ReadBuffer.get(), newFileSize));
            EXPECT_EQ(0, memcmp(m_ReadBuffer.get(), m_WriteBuffer.get() + 1, newFileSize));
            EXPECT_EQ(newFileSize, readSize);
        }
    }

    void CheckFileAfterWriteReOpen() NN_NOEXCEPT
    {
        // 再オープンし、ファイルの中身を確認する
        nn::fs::FileHandle fileHandle;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::fs::OpenFile(
                &fileHandle,
                TestFilePath,
                static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read)
            )
        );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(fileHandle);
        };

        size_t readSize = 0;

        int64_t fileSize = 0;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, fileHandle));
        size_t newFileSize = static_cast<size_t>(fileSize);
        if( m_TestFileSize == newFileSize )
        {
            // ファイルサイズが変化していない場合は、内容が変更されていないことを確認する
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(&readSize, fileHandle, 0, m_ReadBuffer.get(), m_TestFileSize));
            EXPECT_EQ(0, memcmp(m_ReadBuffer.get(), m_WriteBuffer.get(), m_TestFileSize));
            EXPECT_EQ(m_TestFileSize, readSize);
        }
        else
        {
            // ファイルサイズが変化している場合は、内容が拡張時の書き込みのものになっていることを確認する
            EXPECT_LT(m_TestFileSize, newFileSize);
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(&readSize, fileHandle, 0, m_ReadBuffer.get(), newFileSize));
            EXPECT_EQ(0, memcmp(m_ReadBuffer.get(), m_WriteBuffer.get() + 1, newFileSize));
            EXPECT_EQ(newFileSize, readSize);
        }
    }

    void CheckFileAfterSetSizeNoClose(const nn::fs::FileHandle& fileHandle) NN_NOEXCEPT
    {
        // 失敗後のファイルサイズを確認する
        int64_t currentFileSize = 0;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetFileSize(&currentFileSize, fileHandle));
        EXPECT_EQ(m_TestFileSize, currentFileSize);

        // Flush した時にエラーにならないことを確認する
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::FlushFile(fileHandle));
    }

    void CheckFileAfterSetSizeReOpen() NN_NOEXCEPT
    {
        nn::fs::FileHandle fileHandle;

        // 再オープンし、ファイルサイズを確認する
        NNT_EXPECT_RESULT_SUCCESS(
            nn::fs::OpenFile(
                &fileHandle,
                TestFilePath,
                static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read)
            )
        );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(fileHandle);
        };
        int64_t currentFileSize = 0;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetFileSize(&currentFileSize, fileHandle));
        EXPECT_EQ(m_TestFileSize, currentFileSize);

        // ファイルの中身が変わっていないことを確認する
        size_t readSize = 0;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(&readSize, fileHandle, 0, m_ReadBuffer.get(), m_TestFileSize));
        EXPECT_EQ(0, memcmp(m_ReadBuffer.get(), m_WriteBuffer.get(), m_TestFileSize));
        EXPECT_EQ(m_TestFileSize, readSize);
    }

private:
    size_t m_TestFileSize;
    bool m_IsSetupFinished;
    std::unique_ptr<char[]> m_WriteBuffer;
    std::unique_ptr<char[]> m_ReadBuffer;
};

const char AfterExpandFileErrorTest::MountName[] = "test";
const char AfterExpandFileErrorTest::DirectoryPath[] = "test:/test";
const char AfterExpandFileErrorTest::TestFilePath[] = "test:/test/test";
#ifdef NN_BUILD_CONFIG_ADDRESS_32
const size_t AfterExpandFileErrorTest::MaxFileSize = 0x10000000;
#else
const size_t AfterExpandFileErrorTest::MaxFileSize = 0x40000000;
#endif
}

TEST_F(AfterExpandFileErrorTest, MountSaveDataForDebugHeavy)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug(MountName));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(MountName);
    };
    FillFreeSpace();
    NN_UTIL_SCOPE_EXIT
    {
        DeleteAllDummyFiles();
    };
    ASSERT_TRUE(IsSetupFinished());

    DoTest();
}

TEST_F(AfterExpandFileErrorTest, MountSaveDataHeavy)
{
    nn::fs::UserId userId = {{ 0x00ull, 0x01ull }};
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSaveData(MountName, userId));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(MountName);
        nnt::fs::util::DeleteAllTestSaveData();
    };
    FillFreeSpace();
    NN_UTIL_SCOPE_EXIT
{
        DeleteAllDummyFiles();
    };
    ASSERT_TRUE(IsSetupFinished());

    DoTest();
}

TEST_F(AfterExpandFileErrorTest, MountBisHeavy)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis(MountName, nn::fs::BisPartitionId::User));
    NN_UTIL_SCOPE_EXIT
    {
            nn::fs::Unmount(MountName);
    };
    FillFreeSpace();
    NN_UTIL_SCOPE_EXIT
    {
        DeleteAllDummyFiles();
    };
    ASSERT_TRUE(IsSetupFinished());

    DoTest();
}

TEST_F(AfterExpandFileErrorTest, MountImageDirectoryNandHeavy)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory(MountName, nn::fs::ImageDirectoryId::Nand));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(MountName);
    };
    FillFreeSpace();
    NN_UTIL_SCOPE_EXIT
    {
        DeleteAllDummyFiles();
    };
    ASSERT_TRUE(IsSetupFinished());

    DoTest();
}

TEST_F(AfterExpandFileErrorTest, MountImageDirectorySdCardHeavy)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountImageDirectory(MountName, nn::fs::ImageDirectoryId::SdCard));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(MountName);
    };
    FillFreeSpace();
    NN_UTIL_SCOPE_EXIT
    {
        DeleteAllDummyFiles();
    };
    ASSERT_TRUE(IsSetupFinished());

    DoTest();
}

TEST_F(AfterExpandFileErrorTest, MountSdCardForDebugHeavy)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug(MountName));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(MountName);
    };
    FillFreeSpace();
    NN_UTIL_SCOPE_EXIT
    {
        DeleteAllDummyFiles();
    };
    ASSERT_TRUE(IsSetupFinished());

    DoTest();
}
#endif
