﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/os.h>
#include <nn/fs/fs_ResultHandler.h>

#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>

#include "testFs_FsLib_Mount.h"

using namespace nn::fs;

#include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>

nnt::fs::util::TemporaryHostDirectory g_TestDirPath;

namespace
{
    void CreateAndDeleteFileTest(const char* fileName) NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS(CreateFile(fileName, 1024));
        NNT_EXPECT_RESULT_SUCCESS(DeleteFile(fileName));
    }
}

void ValidateFileSystem(const char* path)
{
    DirectoryEntryType entryType;
    NNT_EXPECT_RESULT_SUCCESS(GetEntryType(&entryType, path));

    nnt::fs::util::String filePath(path);
    filePath += "test.txt";
    CreateAndDeleteFileTest(filePath.c_str());
}

nn::Result DumpDirectoryRecursiveWithWhiteList(const char* path)
{
    return nnt::fs::util::IterateDirectoryRecursive(path,
        [](const char* path, const DirectoryEntry& entry){
            NN_UNUSED(entry);
            NN_LOG("DIR  <%s>\n", path);
            NN_RESULT_SUCCESS;
        },
        nullptr,
        [](const char* path, const DirectoryEntry& entry) -> nn::Result {
            nnt::fs::util::Hash hash = {{0}};

            auto result = nnt::fs::util::CalculateFileHash(&hash, path);
            NN_RESULT_TRY(result)
                NN_RESULT_CATCH(ResultTargetLocked)
                {
                    if(
                        strncmp("@System:/ns/apcdb/lru_list.dat", path, strnlen(path, nn::fs::EntryNameLengthMax) + 1) == 0         ||
                        strncmp("@System:/save/80000000", path, strnlen("@System:/save/80000000", nn::fs::EntryNameLengthMax)) == 0 // system save
                        )
                    {
                        NN_LOG("CalculateFileHash failed at <%-40s>, but ignored.\n", path);
                    }
                    else
                    {
                        NN_LOG("CalculateFileHash at <%-40s> with ResultTargetLocked.\n", path);
                        EXPECT_TRUE(false);
                        // continue to retrieve
                    }
                }
                NN_RESULT_CATCH(ResultDataCorrupted)
                {
                    if(strncmp("@SdCardContent:/private", path, strnlen(path, nn::fs::EntryNameLengthMax) + 1) == 0) // 未暗号化のため
                    {
                        NN_LOG("CalculateFileHash failed at <%-40s>, but ignored.\n", path);
                    }
                    else
                    {
                        NN_LOG("CalculateFileHash at <%-40s> with ResultDataCorrupted.\n", path);
                        EXPECT_TRUE(false);
                        // continue to retrieve
                    }
                }
                NN_RESULT_CATCH_ALL
                {
                    NN_LOG("CalculateFileHash failed at <%-40s> with error 0x%08x.\n", path, result.GetInnerValueForDebug());
                    EXPECT_TRUE(false);
                    // continue to retrieve
                }
            NN_RESULT_END_TRY

            NN_LOG("FILE <%-40s> %10lld\n", path, entry.fileSize);
            nnt::fs::util::DumpBuffer(hash.value, sizeof(hash.value));
            NN_RESULT_SUCCESS;
        }
        );
}

#if defined(NN_BUILD_CONFIG_OS_WIN32)
// For GameCardManager.cpp
extern "C" void nninitStartup()
{
    NN_LOG("malloc disabled by %s %d\n", __FUNCTION__, __LINE__);
    const size_t MemoryHeapSize = 1024 * 1024 * 1024;
    nn::os::SetMemoryHeapSize(MemoryHeapSize);
}
#endif

extern "C" void nnMain()
{
    nnt::fs::util::LoadMountTest();

    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    nnt::fs::util::mock::InitializeForFsTest(&argc, argv);

    SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nn::fs::SetEnabledAutoAbort(false);
    nnt::fs::util::ResetAllocateCount();

    g_TestDirPath.Create();

#ifndef NN_BUILD_CONFIG_OS_WIN
    nn::ns::Initialize();
    nn::ns::GameCardStopper stopper;
    nn::ns::GetGameCardStopper(&stopper);
#endif

    auto result = RUN_ALL_TESTS();

    g_TestDirPath.Delete();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(result);
}
