﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>

#include <nn/fs.h>

#include <nn/os/os_MemoryHeap.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>


namespace {

const char* RomMountName = "rom";

const char* TestDataFileName = "testdata";
const char* TestData = "0123456789ABCDEF";

const size_t MinimumFileDataCacheBufferSize = 1 * 1024 * 1024;
nn::Bit8 g_MinimumFileDataCacheBuffer[MinimumFileDataCacheBufferSize];

bool ReadCheck()
{
    char buffer[16];

    char path[32];
    std::snprintf(path, 32, "%s:/%s", RomMountName, TestDataFileName);

    nn::fs::FileHandle fileHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(fileHandle);
    };

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::ReadFile(fileHandle, 0, buffer, 16));

    return std::strncmp(buffer, TestData, 16) == 0;
}

void ExpectNoMemoryLeak()
{
    bool memoryLeaked = nnt::fs::util::CheckMemoryLeak();
    EXPECT_FALSE(memoryLeaked);
    if (memoryLeaked)
    {
        nnt::fs::util::ResetAllocateCount();
    }
}

}

TEST(PreCondition, Violation)
{
    // pBuffer != nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        nn::fs::EnableGlobalFileDataCache(nullptr, 0), "");

    // bufferSize >= 1048756
    EXPECT_DEATH_IF_SUPPORTED(
        nn::fs::EnableGlobalFileDataCache(g_MinimumFileDataCacheBuffer, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(
        nn::fs::EnableGlobalFileDataCache(g_MinimumFileDataCacheBuffer, MinimumFileDataCacheBufferSize - 1), "");

    // double initialization
    nn::fs::EnableGlobalFileDataCache(g_MinimumFileDataCacheBuffer, MinimumFileDataCacheBufferSize);
    EXPECT_DEATH_IF_SUPPORTED(
        nn::fs::EnableGlobalFileDataCache(g_MinimumFileDataCacheBuffer, MinimumFileDataCacheBufferSize), "");

    nn::fs::DisableGlobalFileDataCache();

    ExpectNoMemoryLeak();
}

TEST(Specification, NoRestrictedBufferAlignment)
{
    {
        size_t fsCacheSize;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::QueryMountRomCacheSize(&fsCacheSize));

        void* fsCache = std::malloc(fsCacheSize);
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::MountRom(RomMountName, fsCache, fsCacheSize));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(RomMountName);
            std::free(fsCache);
        };

        size_t dataCacheSize = MinimumFileDataCacheBufferSize + nn::os::MemoryPageSize;
        nn::Bit8* dataCache = static_cast<nn::Bit8*>(std::malloc(dataCacheSize));
        NN_UTIL_SCOPE_EXIT
        {
            std::free(dataCache);
        };

        for (int i = 0; i < nn::os::MemoryPageSize; i++)
        {
            nn::fs::EnableGlobalFileDataCache(dataCache + i, MinimumFileDataCacheBufferSize);
            ASSERT_TRUE(ReadCheck());
            nn::fs::DisableGlobalFileDataCache();
        }
    }
    ExpectNoMemoryLeak();
}

TEST(Specification, DoubleFinalization)
{
    {
        size_t fsCacheSize;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::QueryMountRomCacheSize(&fsCacheSize));

        void* fsCache = std::malloc(fsCacheSize);
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::MountRom(RomMountName, fsCache, fsCacheSize));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(RomMountName);
            std::free(fsCache);
        };

        nn::fs::EnableGlobalFileDataCache(g_MinimumFileDataCacheBuffer, MinimumFileDataCacheBufferSize);
        ASSERT_TRUE(ReadCheck());
        nn::fs::DisableGlobalFileDataCache();
        nn::fs::DisableGlobalFileDataCache();
    }
    ExpectNoMemoryLeak();
}

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
