﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/fs/fs_FileSystemPrivate.h>
#include <nn/fs/fs_ResultHandler.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>

TEST(FileApiShim, MakeReadOption)
{
    nn::fs::ReadOption option;

    option = nn::fs::ReadOption::MakeValue(0);
    EXPECT_EQ(0, option.reserved);
}

TEST(FileApiShim, MakeWriteOption)
{
    nn::fs::WriteOption option;

    option = nn::fs::WriteOption::MakeValue(0);
    EXPECT_EQ(0, option.flags);

    option = nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush);
    EXPECT_EQ(nn::fs::WriteOptionFlag_Flush, option.flags);

    option = nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush | nn::fs::WriteOptionFlag_Flush);
    EXPECT_EQ(nn::fs::WriteOptionFlag_Flush, option.flags);

    option = nn::fs::WriteOption::MakeValue(0);
    EXPECT_EQ(0, option.flags);
}

TEST(FileApiShim, GetFileOpenMode)
{
    nnt::fs::util::ResetGlobalNewDeleteFlag();
    nnt::fs::util::SetStackTraceDumpOnGlobalNewDeleteCallEnabled(true);
    NN_UTIL_SCOPE_EXIT
    {
        nnt::fs::util::SetStackTraceDumpOnGlobalNewDeleteCallEnabled(false);
    };

    nnt::fs::util::TemporaryHostDirectory testRootPath;
    testRootPath.Create();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHostRoot());
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::UnmountHostRoot();
    };

    auto testFilePath = testRootPath.GetPath() + "/test.bin";
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(testFilePath.c_str(), 0));

    static const int OpenModes[] =
    {
        nn::fs::OpenMode_Read,
        nn::fs::OpenMode_Write,
        nn::fs::OpenMode_Read | nn::fs::OpenMode_Write,
        nn::fs::OpenMode_Read | nn::fs::OpenMode_AllowAppend,
        nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend,
        nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend,
    };

    for( auto expectedMode : OpenModes )
    {
        nn::fs::FileHandle handle;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handle, testFilePath.c_str(), static_cast<nn::fs::OpenMode>(expectedMode)));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        auto actualMode = nn::fs::GetFileOpenMode(handle);

        EXPECT_EQ(expectedMode, actualMode);
    }

    EXPECT_FALSE(nnt::fs::util::IsGlobalNewDeleteCalled());
}

TEST(FileApiShim, OpenFileWithIFile)
{
    std::unique_ptr<nn::fs::fsa::IFile> file(new nnt::fs::util::MemoryFile());
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultNullptrArgument,
        nn::fs::OpenFile(nullptr, std::move(file), nn::fs::OpenMode_Read)
    );
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nn::fs::SetEnabledAutoAbort(false);
    nnt::fs::util::ResetAllocateCount();

    auto result = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(result);
}
