﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/result/result_HandlingUtility.h>

#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>

namespace{
    struct TestArgment
    {
        nn::fs::UserId userId;
        int64_t threadId;
        int64_t waitTime;
        int processCount;
    };

    const int SaveDataCount = 4;
    const int ThreadCount = SaveDataCount * 2;
    const size_t StackSize = 64 * 1024;
    const int TestProcessCount = 30;

    const int DefaultSaveDataSize = 16 * 1024 * 1024;
    const int DefaultJournalSize = 8 * 1024 * 1024;
    const int ExtendSaveDataSize = 1 * 1024 * 1024;
    const int ExtendJournalSize = 256 * 1024;

    int g_MountSuccessCount = 0;
    int g_ExtandSuccessCount = 0;

    nn::os::ThreadType g_Threads[ThreadCount];
    NN_OS_ALIGNAS_THREAD_STACK static char g_StackBuffer[StackSize * ThreadCount] = {};
    nn::fs::SaveDataId g_SaveDataIdList[SaveDataCount] = {};

    void MakeMountName(char* buffer, size_t bufferSize, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
    {
        memset(buffer, 0, bufferSize);
        nn::util::SNPrintf(buffer, bufferSize, "save%d", saveDataId);
    }

    nn::fs::SaveDataId UserIdToSaveDataId(nn::fs::UserId userId) NN_NOEXCEPT
    {
        return g_SaveDataIdList[userId._data[1]];
    }

    void ExtendSaveData(void* pThreadArgumnet) NN_NOEXCEPT
    {
        TestArgment* pTestArgment = reinterpret_cast<TestArgment*>(pThreadArgumnet);

        // マウント処理と同時実行されやすくするため、乱数時間停止する
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(pTestArgment->waitTime));

        nn::Result result
            = nn::fs::ExtendSaveData(
                  nn::fs::SaveDataSpaceId::User,
                  UserIdToSaveDataId(pTestArgment->userId),
                  DefaultSaveDataSize + ExtendSaveDataSize * pTestArgment->processCount,
                  DefaultJournalSize + ExtendJournalSize * pTestArgment->processCount
              );

        if( result.IsFailure() )
        {
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultTargetLocked,
                result
            );
            return;
        }

        ++g_ExtandSuccessCount;
    }

    void MountSaveData(void* pThreadArgumnet) NN_NOEXCEPT
    {
        TestArgment* pTestArgment = reinterpret_cast<TestArgment*>(pThreadArgumnet);

        // 拡張処理と同時実行されやすくするため、乱数時間停止する
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(pTestArgment->waitTime));

        std::unique_ptr<char[]> mountName(new char[16]);
        MakeMountName(mountName.get(), 16, pTestArgment->threadId);

        nn::Result result = nn::fs::MountSaveData(mountName.get(), pTestArgment->userId);
        if( result.IsFailure() )
        {
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultSaveDataExtending,
                result
            );
            return;
        }

        nn::fs::Unmount(mountName.get());
        ++g_MountSuccessCount;
    }
}

class ExtendMultiSaveData : public ::testing::Test
{
public:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        nnt::fs::util::DeleteAllTestSaveData();

        for( int i = 0; i < SaveDataCount; ++i )
        {
            NNT_EXPECT_RESULT_SUCCESS(CreateSaveData(i));
        }

        // UserId → SaveDataId のリストをここで作成しておく
        std::unique_ptr<nn::fs::SaveDataIterator> iter;
        nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User);
        std::unique_ptr<nn::fs::SaveDataInfo[]> infoList(new nn::fs::SaveDataInfo[SaveDataCount]);
        int64_t readCount = 0;
        iter->ReadSaveDataInfo(&readCount, infoList.get(), SaveDataCount);
        ASSERT_EQ(SaveDataCount, readCount);
        for( int i = 0; i < SaveDataCount; ++i )
        {
            nn::fs::SaveDataInfo info = infoList[i];
            g_SaveDataIdList[info.saveDataUserId._data[1]] = info.saveDataId;
        }
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        nnt::fs::util::DeleteAllTestSaveData();
    }

private:
    nn::Result CreateSaveData(int userNumber) NN_NOEXCEPT
    {
        nn::fs::UserId userId = {{ 0, static_cast<nn::Bit64>(userNumber + 1) }};
        NN_RESULT_DO(
            nn::fs::CreateSaveData(
                nnt::fs::util::ApplicationId,
                userId,
                0,
                DefaultSaveDataSize,
                DefaultJournalSize,
                0
            )
        );

        const size_t mountNameMaxLength = 16;
        std::unique_ptr<char[]> mountName(new char[mountNameMaxLength]);
        MakeMountName(mountName.get(), mountNameMaxLength, userNumber);
        NN_RESULT_DO(
            nn::fs::MountSaveData(mountName.get(), userId)
        );
        nn::fs::Unmount(mountName.get());
        NN_RESULT_SUCCESS;
    }
};

// 拡張とマウントを同時に行うテスト
TEST_F(ExtendMultiSaveData, ExtendWithMount)
{
    TestArgment arguments[ThreadCount];

    g_MountSuccessCount = 0;
    g_ExtandSuccessCount = 0;

    for( int i = 0; i < ThreadCount; ++i )
    {
        arguments[i].userId = {{ 0, static_cast<nn::Bit64>(i % SaveDataCount) + 1 }};
        arguments[i].threadId = i;
    }

    std::mt19937 random(nnt::fs::util::GetRandomSeed());

    // マウントと拡張の同時実行を TestProcessCount 回行う
    for( int i = 0; i < TestProcessCount; ++i )
    {
        for( int j = 0; j < ThreadCount; ++j )
        {
            arguments[j].processCount = i + 1;
            arguments[j].waitTime
                = std::uniform_int_distribution<int64_t>(0, 1000)(random);

            // 前半のスレッドでは拡張処理、後半のスレッドでマウント処理を行う
            NNT_ASSERT_RESULT_SUCCESS(
                nn::os::CreateThread(
                    &g_Threads[j],
                    (j < SaveDataCount) ? ExtendSaveData : MountSaveData,
                    &arguments[j],
                    g_StackBuffer + StackSize * (j),
                    StackSize,
                    nn::os::DefaultThreadPriority
                )
            );
        }
        for( auto& thread : g_Threads )
        {
            nn::os::StartThread(&thread);
        }
        for( auto& thread : g_Threads )
        {
            nn::os::WaitThread(&thread);
            nn::os::DestroyThread(&thread);
        }
    }

    // マウント/拡張処理が毎回成功/毎回失敗ということは避ける
    EXPECT_NE(0, g_MountSuccessCount);
    EXPECT_NE(0, g_ExtandSuccessCount);
    EXPECT_NE(TestProcessCount, g_MountSuccessCount * SaveDataCount);
    EXPECT_NE(TestProcessCount, g_ExtandSuccessCount * SaveDataCount);
}
