﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>

#include <nn/fs/fs_ResultHandler.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

#include <fs_Allocator.h>

namespace {
    bool g_IsAllocationErrorRequested = false;
}

void* Allocate(const size_t size) NN_NOEXCEPT
{
    if( g_IsAllocationErrorRequested )
    {
        NN_LOG("%zu bytes required, but failed\n", size);
        return nullptr;
    }
    else
    {
        return nnt::fs::util::Allocate(size);
    }
}

void Deallocate(void* p, const size_t size) NN_NOEXCEPT
{
    if( p != nullptr )
    {
        nnt::fs::util::Deallocate(p, size);
    }
}

TEST(Allocator, StlVectorPushBackWithMallocFailure)
{
    g_IsAllocationErrorRequested = true;
    NN_UTIL_SCOPE_EXIT
    {
        g_IsAllocationErrorRequested = false;
    };

    std::vector<int, nn::fs::Allocator<int>> testVector;
    testVector.push_back(1);
    testVector.push_back(2);
    ASSERT_TRUE(testVector.get_allocator().IsAllocationFailed());
}

TEST(Allocator, StlVectorReserveWithMallocFailure)
{
    NN_UTIL_SCOPE_EXIT
    {
        g_IsAllocationErrorRequested = false;
    };

    std::vector<int, nn::fs::Allocator<int>> testVector;
    testVector.push_back(1);
    g_IsAllocationErrorRequested = true;
    testVector.reserve(1024);
    ASSERT_TRUE(testVector.get_allocator().IsAllocationFailed());
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    nn::fs::SetEnabledAutoAbort(false);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetAllocator(Allocate, Deallocate);

    auto testResult = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
