﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/mem.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>

#include <nn/fs/fs_ResultHandler.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

// malloc 群と nninitStartup() を定義しているアプリ

TEST(Allocator, UseDefaultAllocator)
{
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountHostRoot());
    nn::fs::UnmountHostRoot();
}

TEST(Allocator, SetAllocatorAfterUsingDefaultAllocator_Death)
{
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountHostRoot()); // use default
    nn::fs::UnmountHostRoot();

    EXPECT_DEATH_IF_SUPPORTED(nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate), "");
    //nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
}

// sample の InitStartupWithMalloc からコピペ
namespace {

    const size_t                MallocBufferSize = 4 * 1024 * 1024;
    NN_ALIGNAS(4096) uint8_t    g_MallocBuffer[ MallocBufferSize ];
    nn::util::TypedStorage<nn::mem::StandardAllocator,sizeof(nn::mem::StandardAllocator),NN_ALIGNOF(nn::mem::StandardAllocator)>    g_SampleAllocator;

}

extern "C" void* malloc(size_t size)
{
    return Get(g_SampleAllocator).Allocate(size);
}

extern "C" void free(void* p)
{
    if (p)
    {
        Get(g_SampleAllocator).Free(p);
    }
}

extern "C" void* calloc(size_t num, size_t size)
{
    size_t sum = num * size;
    void*  p   = std::malloc(sum);

    if (p)
    {
        std::memset(p, 0, sum);
    }
    return p;
}

extern "C" void* realloc(void* p, size_t newSize)
{
    return Get(g_SampleAllocator).Reallocate(p, newSize);
}

extern "C" void* aligned_alloc(size_t alignment, size_t size)
{
    NN_UNUSED(alignment);
    NN_UNUSED(size);

    return nullptr;
}

extern "C" void nninitStartup()
{
    NN_LOG("malloc enabled by %s %d\n", __FUNCTION__, __LINE__);

    new( &Get(g_SampleAllocator) ) nn::mem::StandardAllocator;
    Get(g_SampleAllocator).Initialize(g_MallocBuffer, MallocBufferSize);
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    nn::fs::SetEnabledAutoAbort(false);
    nnt::fs::util::ResetAllocateCount();

    auto testResult = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
