﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <nn/fs.h>
#include <nn/fs/fs_SaveDataForDebug.h>
#include "testFs_FsLib_AccessLog.h"

class FileDataCache : public testing::Test
{
public:
    static void SetUpTestCase();
    static void TearDownTestCase();

    static void ClearFileDataCache();

protected:
    nn::fs::FileHandle GetHandle() NN_NOEXCEPT
    {
        return g_Handle;
    }

private:
    static const size_t FileDataCacheBufferSize = 16 * 1024 * 1024;
    static nn::fs::FileHandle g_Handle;
    static std::unique_ptr<char, decltype(&nnt::fs::util::DeleterBuffer)> g_FileDataCacheBuffer;
    static std::unique_ptr<char, decltype(&nnt::fs::util::DeleterBuffer)> g_RomFsCacheBuffer;
};

nn::fs::FileHandle FileDataCache::g_Handle;
std::unique_ptr<char, decltype(&nnt::fs::util::DeleterBuffer)> FileDataCache::g_FileDataCacheBuffer(nullptr, nnt::fs::util::DeleterBuffer);
std::unique_ptr<char, decltype(&nnt::fs::util::DeleterBuffer)> FileDataCache::g_RomFsCacheBuffer(nullptr, nnt::fs::util::DeleterBuffer);

void FileDataCache::SetUpTestCase()
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "EnableGlobalFileDataCache", size: 16777216 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "QueryMountRomCacheSize", size: 88 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountRom", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "OpenFile", path: "test:/4MB", open_mode: 0x1 }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "EnableGlobalFileDataCache", size: 16777216 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "QueryMountRomCacheSize", size: 88 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountRom", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "OpenFile", path: "test:/4MB", open_mode: 0x1 }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::EnableGlobalFileDataCache", size: 16777216 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::QueryMountRomCacheSize", size: 32 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountRom", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "test:/4MB", open_mode: 0x1 }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::EnableGlobalFileDataCache", size: 16777216 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::QueryMountRomCacheSize", size: 32 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountRom", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "test:/4MB", open_mode: 0x1 }
*/
{
    {
        auto fileDataBuffer = nnt::fs::util::AllocateBuffer(FileDataCacheBufferSize);
        g_FileDataCacheBuffer.swap(fileDataBuffer);
        ASSERT_NE(nullptr, g_FileDataCacheBuffer.get());
        nn::fs::EnableGlobalFileDataCache(g_FileDataCacheBuffer.get(), FileDataCacheBufferSize);
    }
    {
        size_t cacheBufferSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheBufferSize));
        auto romBuffer = nnt::fs::util::AllocateBuffer(cacheBufferSize);
        g_RomFsCacheBuffer.swap(romBuffer);
        ASSERT_NE(nullptr, g_RomFsCacheBuffer.get());
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountRom(nnt::fs::MountName, g_RomFsCacheBuffer.get(), cacheBufferSize));
    }
    const char* filePath = "test:/4MB";
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&g_Handle, filePath, nn::fs::OpenMode_Read));
}

void FileDataCache::ClearFileDataCache()
{
    nn::fs::DisableGlobalFileDataCache();
    nn::fs::EnableGlobalFileDataCache(g_FileDataCacheBuffer.get(), FileDataCacheBufferSize);
}

TEST_F(FileDataCache, ReadFile_Success)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1024, fetched: [{ offset: 0, size: 32256 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 65536, size: 1024, fetched: [{ offset: 65024, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 131072, size: 1024, fetched: [{ offset: 130560, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 196608, size: 1024, fetched: [{ offset: 196096, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 262144, size: 1024, fetched: [{ offset: 261632, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 327680, size: 1024, fetched: [{ offset: 327168, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 393216, size: 1024, fetched: [{ offset: 392704, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 458752, size: 1024, fetched: [{ offset: 458240, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 524288, size: 1024, fetched: [{ offset: 523776, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 589824, size: 1024, fetched: [{ offset: 589312, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 655360, size: 1024, fetched: [{ offset: 654848, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 720896, size: 1024, fetched: [{ offset: 720384, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 786432, size: 1024, fetched: [{ offset: 785920, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 851968, size: 1024, fetched: [{ offset: 851456, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 917504, size: 1024, fetched: [{ offset: 916992, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 983040, size: 1024, fetched: [{ offset: 982528, size: 16384 }] }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1024, fetched: [{ offset: 0, size: 32256 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 65536, size: 1024, fetched: [{ offset: 65024, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 131072, size: 1024, fetched: [{ offset: 130560, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 196608, size: 1024, fetched: [{ offset: 196096, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 262144, size: 1024, fetched: [{ offset: 261632, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 327680, size: 1024, fetched: [{ offset: 327168, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 393216, size: 1024, fetched: [{ offset: 392704, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 458752, size: 1024, fetched: [{ offset: 458240, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 524288, size: 1024, fetched: [{ offset: 523776, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 589824, size: 1024, fetched: [{ offset: 589312, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 655360, size: 1024, fetched: [{ offset: 654848, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 720896, size: 1024, fetched: [{ offset: 720384, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 786432, size: 1024, fetched: [{ offset: 785920, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 851968, size: 1024, fetched: [{ offset: 851456, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 917504, size: 1024, fetched: [{ offset: 916992, size: 16384 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 983040, size: 1024, fetched: [{ offset: 982528, size: 16384 }] }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 65536, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 131072, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 196608, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 262144, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 327680, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 393216, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 458752, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 524288, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 589824, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 655360, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 720896, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 786432, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 851968, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 917504, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 983040, size: 1024 }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 65536, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 131072, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 196608, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 262144, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 327680, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 393216, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 458752, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 524288, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 589824, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 655360, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 720896, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 786432, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 851968, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 917504, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 983040, size: 1024 }
*/
{
    size_t readBufferSize = 4 * 1024 * 1024;
    auto readBuffer = nnt::fs::util::AllocateBuffer(readBufferSize);
    for( int i = 0; i < 16; ++i)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(GetHandle(), i * 64 * 1024, readBuffer.get(), 1024));
    }
}

TEST_F(FileDataCache, ReadFile_Failure)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 4194304, size: 1048576, fetched: [{ offset: 4193792, size: 512 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 5242880, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 6291456, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 7340032, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 8388608, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 9437184, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 10485760, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 11534336, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 12582912, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 13631488, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 14680064, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 15728640, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 16777216, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 17825792, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 18874368, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 19922944, size: 1048576 }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 4194304, size: 1048576, fetched: [{ offset: 4193792, size: 512 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 5242880, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 6291456, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 7340032, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 8388608, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 9437184, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 10485760, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 11534336, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 12582912, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 13631488, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 14680064, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 15728640, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 16777216, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 17825792, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 18874368, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 19922944, size: 1048576 }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 4194304, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 5242880, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 6291456, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 7340032, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 8388608, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 9437184, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 10485760, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 11534336, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 12582912, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 13631488, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 14680064, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 15728640, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 16777216, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 17825792, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 18874368, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 19922944, size: 1048576 }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 4194304, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 5242880, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 6291456, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 7340032, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 8388608, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 9437184, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 10485760, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 11534336, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 12582912, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 13631488, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 14680064, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 15728640, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 16777216, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 17825792, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 18874368, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00177A02, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 19922944, size: 1048576 }
*/
{
    size_t readBufferSize = 4 * 1024 * 1024;
    auto readBuffer = nnt::fs::util::AllocateBuffer(readBufferSize);
    for( int i = 4; i < 20; ++i)
    {
        ASSERT_TRUE(nn::fs::ReadFile(GetHandle(), i * 1024 * 1024, readBuffer.get(), 1024 * 1024).IsFailure());
    }
}

TEST_F(FileDataCache, FetchedEntriesCanBeObtainedInAscendingOrder)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DisableGlobalFileDataCache" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "EnableGlobalFileDataCache", size: 16777216 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 32768, size: 262144, fetched: [{ offset: 32256, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 327680, size: 262144, fetched: [{ offset: 327168, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 622592, size: 262144, fetched: [{ offset: 622080, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 917504, size: 262144, fetched: [{ offset: 916992, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1212416, size: 262144, fetched: [{ offset: 1211904, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1507328, size: 262144, fetched: [{ offset: 1506816, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1802240, size: 262144, fetched: [{ offset: 1801728, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 4194304, fetched: [{ offset: 0, size: 32256 }, { offset: 310784, size: 16384 }, { offset: 605696, size: 16384 }, { offset: 900608, size: 16384 }, { offset: 1195520, size: 16384 }, { offset: 1490432, size: 16384 }, { offset: 1785344, size: 16384 }, { offset: 2080256, size: 2114048 }] }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DisableGlobalFileDataCache" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "EnableGlobalFileDataCache", size: 16777216 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 32768, size: 262144, fetched: [{ offset: 32256, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 327680, size: 262144, fetched: [{ offset: 327168, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 622592, size: 262144, fetched: [{ offset: 622080, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 917504, size: 262144, fetched: [{ offset: 916992, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1212416, size: 262144, fetched: [{ offset: 1211904, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1507328, size: 262144, fetched: [{ offset: 1506816, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1802240, size: 262144, fetched: [{ offset: 1801728, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 4194304, fetched: [{ offset: 0, size: 32256 }, { offset: 310784, size: 16384 }, { offset: 605696, size: 16384 }, { offset: 900608, size: 16384 }, { offset: 1195520, size: 16384 }, { offset: 1490432, size: 16384 }, { offset: 1785344, size: 16384 }, { offset: 2080256, size: 2114048 }] }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DisableGlobalFileDataCache" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::EnableGlobalFileDataCache", size: 16777216 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 32768, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 327680, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 622592, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 917504, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1212416, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1507328, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1802240, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 4194304 }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DisableGlobalFileDataCache" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::EnableGlobalFileDataCache", size: 16777216 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 32768, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 327680, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 622592, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 917504, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1212416, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1507328, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1802240, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 4194304 }
*/
{
    ClearFileDataCache();

    size_t readBufferSize = 4 * 1024 * 1024;
    auto readBuffer = nnt::fs::util::AllocateBuffer(readBufferSize);

    int64_t stride = 32 * 1024;
    size_t smallReadSize = 256 * 1024;
    size_t largeReadSize = readBufferSize;

    for (int64_t i = 0; i < 7; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(GetHandle(), stride + i * (smallReadSize + stride), readBuffer.get(), smallReadSize));
    }
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(GetHandle(), 0, readBuffer.get(), largeReadSize));
}

TEST_F(FileDataCache, LastFetchedEntryAlwaysPointsToTheTailEndRegion)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DisableGlobalFileDataCache" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "EnableGlobalFileDataCache", size: 16777216 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 32768, size: 262144, fetched: [{ offset: 32256, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 327680, size: 262144, fetched: [{ offset: 327168, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 622592, size: 262144, fetched: [{ offset: 622080, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 917504, size: 262144, fetched: [{ offset: 916992, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1212416, size: 262144, fetched: [{ offset: 1211904, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1507328, size: 262144, fetched: [{ offset: 1506816, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1802240, size: 262144, fetched: [{ offset: 1801728, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 2097152, size: 262144, fetched: [{ offset: 2096640, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 2392064, size: 262144, fetched: [{ offset: 2391552, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 2686976, size: 262144, fetched: [{ offset: 2686464, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 2981888, size: 262144, fetched: [{ offset: 2981376, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 3276800, size: 262144, fetched: [{ offset: 3276288, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 4194304, fetched: [{ offset: 0, size: 32256 }, { offset: 310784, size: 16384 }, { offset: 605696, size: 16384 }, { offset: 900608, size: 16384 }, { offset: 1195520, size: 16384 }, { offset: 1490432, size: 16384 }, { offset: 1785344, size: 16384 }, { offset: 3554816, size: 639488 }], fetch_exceeded: true }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DisableGlobalFileDataCache" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "EnableGlobalFileDataCache", size: 16777216 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 32768, size: 262144, fetched: [{ offset: 32256, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 327680, size: 262144, fetched: [{ offset: 327168, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 622592, size: 262144, fetched: [{ offset: 622080, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 917504, size: 262144, fetched: [{ offset: 916992, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1212416, size: 262144, fetched: [{ offset: 1211904, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1507328, size: 262144, fetched: [{ offset: 1506816, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1802240, size: 262144, fetched: [{ offset: 1801728, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 2097152, size: 262144, fetched: [{ offset: 2096640, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 2392064, size: 262144, fetched: [{ offset: 2391552, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 2686976, size: 262144, fetched: [{ offset: 2686464, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 2981888, size: 262144, fetched: [{ offset: 2981376, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 3276800, size: 262144, fetched: [{ offset: 3276288, size: 278528 }] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 4194304, fetched: [{ offset: 0, size: 32256 }, { offset: 310784, size: 16384 }, { offset: 605696, size: 16384 }, { offset: 900608, size: 16384 }, { offset: 1195520, size: 16384 }, { offset: 1490432, size: 16384 }, { offset: 1785344, size: 16384 }, { offset: 3554816, size: 639488 }], fetch_exceeded: true }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DisableGlobalFileDataCache" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::EnableGlobalFileDataCache", size: 16777216 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 32768, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 327680, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 622592, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 917504, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1212416, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1507328, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1802240, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 2097152, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 2392064, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 2686976, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 2981888, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 3276800, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 4194304 }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DisableGlobalFileDataCache" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::EnableGlobalFileDataCache", size: 16777216 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 32768, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 327680, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 622592, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 917504, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1212416, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1507328, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1802240, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 2097152, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 2392064, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 2686976, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 2981888, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 3276800, size: 262144 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 4194304 }
*/
{
    ClearFileDataCache();

    size_t readBufferSize = 4 * 1024 * 1024;
    auto readBuffer = nnt::fs::util::AllocateBuffer(readBufferSize);

    int64_t stride = 32 * 1024;
    size_t smallReadSize = 256 * 1024;
    size_t largeReadSize = readBufferSize;

    for (int64_t i = 0; i < 12; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(GetHandle(), stride + i * (smallReadSize + stride), readBuffer.get(), smallReadSize));
    }
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(GetHandle(), 0, readBuffer.get(), largeReadSize));
}

void FileDataCache::TearDownTestCase()
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DisableGlobalFileDataCache" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DisableGlobalFileDataCache" }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DisableGlobalFileDataCache" }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DisableGlobalFileDataCache" }
*/
{
    nn::fs::CloseFile(g_Handle);
    nn::fs::Unmount(nnt::fs::MountName);
    nn::fs::DisableGlobalFileDataCache();
    g_FileDataCacheBuffer.reset();
    g_RomFsCacheBuffer.reset();
}

TEST(IndividualFileDataCache, IndividualFileDataCacheSuccess)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateFile", path: "test:/file1", size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateFile", path: "test:/file2", size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "EnableIndividualFileDataCache", path: "test:/file1", buffer_size: 2097152, file_size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "OpenFile", path: "test:/file1", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 0, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1048576, fetched: [] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "EnableIndividualFileDataCache", path: "test:/file2", buffer_size: 2097152, file_size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "OpenFile", path: "test:/file2", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 0, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1048576, fetched: [] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 0, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1048576, fetched: [] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DisableIndividualFileDataCache", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 0, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DisableIndividualFileDataCache", path: "test:/file1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DisableIndividualFileDataCache", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteFile", path: "test:/file1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteFile", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateFile", path: "test:/file1", size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateFile", path: "test:/file2", size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "EnableIndividualFileDataCache", path: "test:/file1", buffer_size: 2097152, file_size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "OpenFile", path: "test:/file1", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 0, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1048576, fetched: [] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "EnableIndividualFileDataCache", path: "test:/file2", buffer_size: 2097152, file_size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "OpenFile", path: "test:/file2", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 0, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1048576, fetched: [] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 0, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1048576, fetched: [] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DisableIndividualFileDataCache", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 0, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DisableIndividualFileDataCache", path: "test:/file1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DisableIndividualFileDataCache", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteFile", path: "test:/file1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteFile", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateFile", path: "test:/file1", size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateFile", path: "test:/file2", size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::EnableIndividualFileDataCache", path: "test:/file1", buffer_size: 2097152, file_size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "test:/file1", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 0, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1048576, fetched: [] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::EnableIndividualFileDataCache", path: "test:/file2", buffer_size: 2097152, file_size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "test:/file2", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 0, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1048576, fetched: [] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 0, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1048576, fetched: [] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DisableIndividualFileDataCache", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 0, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DisableIndividualFileDataCache", path: "test:/file1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DisableIndividualFileDataCache", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteFile", path: "test:/file1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteFile", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateFile", path: "test:/file1", size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateFile", path: "test:/file2", size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::EnableIndividualFileDataCache", path: "test:/file1", buffer_size: 2097152, file_size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "test:/file1", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 0, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1048576, fetched: [] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::EnableIndividualFileDataCache", path: "test:/file2", buffer_size: 2097152, file_size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "test:/file2", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 0, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1048576, fetched: [] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 0, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1048576, fetched: [] }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DisableIndividualFileDataCache", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 0, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DisableIndividualFileDataCache", path: "test:/file1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DisableIndividualFileDataCache", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteFile", path: "test:/file1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteFile", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
*/
{
    const char* MountName = "test";
    const char* TestFilePath1 = "test:/file1";
    const char* TestFilePath2 = "test:/file2";
    size_t testFileSize = 1024 * 1024;
    size_t testBufferSize = testFileSize * 2;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::DisableIndividualFileDataCache(TestFilePath1);
        nn::fs::DisableIndividualFileDataCache(TestFilePath2);
        (void)nn::fs::DeleteFile(TestFilePath1);
        (void)nn::fs::DeleteFile(TestFilePath2);
        nn::fs::Unmount(MountName);
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug(MountName));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(TestFilePath1, testFileSize));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(TestFilePath2, testFileSize));

    auto cacheBuffer1 = nnt::fs::util::AllocateBuffer(testBufferSize);
    auto cacheBuffer2 = nnt::fs::util::AllocateBuffer(testBufferSize);
    ASSERT_NE(nullptr, cacheBuffer1);
    ASSERT_NE(nullptr, cacheBuffer2);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::EnableIndividualFileDataCache(TestFilePath1, cacheBuffer1.get(), testBufferSize));

    nn::fs::FileHandle handle1;
    nn::fs::FileHandle handle2;
    auto buffer = nnt::fs::util::AllocateBuffer(testFileSize);
    ASSERT_NE(nullptr, buffer);
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handle1, TestFilePath1, nn::fs::OpenMode_Write | nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle1);
        };

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(handle1, 0, buffer.get(), testFileSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(handle1, 0, buffer.get(), testFileSize));

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::EnableIndividualFileDataCache(TestFilePath2, cacheBuffer2.get(), testBufferSize));

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handle2, TestFilePath2, nn::fs::OpenMode_Write | nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle2);
        };

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(handle2, 0, buffer.get(), testFileSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(handle2, 0, buffer.get(), testFileSize));

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(handle1, 0, buffer.get(), testFileSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(handle1, 0, buffer.get(), testFileSize));

        nn::fs::DisableIndividualFileDataCache(TestFilePath2);

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(handle2, 0, buffer.get(), testFileSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(handle2, 0, buffer.get(), testFileSize));
    }
}

TEST(IndividualFileDataCache, IndividualFileDataCacheFailure)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateFile", path: "test:/file1", size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateFile", path: "test:/file2", size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "EnableIndividualFileDataCache", path: "test:/file1", buffer_size: 2097152, file_size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "OpenFile", path: "test:/file1", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 1048576, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1048576, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "EnableIndividualFileDataCache", path: "test:/file2", buffer_size: 2097152, file_size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "OpenFile", path: "test:/file2", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 1048576, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1048576, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 0, size: 2097152, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 2097152 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DisableIndividualFileDataCache", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 0, size: 2097152, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 2097152 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DisableIndividualFileDataCache", path: "test:/file1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DisableIndividualFileDataCache", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteFile", path: "test:/file1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteFile", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateFile", path: "test:/file1", size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateFile", path: "test:/file2", size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "EnableIndividualFileDataCache", path: "test:/file1", buffer_size: 2097152, file_size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "OpenFile", path: "test:/file1", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 1048576, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1048576, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "EnableIndividualFileDataCache", path: "test:/file2", buffer_size: 2097152, file_size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "OpenFile", path: "test:/file2", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 1048576, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1048576, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 0, size: 2097152, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 2097152 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DisableIndividualFileDataCache", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 0, size: 2097152, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 2097152 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DisableIndividualFileDataCache", path: "test:/file1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DisableIndividualFileDataCache", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteFile", path: "test:/file1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteFile", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateFile", path: "test:/file1", size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateFile", path: "test:/file2", size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::EnableIndividualFileDataCache", path: "test:/file1", buffer_size: 2097152, file_size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "test:/file1", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 1048576, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1048576, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::EnableIndividualFileDataCache", path: "test:/file2", buffer_size: 2097152, file_size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "test:/file2", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 1048576, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1048576, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 0, size: 2097152, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 2097152 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DisableIndividualFileDataCache", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 0, size: 2097152, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 2097152 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DisableIndividualFileDataCache", path: "test:/file1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DisableIndividualFileDataCache", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteFile", path: "test:/file1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteFile", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateFile", path: "test:/file1", size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateFile", path: "test:/file2", size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::EnableIndividualFileDataCache", path: "test:/file1", buffer_size: 2097152, file_size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "test:/file1", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 1048576, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1048576, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::EnableIndividualFileDataCache", path: "test:/file2", buffer_size: 2097152, file_size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "test:/file2", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 1048576, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1048576, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 0, size: 2097152, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 2097152 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DisableIndividualFileDataCache", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 0, size: 2097152, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 2097152 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DisableIndividualFileDataCache", path: "test:/file1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DisableIndividualFileDataCache", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteFile", path: "test:/file1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteFile", path: "test:/file2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
*/
{
    const char* MountName = "test";
    const char* TestFilePath1 = "test:/file1";
    const char* TestFilePath2 = "test:/file2";
    size_t testFileSize = 1024 * 1024;
    size_t testBufferSize = testFileSize * 2;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::DisableIndividualFileDataCache(TestFilePath1);
        nn::fs::DisableIndividualFileDataCache(TestFilePath2);
        (void)nn::fs::DeleteFile(TestFilePath1);
        (void)nn::fs::DeleteFile(TestFilePath2);
        nn::fs::Unmount(MountName);
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug(MountName));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(TestFilePath1, testFileSize));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(TestFilePath2, testFileSize));

    auto cacheBuffer1 = nnt::fs::util::AllocateBuffer(testBufferSize);
    auto cacheBuffer2 = nnt::fs::util::AllocateBuffer(testBufferSize);
    ASSERT_NE(nullptr, cacheBuffer1);
    ASSERT_NE(nullptr, cacheBuffer2);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::EnableIndividualFileDataCache(TestFilePath1, cacheBuffer1.get(), testBufferSize));

    nn::fs::FileHandle handle1;
    nn::fs::FileHandle handle2;
    auto buffer = nnt::fs::util::AllocateBuffer(testFileSize);
    ASSERT_NE(nullptr, buffer);
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handle1, TestFilePath1, nn::fs::OpenMode_Write | nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle1);
        };

        ASSERT_TRUE(nn::fs::WriteFile(handle1, testFileSize, buffer.get(), testFileSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)).IsFailure());
        ASSERT_TRUE(nn::fs::ReadFile(handle1, testFileSize, buffer.get(), testFileSize).IsFailure());

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::EnableIndividualFileDataCache(TestFilePath2, cacheBuffer2.get(), testBufferSize));

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handle2, TestFilePath2, nn::fs::OpenMode_Write | nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle2);
        };

        ASSERT_TRUE(nn::fs::WriteFile(handle2, testFileSize, buffer.get(), testFileSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)).IsFailure());
        ASSERT_TRUE(nn::fs::ReadFile(handle2, testFileSize, buffer.get(), testFileSize).IsFailure());

        ASSERT_TRUE(nn::fs::WriteFile(handle1, 0, buffer.get(), testFileSize * 2, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)).IsFailure());
        ASSERT_TRUE(nn::fs::ReadFile(handle1, 0, buffer.get(), testFileSize * 2).IsFailure());

        nn::fs::DisableIndividualFileDataCache(TestFilePath2);

        ASSERT_TRUE(nn::fs::WriteFile(handle2, 0, buffer.get(), testFileSize * 2, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)).IsFailure());
        ASSERT_TRUE(nn::fs::ReadFile(handle2, 0, buffer.get(), testFileSize * 2).IsFailure());
    }
}

TEST(IndividualFileDataCache, EnableIndividualFileDataCacheFailure)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "EnableIndividualFileDataCache", path: "test:/file" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateFile", path: "test:/file", size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x002F5C02, handle: 0x0000000000000000, priority: Normal, function: "EnableIndividualFileDataCache", path: "test:/file", buffer_size: 1048575, file_size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "OpenFile", path: "test:/file", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000E02, handle: 0x0000000000000000, priority: Normal, function: "EnableIndividualFileDataCache", path: "test:/file", buffer_size: 2097152, file_size: 0 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 0, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteFile", path: "test:/file" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "EnableIndividualFileDataCache", path: "test:/file" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateFile", path: "test:/file", size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x002F5C02, handle: 0x0000000000000000, priority: Normal, function: "EnableIndividualFileDataCache", path: "test:/file", buffer_size: 1048575, file_size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "OpenFile", path: "test:/file", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000E02, handle: 0x0000000000000000, priority: Normal, function: "EnableIndividualFileDataCache", path: "test:/file", buffer_size: 2097152, file_size: 0 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 0, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteFile", path: "test:/file" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::EnableIndividualFileDataCache", path: "test:/file" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateFile", path: "test:/file", size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x002F5C02, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::EnableIndividualFileDataCache", path: "test:/file", buffer_size: 1048575, file_size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "test:/file", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000E02, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::EnableIndividualFileDataCache", path: "test:/file", buffer_size: 2097152, file_size: 0 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 0, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteFile", path: "test:/file" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::EnableIndividualFileDataCache", path: "test:/file" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateFile", path: "test:/file", size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x002F5C02, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::EnableIndividualFileDataCache", path: "test:/file", buffer_size: 1048575, file_size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "test:/file", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000E02, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::EnableIndividualFileDataCache", path: "test:/file", buffer_size: 2097152, file_size: 0 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 0, size: 1048576, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1048576 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteFile", path: "test:/file" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
*/
{
    const char* MountName = "test";
    const char* TestFilePath = "test:/file";
    size_t testFileSize = 1024 * 1024;
    size_t testBufferSize = testFileSize * 2;

    auto cacheBuffer = nnt::fs::util::AllocateBuffer(testBufferSize);

    // マウント前にキャッシュ
    ASSERT_TRUE(nn::fs::EnableIndividualFileDataCache(TestFilePath, cacheBuffer.get(), testBufferSize).IsFailure());

    NN_UTIL_SCOPE_EXIT
    {
        (void)nn::fs::DeleteFile(TestFilePath);
        nn::fs::Unmount(MountName);
    };
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug(MountName));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(TestFilePath, testFileSize));
    nn::fs::FileHandle handle;
    auto buffer = nnt::fs::util::AllocateBuffer(testFileSize);
    ASSERT_NE(nullptr, buffer);
    {
        // バッファサイズが小さい
        ASSERT_TRUE(nn::fs::EnableIndividualFileDataCache(TestFilePath, cacheBuffer.get(), testFileSize - 1).IsFailure());

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };
        // Open 後にキャッシュ
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handle, TestFilePath, nn::fs::OpenMode_Write | nn::fs::OpenMode_Read));
        ASSERT_TRUE(nn::fs::EnableIndividualFileDataCache(TestFilePath, cacheBuffer.get(), testBufferSize).IsFailure());

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(handle, 0, buffer.get(), testFileSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(handle, 0, buffer.get(), testFileSize));
    }
}
