﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/fs/fs_ContentStorage.h>
#include <nn/fs/fs_SaveDataForDebug.h>

#include "testFs_FsLib_AccessLog.h"
namespace
{
    const char* TestDirectoryPath1 = "test:/test1";
    const char* TestDirectoryPath2 = "test:/test1/test2";
    const char* TestDirectoryPath3 = "test:/test1/test3";
    const char* TestFilePath4 = "test:/test4";
    const char* TestFilePath5 = "test:/test1/test2/test5";
}

class OperateEntry : public testing::Test
{
public:
    static void SetUpTestCase();
    static void TearDownTestCase();
};

void OperateEntry::SetUpTestCase()
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateDirectory", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateDirectory", path: "test:/test1/test2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateDirectory", path: "test:/test1/test3" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateFile", path: "test:/test4", size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateFile", path: "test:/test1/test2/test5", size: 1024 }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateDirectory", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateDirectory", path: "test:/test1/test2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateDirectory", path: "test:/test1/test3" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateFile", path: "test:/test4", size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateFile", path: "test:/test1/test2/test5", size: 1024 }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateDirectory", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateDirectory", path: "test:/test1/test2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateDirectory", path: "test:/test1/test3" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateFile", path: "test:/test4", size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateFile", path: "test:/test1/test2/test5", size: 1024 }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateDirectory", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateDirectory", path: "test:/test1/test2" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateDirectory", path: "test:/test1/test3" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateFile", path: "test:/test4", size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateFile", path: "test:/test1/test2/test5", size: 1024 }
*/
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug("test"));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory(TestDirectoryPath1));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory(TestDirectoryPath2));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory(TestDirectoryPath3));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(TestFilePath4, 1024));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(TestFilePath5, 1024));
}

TEST_F(OperateEntry, CreateDeleteDirectory_Success)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateDirectory", path: "test:/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateDirectory", path: "test:/A/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateDirectory", path: "test:/A/A/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateDirectory", path: "test:/A/A/B" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateDirectory", path: "test:/A/A/B/C" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteDirectory", path: "test:/A/A/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CleanDirectoryRecursively", path: "test:/A/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CleanDirectoryRecursively", path: "test:/A/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteDirectoryRecursively", path: "test:/A" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateDirectory", path: "test:/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateDirectory", path: "test:/A/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateDirectory", path: "test:/A/A/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateDirectory", path: "test:/A/A/B" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateDirectory", path: "test:/A/A/B/C" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteDirectory", path: "test:/A/A/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CleanDirectoryRecursively", path: "test:/A/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CleanDirectoryRecursively", path: "test:/A/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteDirectoryRecursively", path: "test:/A" }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateDirectory", path: "test:/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateDirectory", path: "test:/A/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateDirectory", path: "test:/A/A/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateDirectory", path: "test:/A/A/B" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateDirectory", path: "test:/A/A/B/C" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteDirectory", path: "test:/A/A/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CleanDirectoryRecursively", path: "test:/A/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CleanDirectoryRecursively", path: "test:/A/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteDirectoryRecursively", path: "test:/A" }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateDirectory", path: "test:/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateDirectory", path: "test:/A/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateDirectory", path: "test:/A/A/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateDirectory", path: "test:/A/A/B" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateDirectory", path: "test:/A/A/B/C" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteDirectory", path: "test:/A/A/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CleanDirectoryRecursively", path: "test:/A/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CleanDirectoryRecursively", path: "test:/A/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteDirectoryRecursively", path: "test:/A" }
*/
{
    const char* directoryPathA = "test:/A";
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory(directoryPathA));
    const char* directoryPathAA = "test:/A/A";
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory(directoryPathAA));
    const char* directoryPathAAA = "test:/A/A/A";
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory(directoryPathAAA));
    const char* directoryPathAAB = "test:/A/A/B";
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory(directoryPathAAB));
    const char* directoryPathAABC = "test:/A/A/B/C";
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory(directoryPathAABC));

    // AAA を削除
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteDirectory(directoryPathAAA));
    // AAB と AABC を削除
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CleanDirectoryRecursively(directoryPathAA));
    // 何も削除しない
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CleanDirectoryRecursively(directoryPathAA));
    // A と AA を削除
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteDirectoryRecursively(directoryPathA));
}

TEST_F(OperateEntry, CreateDeleteDirectory_Failuer)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "CreateDirectory", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "DeleteDirectory", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "CleanDirectoryRecursively", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "DeleteDirectoryRecursively", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "CreateDirectory", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "DeleteDirectory", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "CleanDirectoryRecursively", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "DeleteDirectoryRecursively", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000402, handle: 0x0000000000000000, priority: Normal, function: "CreateDirectory", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00001002, handle: 0x0000000000000000, priority: Normal, function: "DeleteDirectory", path: "test:/test1" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "CreateDirectory", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "DeleteDirectory", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "CleanDirectoryRecursively", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "DeleteDirectoryRecursively", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "CreateDirectory", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "DeleteDirectory", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "CleanDirectoryRecursively", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "DeleteDirectoryRecursively", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000402, handle: 0x0000000000000000, priority: Normal, function: "CreateDirectory", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00001002, handle: 0x0000000000000000, priority: Normal, function: "DeleteDirectory", path: "test:/test1" }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateDirectory", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteDirectory", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CleanDirectoryRecursively", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteDirectoryRecursively", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateDirectory", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteDirectory", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CleanDirectoryRecursively", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteDirectoryRecursively", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateDirectory", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00001002, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteDirectory", path: "test:/test1" }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateDirectory", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteDirectory", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CleanDirectoryRecursively", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteDirectoryRecursively", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateDirectory", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteDirectory", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CleanDirectoryRecursively", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteDirectoryRecursively", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateDirectory", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00001002, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteDirectory", path: "test:/test1" }
*/
{
    const char* directoryPathInvalid = "Invalid:/test";
    ASSERT_TRUE(nn::fs::CreateDirectory(directoryPathInvalid).IsFailure());
    ASSERT_TRUE(nn::fs::DeleteDirectory(directoryPathInvalid).IsFailure());
    ASSERT_TRUE(nn::fs::CleanDirectoryRecursively(directoryPathInvalid).IsFailure());
    ASSERT_TRUE(nn::fs::DeleteDirectoryRecursively(directoryPathInvalid).IsFailure());

    const char* directoryPathNotExist = "test:/not/exist";
    ASSERT_TRUE(nn::fs::CreateDirectory(directoryPathNotExist).IsFailure());
    ASSERT_TRUE(nn::fs::DeleteDirectory(directoryPathNotExist).IsFailure());
    ASSERT_TRUE(nn::fs::CleanDirectoryRecursively(directoryPathNotExist).IsFailure());
    ASSERT_TRUE(nn::fs::DeleteDirectoryRecursively(directoryPathNotExist).IsFailure());

    ASSERT_TRUE(nn::fs::CreateDirectory(TestDirectoryPath1).IsFailure());
    ASSERT_TRUE(nn::fs::DeleteDirectory(TestDirectoryPath1).IsFailure());
}

TEST_F(OperateEntry, OpenDirectory_Success)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "OpenDirectory", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadDirectory" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "CloseDirectory" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "OpenDirectory", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadDirectory" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "CloseDirectory" }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenDirectory", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::ReadDirectory" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseDirectory" }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenDirectory", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::ReadDirectory" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseDirectory" }
*/
{
    nn::fs::DirectoryHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(&handle, TestDirectoryPath1, nn::fs::OpenDirectoryMode_All));

    int64_t readNum = 0;
    const int EntryBufferNum = 3;
    nn::fs::DirectoryEntry dirEntry[EntryBufferNum];
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadDirectory(&readNum, dirEntry, handle, EntryBufferNum));
    nn::fs::CloseDirectory(handle);
}

TEST_F(OperateEntry, OpenDirectory_Failure)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "OpenDirectory", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "OpenDirectory", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x002EE202, handle: 0x0000000000000000, priority: Normal, function: "OpenDirectory", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x002F5E02, handle: 0x0000000000000000, priority: Normal, function: "OpenDirectory", path: "test:/test1" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "OpenDirectory", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "OpenDirectory", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x002EE202, handle: 0x0000000000000000, priority: Normal, function: "OpenDirectory", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x002F5E02, handle: 0x0000000000000000, priority: Normal, function: "OpenDirectory", path: "test:/test1" }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenDirectory", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenDirectory", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x002EE202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenDirectory", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x002F5E02, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenDirectory", path: "test:/test1" }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenDirectory", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenDirectory", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x002EE202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenDirectory", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x002F5E02, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenDirectory", path: "test:/test1" }
*/
{
    nn::fs::DirectoryHandle handle;
    ASSERT_TRUE(nn::fs::OpenDirectory(&handle, "test:/not/exist", nn::fs::OpenDirectoryMode_All).IsFailure());
    ASSERT_TRUE(nn::fs::OpenDirectory(&handle, "Invalid:/test", nn::fs::OpenDirectoryMode_All).IsFailure());

    ASSERT_TRUE(nn::fs::OpenDirectory(&handle, TestDirectoryPath1, static_cast<nn::fs::OpenDirectoryMode>(9999)).IsFailure());

    ASSERT_TRUE(nn::fs::OpenDirectory(nullptr, TestDirectoryPath1, 0).IsFailure());
}

TEST_F(OperateEntry, OperateDirectory_Success)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "GetEntryType", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "RenameDirectory", path: "test:/test1/test2", new_path: "test:/tmp" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "RenameDirectory", path: "test:/tmp", new_path: "test:/test1/test2" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "GetEntryType", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "RenameDirectory", path: "test:/test1/test2", new_path: "test:/tmp" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "RenameDirectory", path: "test:/tmp", new_path: "test:/test1/test2" }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::GetEntryType", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameDirectory", path: "test:/test1/test2", new_path: "test:/tmp" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameDirectory", path: "test:/tmp", new_path: "test:/test1/test2" }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::GetEntryType", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameDirectory", path: "test:/test1/test2", new_path: "test:/tmp" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameDirectory", path: "test:/tmp", new_path: "test:/test1/test2" }
*/
{
    nn::fs::DirectoryEntryType entryType;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetEntryType(&entryType, TestDirectoryPath1));
    const char* directoryPathTemp = "test:/tmp";

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::RenameDirectory(TestDirectoryPath2, directoryPathTemp));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::RenameDirectory(directoryPathTemp, TestDirectoryPath2));
}

TEST_F(OperateEntry, OperateDirectory_Failure)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "GetEntryType", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "GetEntryType", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "RenameDirectory", path: "test:/test1/test3", new_path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "RenameDirectory", path: "test:/test1/test3", new_path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "RenameDirectory", path: "test:/not/exist", new_path: "test:/test1/test3" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "RenameDirectory", path: "Invalid:/test", new_path: "test:/test1/test3" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000402, handle: 0x0000000000000000, priority: Normal, function: "RenameDirectory", path: "test:/test1/test2", new_path: "test:/test1/test3" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "GetEntryType", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "GetEntryType", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "RenameDirectory", path: "test:/test1/test3", new_path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "RenameDirectory", path: "test:/test1/test3", new_path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "RenameDirectory", path: "test:/not/exist", new_path: "test:/test1/test3" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "RenameDirectory", path: "Invalid:/test", new_path: "test:/test1/test3" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000402, handle: 0x0000000000000000, priority: Normal, function: "RenameDirectory", path: "test:/test1/test2", new_path: "test:/test1/test3" }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::GetEntryType", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::GetEntryType", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameDirectory", path: "test:/test1/test3", new_path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameDirectory", path: "test:/test1/test3", new_path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameDirectory", path: "test:/not/exist", new_path: "test:/test1/test3" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameDirectory", path: "Invalid:/test", new_path: "test:/test1/test3" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameDirectory", path: "test:/test1/test2", new_path: "test:/test1/test3" }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::GetEntryType", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::GetEntryType", path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameDirectory", path: "test:/test1/test3", new_path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameDirectory", path: "test:/test1/test3", new_path: "Invalid:/test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameDirectory", path: "test:/not/exist", new_path: "test:/test1/test3" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameDirectory", path: "Invalid:/test", new_path: "test:/test1/test3" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameDirectory", path: "test:/test1/test2", new_path: "test:/test1/test3" }
*/
{
    nn::fs::DirectoryEntryType entryType;
    ASSERT_TRUE(nn::fs::GetEntryType(&entryType, "test:/not/exist").IsFailure());
    ASSERT_TRUE(nn::fs::GetEntryType(&entryType, "Invalid:/test").IsFailure());

    ASSERT_TRUE(nn::fs::RenameDirectory(TestDirectoryPath3, "test:/not/exist").IsFailure());
    ASSERT_TRUE(nn::fs::RenameDirectory(TestDirectoryPath3, "Invalid:/test").IsFailure());
    ASSERT_TRUE(nn::fs::RenameDirectory("test:/not/exist", TestDirectoryPath3).IsFailure());
    ASSERT_TRUE(nn::fs::RenameDirectory("Invalid:/test", TestDirectoryPath3).IsFailure());
    ASSERT_TRUE(nn::fs::RenameDirectory(TestDirectoryPath2, TestDirectoryPath3).IsFailure());
}

TEST_F(OperateEntry, CreateDeleteFile_Success)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateFile", path: "test:/test1/A", size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteFile", path: "test:/test1/A" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateFile", path: "test:/test1/A", size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteFile", path: "test:/test1/A" }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateFile", path: "test:/test1/A", size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteFile", path: "test:/test1/A" }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateFile", path: "test:/test1/A", size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteFile", path: "test:/test1/A" }
*/
{
    const char* TestFilePath = "test:/test1/A";

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(TestFilePath, 1024));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteFile(TestFilePath));
}

TEST_F(OperateEntry, CreateDeleteFile_Failure)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "CreateFile", path: "Invalid:/test1/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "DeleteFile", path: "Invalid:/test1/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "CreateFile", path: "test:/not/exist", size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "DeleteFile", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000402, handle: 0x0000000000000000, priority: Normal, function: "CreateFile", path: "test:/test4", size: 1024 }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "CreateFile", path: "Invalid:/test1/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "DeleteFile", path: "Invalid:/test1/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "CreateFile", path: "test:/not/exist", size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "DeleteFile", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000402, handle: 0x0000000000000000, priority: Normal, function: "CreateFile", path: "test:/test4", size: 1024 }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateFile", path: "Invalid:/test1/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteFile", path: "Invalid:/test1/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateFile", path: "test:/not/exist", size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteFile", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateFile", path: "test:/test4", size: 1024 }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateFile", path: "Invalid:/test1/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteFile", path: "Invalid:/test1/A" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateFile", path: "test:/not/exist", size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteFile", path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateFile", path: "test:/test4", size: 1024 }
*/
{
    const char* TestFilePathInvalid = "Invalid:/test1/A";
    ASSERT_TRUE(nn::fs::CreateFile(TestFilePathInvalid, 1024).IsFailure());
    ASSERT_TRUE(nn::fs::DeleteFile(TestFilePathInvalid).IsFailure());

    const char* TestFilePathNotExist = "test:/not/exist";
    ASSERT_TRUE(nn::fs::CreateFile(TestFilePathNotExist, 1024).IsFailure());
    ASSERT_TRUE(nn::fs::DeleteFile(TestFilePathNotExist).IsFailure());

    ASSERT_TRUE(nn::fs::CreateFile(TestFilePath4, 1024).IsFailure());
}

TEST_F(OperateEntry, OpenFile_Success)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "OpenFile", path: "test:/test4", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 0, size: 1024, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 0, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "FlushFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "SetFileSize", size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "CloseFile" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "OpenFile", path: "test:/test4", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 0, size: 1024, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 0, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "FlushFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "SetFileSize", size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "CloseFile" }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "test:/test4", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 0, size: 1024, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 0, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::FlushFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::SetFileSize", size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "test:/test4", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 0, size: 1024, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 0, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::FlushFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::SetFileSize", size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
*/
{
    nn::fs::FileHandle handle;
    char buffer[1024];
    int64_t fileSize = 0;

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handle, TestFilePath4, nn::fs::OpenMode_Write | nn::fs::OpenMode_Read));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(handle, 0, buffer, 1024, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(handle, 0, buffer, 1024, nn::fs::WriteOption::MakeValue(0)));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(handle));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(handle, 0, buffer, 1024));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, handle));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetFileSize(handle, 1024));
    nn::fs::CloseFile(handle);
}

TEST_F(OperateEntry, OpenFile_Failure)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "OpenFile", path: "test:/not/exist", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "OpenFile", path: "Invalid:/test1/test4", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x002F5E02, handle: 0x0000000000000000, priority: Normal, function: "OpenFile", path: "test:/test4", open_mode: 0x1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "OpenFile", path: "test:/test4", open_mode: 0x1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000E02, handle: 0x0000000000000000, priority: Normal, function: "OpenFile", path: "test:/test4", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00307602, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 0, size: 1024, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307602, handle: 0xFEDCBA0987654321, priority: Normal, function: "FlushFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00307602, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1024, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "OpenFile", path: "test:/test4", open_mode: 0x2 }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 1024, size: 1024, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 0, size: 1024, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "CloseFile" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "OpenFile", path: "test:/not/exist", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "OpenFile", path: "Invalid:/test1/test4", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x002F5E02, handle: 0x0000000000000000, priority: Normal, function: "OpenFile", path: "test:/test4", open_mode: 0x1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "OpenFile", path: "test:/test4", open_mode: 0x1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000E02, handle: 0x0000000000000000, priority: Normal, function: "OpenFile", path: "test:/test4", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00307602, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 0, size: 1024, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307602, handle: 0xFEDCBA0987654321, priority: Normal, function: "FlushFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00307602, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1024, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "OpenFile", path: "test:/test4", open_mode: 0x2 }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 1024, size: 1024, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "WriteFile", offset: 0, size: 1024, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "CloseFile" }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenFile", path: "test:/not/exist", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenFile", path: "Invalid:/test1/test4", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x002F5E02, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenFile", path: "test:/test4", open_mode: 0x1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "test:/test4", open_mode: 0x1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000E02, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenFile", path: "test:/test4", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00307602, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 0, size: 1024, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307602, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::FlushFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00307602, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1024, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "test:/test4", open_mode: 0x2 }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 1024, size: 1024, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 0, size: 1024, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenFile", path: "test:/not/exist", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenFile", path: "Invalid:/test1/test4", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x002F5E02, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenFile", path: "test:/test4", open_mode: 0x1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "test:/test4", open_mode: 0x1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000E02, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenFile", path: "test:/test4", open_mode: 0x3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00307602, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 0, size: 1024, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307602, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::FlushFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00307602, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 1024, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::OpenFile", path: "test:/test4", open_mode: 0x2 }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 1024, size: 1024, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::WriteFile", offset: 0, size: 1024, write_option: Flush }
FS_ACCESS: { start: 0, end: 0, result: 0x00307202, handle: 0xFEDCBA0987654321, priority: Normal, function: "ReadFile", offset: 0, size: 1024 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0xFEDCBA0987654321, priority: Normal, function: "nn::fs::CloseFile" }
*/
{
    nn::fs::FileHandle handle;
    nn::fs::FileHandle subHandle;
    char buffer[1024];
    nn::fs::WriteOption invalidOption = {9999};

    ASSERT_TRUE(nn::fs::OpenFile(&handle, "test:/not/exist", nn::fs::OpenMode_Write | nn::fs::OpenMode_Read).IsFailure());
    ASSERT_TRUE(nn::fs::OpenFile(&handle, "Invalid:/test1/test4", nn::fs::OpenMode_Write | nn::fs::OpenMode_Read).IsFailure());
    ASSERT_TRUE(nn::fs::OpenFile(nullptr, TestFilePath4, nn::fs::OpenMode_Read).IsFailure());

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handle, TestFilePath4, nn::fs::OpenMode_Read));
    ASSERT_TRUE(nn::fs::OpenFile(&subHandle, TestFilePath4, nn::fs::OpenMode_Write | nn::fs::OpenMode_Read).IsFailure());
    ASSERT_TRUE(nn::fs::WriteFile(handle, 0, buffer, 1024, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)).IsFailure());
    ASSERT_TRUE(nn::fs::FlushFile(handle).IsFailure());
    ASSERT_TRUE(nn::fs::ReadFile(handle, 1024, buffer, 1024).IsFailure());
    nn::fs::CloseFile(handle);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handle, TestFilePath4, nn::fs::OpenMode_Write));
    ASSERT_TRUE(nn::fs::WriteFile(handle, 1024, buffer, 1024, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)).IsFailure());
    ASSERT_TRUE(nn::fs::WriteFile(handle, 0, buffer, 1024, invalidOption).IsFailure());
    ASSERT_TRUE(nn::fs::ReadFile(handle, 0, buffer, 1024).IsFailure());
    nn::fs::CloseFile(handle);
}

TEST_F(OperateEntry, OperateFile_Success)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "RenameFile", path: "test:/test4", new_path: "test:/temp" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "RenameFile", path: "test:/temp", new_path: "test:/test4" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "RenameFile", path: "test:/test4", new_path: "test:/temp" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "RenameFile", path: "test:/temp", new_path: "test:/test4" }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameFile", path: "test:/test4", new_path: "test:/temp" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameFile", path: "test:/temp", new_path: "test:/test4" }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameFile", path: "test:/test4", new_path: "test:/temp" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameFile", path: "test:/temp", new_path: "test:/test4" }
*/
{
    const char* TestFilePathTemp = "test:/temp";
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::RenameFile(TestFilePath4, TestFilePathTemp));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::RenameFile(TestFilePathTemp, TestFilePath4));
}

TEST_F(OperateEntry, OperateFile_Failure)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "RenameFile", path: "test:/test4", new_path: "Invalid:/test/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "RenameFile", path: "Invalid:/test/test1", new_path: "test:/test4" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "RenameFile", path: "test:/test4", new_path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "RenameFile", path: "test:/not/exist", new_path: "test:/test4" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000402, handle: 0x0000000000000000, priority: Normal, function: "RenameFile", path: "test:/test4", new_path: "test:/test1/test2/test5" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "RenameFile", path: "test:/test4", new_path: "Invalid:/test/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "RenameFile", path: "Invalid:/test/test1", new_path: "test:/test4" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "RenameFile", path: "test:/test4", new_path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "RenameFile", path: "test:/not/exist", new_path: "test:/test4" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000402, handle: 0x0000000000000000, priority: Normal, function: "RenameFile", path: "test:/test4", new_path: "test:/test1/test2/test5" }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameFile", path: "test:/test4", new_path: "Invalid:/test/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameFile", path: "Invalid:/test/test1", new_path: "test:/test4" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameFile", path: "test:/test4", new_path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameFile", path: "test:/not/exist", new_path: "test:/test4" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameFile", path: "test:/test4", new_path: "test:/test1/test2/test5" }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameFile", path: "test:/test4", new_path: "Invalid:/test/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x0035F202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameFile", path: "Invalid:/test/test1", new_path: "test:/test4" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameFile", path: "test:/test4", new_path: "test:/not/exist" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameFile", path: "test:/not/exist", new_path: "test:/test4" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::RenameFile", path: "test:/test4", new_path: "test:/test1/test2/test5" }
*/
{
    const char* TestFilePathInvalid = "Invalid:/test/test1";
    ASSERT_TRUE(nn::fs::RenameFile(TestFilePath4, TestFilePathInvalid).IsFailure());
    ASSERT_TRUE(nn::fs::RenameFile(TestFilePathInvalid, TestFilePath4).IsFailure());

    const char* TestFilePathNotExist = "test:/not/exist";
    ASSERT_TRUE(nn::fs::RenameFile(TestFilePath4, TestFilePathNotExist).IsFailure());
    ASSERT_TRUE(nn::fs::RenameFile(TestFilePathNotExist, TestFilePath4).IsFailure());

    ASSERT_TRUE(nn::fs::RenameFile(TestFilePath4, TestFilePath5).IsFailure());
}

void OperateEntry::TearDownTestCase()
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteDirectoryRecursively", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteDirectoryRecursively", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteDirectoryRecursively", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteDirectoryRecursively", path: "test:/test1" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
*/
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteDirectoryRecursively(TestDirectoryPath1));
    nn::fs::Unmount("test");
}
