﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <new>
#include <memory>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/fs/fs_Content.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/fs/fs_AccessFailureDetection.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_IEventNotifier.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Thread.h>
#include <nn/svc/svc_Base.h>
#include <nn/util/util_ScopeExit.h>

using namespace nn;
using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nnt::fs::util;

namespace {

    struct Context
    {
        int64_t count;
        std::atomic<uint64_t> activeCount;
        bool useHtml;
    };

    const int ThreadCount = 1;
    const int ThreadStackSize = 4 * 1024 * 1024;

    os::ThreadType g_FsThread[ThreadCount];
    NN_OS_ALIGNAS_THREAD_STACK char g_FsThreadStack[ThreadCount][ThreadStackSize];
    Context g_FsThreadContext[ThreadCount];

    std::atomic<bool> g_Exit(false);

    Bit64 g_CurrentProcessId;

    void WaitAttachDetach(bool isWaitAttach)
    {
        if (IsGameCardInserted() == isWaitAttach)
        {
            return;
        }

        std::unique_ptr<IEventNotifier> deviceDetectionEventNotifier;
        NNT_ASSERT_RESULT_SUCCESS(OpenGameCardDetectionEventNotifier(&deviceDetectionEventNotifier));

        os::SystemEventType deviceDetectionEvent;
        NNT_ASSERT_RESULT_SUCCESS(deviceDetectionEventNotifier->BindEvent(&deviceDetectionEvent, os::EventClearMode_ManualClear));

        while (NN_STATIC_CONDITION(true))
        {
            NN_LOG("Waiting %s...\n", isWaitAttach ? "attach" : "detach");
            os::WaitSystemEvent(&deviceDetectionEvent);
            os::ClearSystemEvent(&deviceDetectionEvent);

            bool isGameCardInserted = IsGameCardInserted();
            if (isGameCardInserted == isWaitAttach)
            {
                // TORIAEZU: ncm のデータベース構築待ち
                if (isWaitAttach)
                {
                    os::SleepThread(TimeSpan::FromMilliSeconds(2000));
                }
                NN_LOG("Detect %s.\n", isGameCardInserted ? "attach" : "detach");
                break;
            }
        }
    }

    void FileSystemAccess(void* arg)
    {
        Result result;

        auto pContext = reinterpret_cast<Context*>(arg);
        auto startCount = pContext->count;

        const char* FilePath = pContext->useHtml ? "rom:/html-document/1mb.bin" : "rom:/1mb.bin";

        FileHandle file;
        NNT_EXPECT_RESULT_SUCCESS(OpenFile(&file, FilePath, OpenMode_Read));

        const size_t BufferSize = 32 * 1024;
        char buffer[BufferSize];
        size_t readSize = 0;

        int count = 0;
        int64_t offset = startCount * sizeof(buffer);

        while (NN_STATIC_CONDITION(true))
        {
            if (g_Exit.load())
            {
                break;
            }

            if (count++ % (32 / ThreadCount) == 0)
            {
                offset = startCount * sizeof(buffer);
            }

            // NN_LOG("#%8d Read: ofs = %lld\n", count, offset);

            auto result = ReadFile(&readSize, file, offset, buffer, sizeof(buffer));
            if (result.IsFailure())
            {
                // エラーが返されたらスレッドを終了
                NN_LOG("exit thread by result = 0x%08x\n", result.GetInnerValueForDebug());
                break;
            }

            if (!IsFilledWith32BitCount(buffer, sizeof(buffer), offset))
            {
                // データの検証に失敗したらスレッドを終了
                NN_LOG("exit thread by wrong data detection.\n");
                break;
            }

            pContext->activeCount++;
            offset += sizeof(buffer);
        }

        CloseFile(file);
    }

    void StartFileAccessThread(bool useHtml = false)
    {
        for (int i = 0; i < ThreadCount; i++)
        {
            g_FsThreadContext[i].count = i * (32 / ThreadCount);
            g_FsThreadContext[i].activeCount.store(0);
            g_FsThreadContext[i].useHtml = useHtml;

            NNT_ASSERT_RESULT_SUCCESS(os::CreateThread(&g_FsThread[i], FileSystemAccess, &g_FsThreadContext[i], g_FsThreadStack[i], ThreadStackSize, os::DefaultThreadPriority));
            os::StartThread(&g_FsThread[i]);
        }
    }

    void ExitFileAccessThread()
    {
        g_Exit.store(true);
        for (int i = 0; i < ThreadCount; i++)
        {
            os::DestroyThread(&g_FsThread[i]);
        }
        g_Exit.store(false);
    }

    bool CheckFsThreadAlive()
    {
        for (int i = 0; i < ThreadCount; i++)
        {
            g_FsThreadContext[i].activeCount.store(0);
        }
        os::SleepThread(TimeSpan::FromMilliSeconds(3000));
        for (int i = 0; i < ThreadCount; i++)
        {
            if (g_FsThreadContext[i].activeCount.load() == 0)
            {
                return false;
            }
        }
        return true;
    }

    bool CheckFsThreadAllExited()
    {
        for (int i = 0; i < ThreadCount; i++)
        {
            if (g_FsThread[i]._state != os::ThreadType::State::State_Exited)
            {
                return false;
            }
        }
        return true;
    }

}

TEST(AccessFailureDetection, SelfAbandon)
{
    WaitAttachDetach(true);

    size_t cacheSize = 0;
    QueryMountRomCacheSize(&cacheSize);

    std::unique_ptr<char[]> cacheBuffer(new char[cacheSize]);
    NN_ABORT_UNLESS_NOT_NULL(cacheBuffer);
    NNT_ASSERT_RESULT_SUCCESS(MountRom("rom", cacheBuffer.get(), cacheSize));

    os::SystemEvent event;
    std::unique_ptr<IEventNotifier> eventNotifier;

    NNT_ASSERT_RESULT_SUCCESS(OpenAccessFailureDetectionEventNotifier(&eventNotifier, g_CurrentProcessId));
    NNT_ASSERT_RESULT_SUCCESS(eventNotifier->BindEvent(event.GetBase(), os::EventClearMode_ManualClear));

    StartFileAccessThread();

    // Abandon は非エラー時にも実行可能。ただし eventNotifier が無効化される
    AbandonAccessFailure(g_CurrentProcessId);

    eventNotifier.reset();
    NNT_ASSERT_RESULT_SUCCESS(OpenAccessFailureDetectionEventNotifier(&eventNotifier, g_CurrentProcessId));
    NNT_ASSERT_RESULT_SUCCESS(eventNotifier->BindEvent(event.GetBase(), os::EventClearMode_ManualClear));

    WaitAttachDetach(false);

    // カード挿抜によりエラー検知
    ASSERT_TRUE(event.TimedWait(TimeSpan::FromMilliSeconds(10000)));
    event.Clear();

    // FS アクセススレッドがブロックされている
    EXPECT_FALSE(CheckFsThreadAlive());
    EXPECT_FALSE(CheckFsThreadAllExited());

    ResolveAccessFailure(g_CurrentProcessId);

    // カードが挿入されていないため解決できず、再度通知される
    ASSERT_TRUE(event.TimedWait(TimeSpan::FromMilliSeconds(1000)));
    event.Clear();

    AbandonAccessFailure(g_CurrentProcessId);

    // 解決を取りやめたため、FS アクセススレッドにエラーが返され、スレッドが終了している
    EXPECT_FALSE(CheckFsThreadAlive());
    EXPECT_TRUE(CheckFsThreadAllExited());

    ExitFileAccessThread();

    Unmount("rom");
}

TEST(AccessFailureDetection, SelfResolve)
{
    WaitAttachDetach(true);

    size_t cacheSize = 0;
    QueryMountRomCacheSize(&cacheSize);

    std::unique_ptr<char[]> cacheBuffer(new char[cacheSize]);
    NN_ABORT_UNLESS_NOT_NULL(cacheBuffer);
    NNT_ASSERT_RESULT_SUCCESS(MountRom("rom", cacheBuffer.get(), cacheSize));

    std::unique_ptr<IEventNotifier> eventNotifier;
    os::SystemEvent event;

    NNT_ASSERT_RESULT_SUCCESS(OpenAccessFailureDetectionEventNotifier(&eventNotifier, g_CurrentProcessId));
    NNT_ASSERT_RESULT_SUCCESS(eventNotifier->BindEvent(event.GetBase(), os::EventClearMode_ManualClear));

    StartFileAccessThread();

    // Resolve は非エラー時にも実行可能
    ResolveAccessFailure(g_CurrentProcessId);

    WaitAttachDetach(false);

    // カード挿抜によりエラー検知
    ASSERT_TRUE(event.TimedWait(TimeSpan::FromMilliSeconds(10000)));
    event.Clear();

    // FS アクセススレッドがブロックされている
    EXPECT_FALSE(CheckFsThreadAlive());
    EXPECT_FALSE(CheckFsThreadAllExited());

    ResolveAccessFailure(g_CurrentProcessId);

    // カードが挿入されていないため解決できず、再度通知される
    ASSERT_TRUE(event.TimedWait(TimeSpan::FromMilliSeconds(1000)));
    event.Clear();

    WaitAttachDetach(true);

    ResolveAccessFailure(g_CurrentProcessId);

    ASSERT_FALSE(event.TimedWait(TimeSpan::FromMilliSeconds(1000)));

    // アクセスエラーが解消され、FS アクセススレッドが再開している
    EXPECT_TRUE(CheckFsThreadAlive());

    ExitFileAccessThread();

    Unmount("rom");
}

TEST(AccessFailureDetection, NoDetection)
{
    WaitAttachDetach(true);

    size_t cacheSize = 0;
    QueryMountRomCacheSize(&cacheSize);

    std::unique_ptr<char[]> cacheBuffer(new char[cacheSize]);
    NN_ABORT_UNLESS_NOT_NULL(cacheBuffer);
    NNT_ASSERT_RESULT_SUCCESS(MountRom("rom", cacheBuffer.get(), cacheSize));

    StartFileAccessThread();

    // Resolve/Abandon はイベント監視していなくても実行可能
    ResolveAccessFailure(g_CurrentProcessId);
    AbandonAccessFailure(g_CurrentProcessId);

    WaitAttachDetach(false);

    // イベント監視されていなかったため、FS アクセススレッドにエラーが返され、スレッドが終了している
    EXPECT_FALSE(CheckFsThreadAlive());
    EXPECT_TRUE(CheckFsThreadAllExited());

    ExitFileAccessThread();

    Unmount("rom");
}

TEST(AccessFailureDetection, SelfResolveByResolver)
{
    WaitAttachDetach(true);

    size_t cacheSize = 0;
    QueryMountRomCacheSize(&cacheSize);

    std::unique_ptr<char[]> cacheBuffer(new char[cacheSize]);
    NN_ABORT_UNLESS_NOT_NULL(cacheBuffer);
    NNT_ASSERT_RESULT_SUCCESS(MountRom("rom", cacheBuffer.get(), cacheSize));

    std::unique_ptr<AccessFailureResolver> resolver;
    os::SystemEvent event;

    NNT_ASSERT_RESULT_SUCCESS(OpenAccessFailureResolver(&resolver, g_CurrentProcessId));
    NNT_ASSERT_RESULT_SUCCESS(GetAccessFailureDetectionEvent(&event));
    event.Clear();

    StartFileAccessThread();

    // Resolve は非エラー時にも実行可能
    resolver->Resolve();

    EXPECT_FALSE(resolver->IsAccessFailureDetected());

    WaitAttachDetach(false);

    // カード挿抜によりエラー検知
    ASSERT_TRUE(event.TimedWait(TimeSpan::FromMilliSeconds(10000)));
    event.Clear();

    EXPECT_TRUE(resolver->IsAccessFailureDetected());
    EXPECT_FALSE(IsAccessFailureDetected(g_CurrentProcessId + 1));

    // FS アクセススレッドがブロックされている
    EXPECT_FALSE(CheckFsThreadAlive());
    EXPECT_FALSE(CheckFsThreadAllExited());

    resolver->Resolve();

    // カードが挿入されていないため解決できず、再度通知される
    ASSERT_TRUE(event.TimedWait(TimeSpan::FromMilliSeconds(1000)));
    event.Clear();

    EXPECT_TRUE(resolver->IsAccessFailureDetected());
    EXPECT_FALSE(IsAccessFailureDetected(g_CurrentProcessId + 1));

    WaitAttachDetach(true);

    resolver->Resolve();

    ASSERT_FALSE(event.TimedWait(TimeSpan::FromMilliSeconds(1000)));

    EXPECT_FALSE(resolver->IsAccessFailureDetected());

    // アクセスエラーが解消され、FS アクセススレッドが再開している
    EXPECT_TRUE(CheckFsThreadAlive());

    ExitFileAccessThread();

    Unmount("rom");
}

TEST(AccessFailureDetection, SelfAbandonByResolver)
{
    WaitAttachDetach(true);

    size_t cacheSize = 0;
    QueryMountRomCacheSize(&cacheSize);

    std::unique_ptr<char[]> cacheBuffer(new char[cacheSize]);
    NN_ABORT_UNLESS_NOT_NULL(cacheBuffer);
    NNT_ASSERT_RESULT_SUCCESS(MountRom("rom", cacheBuffer.get(), cacheSize));

    std::unique_ptr<AccessFailureResolver> resolver;
    os::SystemEvent event;

    NNT_ASSERT_RESULT_SUCCESS(OpenAccessFailureResolver(&resolver, g_CurrentProcessId));
    NNT_ASSERT_RESULT_SUCCESS(GetAccessFailureDetectionEvent(&event));

    StartFileAccessThread();

    // Abandon は非エラー時にも実行可能。
    resolver.reset();

    NNT_ASSERT_RESULT_SUCCESS(OpenAccessFailureResolver(&resolver, g_CurrentProcessId));

    WaitAttachDetach(false);

    // カード挿抜によりエラー検知
    ASSERT_TRUE(event.TimedWait(TimeSpan::FromMilliSeconds(10000)));
    event.Clear();

    // FS アクセススレッドがブロックされている
    EXPECT_FALSE(CheckFsThreadAlive());
    EXPECT_FALSE(CheckFsThreadAllExited());

    resolver->Resolve();

    // カードが挿入されていないため解決できず、再度通知される
    ASSERT_TRUE(event.TimedWait(TimeSpan::FromMilliSeconds(1000)));
    event.Clear();

    resolver.reset();

    // 解決を取りやめたため、FS アクセススレッドにエラーが返され、スレッドが終了している
    EXPECT_FALSE(CheckFsThreadAlive());
    EXPECT_TRUE(CheckFsThreadAllExited());

    ExitFileAccessThread();

    Unmount("rom");
}

TEST(AccessFailureDetection, SelfResolveByResolverForHtmlDocument)
{
    WaitAttachDetach(true);

    NNT_ASSERT_RESULT_SUCCESS(MountContent("rom", { 0x0005000C10000000 }, fs::ContentType_Manual));

    std::unique_ptr<AccessFailureResolver> resolver;
    os::SystemEvent event;

    NNT_ASSERT_RESULT_SUCCESS(OpenAccessFailureResolver(&resolver, g_CurrentProcessId));
    NNT_ASSERT_RESULT_SUCCESS(GetAccessFailureDetectionEvent(&event));
    event.Clear();

    StartFileAccessThread(true);

    // Resolve は非エラー時にも実行可能
    resolver->Resolve();

    EXPECT_FALSE(resolver->IsAccessFailureDetected());

    WaitAttachDetach(false);

    // カード挿抜によりエラー検知
    ASSERT_TRUE(event.TimedWait(TimeSpan::FromMilliSeconds(10000)));
    event.Clear();

    EXPECT_TRUE(resolver->IsAccessFailureDetected());
    EXPECT_FALSE(IsAccessFailureDetected(g_CurrentProcessId + 1));

    // FS アクセススレッドがブロックされている
    EXPECT_FALSE(CheckFsThreadAlive());
    EXPECT_FALSE(CheckFsThreadAllExited());

    resolver->Resolve();

    // カードが挿入されていないため解決できず、再度通知される
    ASSERT_TRUE(event.TimedWait(TimeSpan::FromMilliSeconds(1000)));
    event.Clear();

    EXPECT_TRUE(resolver->IsAccessFailureDetected());
    EXPECT_FALSE(IsAccessFailureDetected(g_CurrentProcessId + 1));

    WaitAttachDetach(true);

    resolver->Resolve();

    ASSERT_FALSE(event.TimedWait(TimeSpan::FromMilliSeconds(1000)));

    EXPECT_FALSE(resolver->IsAccessFailureDetected());

    // アクセスエラーが解消され、FS アクセススレッドが再開している
    EXPECT_TRUE(CheckFsThreadAlive());

    ExitFileAccessThread();

    Unmount("rom");
}

#if 0
TEST(AccessFailureDetection, CreateTestData)
{
    NNT_ASSERT_RESULT_SUCCESS(MountHost("host", "./"));
    NNT_ASSERT_RESULT_SUCCESS(CreateFileWith32BitCount("host:/1mb.bin", 1024 * 1024, 0));
    Unmount("host");
}
#endif

extern "C" void nnMain()
{
    int     argc = os::GetHostArgc();
    char**  argv = os::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    svc::GetProcessId(&g_CurrentProcessId, svc::PSEUDO_HANDLE_CURRENT_PROCESS);

    SetEnabledAutoAbort(false);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
