﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/crypto.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsApi/testFs_Unit_Api_SaveData.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h.>

#include <nn/fssystem/fs_SaveDataFileSystem.h>
#include <nn/fat/fat_FatFileSystem.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>

// #define USE_FILE

using namespace nn::fssystem;

#include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>

namespace {
    const int HeapSize = 10 * 1024 * 1024;
    char g_HeapStack[HeapSize];

    // SaveData Format Parameters
    const int64_t AvailableSize = 32 * 1024 * 1024;
    const int64_t ReservedSize = 32 * 1024 * 1024;
    const size_t BlockSize = 16 * 1024;
    const uint32_t CountBlock = AvailableSize / BlockSize;
    const uint32_t CountReservedBlock = ReservedSize / BlockSize;
    const int CountExpandMax = 1;
    int64_t g_TotalSize = 0;
    save::HierarchicalDuplexStorageControlArea::InputParam g_ParamDuplex;
    save::HierarchicalIntegrityVerificationStorageControlArea::InputParam g_ParamIntegrity;

#ifdef USE_FILE
    // File Storage version
    nn::fs::FileHandle baseFileHandle;
#else
    // Memory Storage version
    std::unique_ptr<char[]> storageBuffer;
#endif

    std::unique_ptr<nn::fs::IStorage> baseStorage;

    const auto MaxCacheCount = 1024;
    const auto SizeBlock = 16 * 1024;
    const auto BufferManagerHeapSize = 8 * 1024 * 1024;
    const size_t BufferManagerWorkBufferSize = 128 * 1024;
    NN_ALIGNAS(4096) char g_BufferManagerHeap[BufferManagerHeapSize];
    char g_BufferManagerWorkBuffer[BufferManagerWorkBufferSize];
    nn::fssystem::FileSystemBufferManager g_BufferManager;

    void SetUpBaseStorage() NN_NOEXCEPT
    {
#ifdef USE_FILE
        baseStorage.reset(new FileHandleStorage(baseFileHandle));
#else
        ASSERT_NE(storageBuffer, nullptr);
        memset(storageBuffer.get(), 0xCD, (size_t)g_TotalSize);

        baseStorage.reset(new nn::fs::MemoryStorage(storageBuffer.get(), g_TotalSize));
#endif

        g_BufferManager.Initialize(
            MaxCacheCount,
            reinterpret_cast<uintptr_t>(g_BufferManagerHeap),
            sizeof(g_BufferManagerHeap),
            SizeBlock,
            g_BufferManagerWorkBuffer,
            BufferManagerWorkBufferSize);

        nn::fs::SaveDataHashSalt salt;
        // フォーマット（実装は fssrv_FileSystemProxyImpl.cpp を参照）
        nn::fs::SubStorage storage(baseStorage.get(), 0, g_TotalSize);
        ASSERT_TRUE(save::JournalIntegritySaveDataFileSystemDriver::Format(storage, BlockSize, CountExpandMax, g_ParamDuplex, g_ParamIntegrity, CountBlock, CountReservedBlock, &g_BufferManager, nnt::fs::util::GetMacGenerator(), salt).IsSuccess());
    }

    void TearDownBaseStorage() NN_NOEXCEPT
    {
        baseStorage.reset();

#ifdef USE_FILE
        nn::fs::FlushFile(baseFileHandle);
        nn::fs::CloseFile(baseFileHandle);
        nn::fs::DeleteFile(baseFilePath.c_str());
        nn::fs::DeleteDirectory(tempRootDirPath.c_str());
        nn::fs::UnmountHostRoot();
#endif

        g_BufferManager.Finalize();
    }

    void SetUpSaveDataFs(std::unique_ptr<nn::fssystem::SaveDataFileSystem>* pOutValue)
    {
        // SaveDataFs を生成
        std::unique_ptr<nn::fssystem::SaveDataFileSystem> saveDataFs(new nn::fssystem::SaveDataFileSystem());
        ASSERT_NE(saveDataFs, nullptr);

        // 初期化・マウント
        ASSERT_TRUE(saveDataFs->Initialize(baseStorage.get(), &g_BufferManager, nnt::fs::util::GetMacGenerator(), nnt::fs::util::SaveDataMinimumVersion).IsSuccess());

        *pOutValue = std::move(saveDataFs);
    }
}

namespace nnt { namespace fs { namespace api {

    void GetTestFileSystemInfo(std::unique_ptr<TestFileSystemInfo>* outValue, int index) NN_NOEXCEPT
    {
        NN_UNUSED(index);

        std::unique_ptr<nn::fssystem::SaveDataFileSystem> saveDataFs;
        SetUpSaveDataFs(&saveDataFs);

        std::unique_ptr<TestFileSystemInfo> info(new TestFileSystemInfo(FsApiTestType::Unit));

        ISaveDataExtraDataAccessor* pSaveDataExtraDataAccessor = saveDataFs.get();
        info->fileSystem = std::unique_ptr<ITestFileSystem>(new FsApiUnitTestSaveDataFileSystem(std::move(saveDataFs), pSaveDataExtraDataAccessor));
        info->type = FileSystemType_SaveDataFileSystem;
        info->rootDirPath = "";

        auto& attribute = info->attribute;

        attribute.directoryNameLengthMax = dbm::MaxDirectoryNameLength;
        attribute.fileNameLengthMax      = dbm::MaxFileNameLength;
        // EntryNameLengthMax 以上のパスが fs_SavePath で弾かれるので設定できるのは EntryNameLengthMax - 1 まで
        attribute.directoryPathLengthMax = nn::fs::EntryNameLengthMax - 1;
        attribute.filePathLengthMax      = nn::fs::EntryNameLengthMax - 1;
        attribute.fileSizeMax = 0xFFFFFFFF;
        attribute.storageSize = g_TotalSize;

        attribute.fileOpenMax      = 30;
        attribute.directoryOpenMax = 20;

        attribute.isSupportedMultiBytePath = true;
        attribute.isSupportedGetFreeSpaceSize = true;
        attribute.isConcatenationFileSystem = false;
        attribute.isSaveFileSystem = true;
        attribute.isSupportedQueryRange = true;

#if !defined(USE_FILE)
        attribute.isMemoryStorage = true;
#endif // !defined(USE_FILE)

        attribute.fileSizeAlignment = 1;

        *outValue = std::move(info);
    }

    void GetTestSaveDataExtraDataInfo(std::unique_ptr<TestSaveDataExtraDataInfo>* outValue, ITestFileSystem* pFileSystem) NN_NOEXCEPT
    {
        std::unique_ptr<TestSaveDataExtraDataInfo> info(new TestSaveDataExtraDataInfo);
        info->saveDataExtraDataAccessor = reinterpret_cast<FsApiUnitTestSaveDataFileSystem*>(pFileSystem)->GetSaveDataExtraDataAccessor();

        info->staticSaveDataExtraDataReader = [](void* pExtraData, size_t sizeExtraData) NN_NOEXCEPT -> nn::Result
        {
            NN_ABORT_UNLESS_LESS_EQUAL(sizeExtraData, sizeof(nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData));

            int64_t storageSize;
            NN_RESULT_DO(baseStorage->GetSize(&storageSize));

            nn::fs::SubStorage subStorage(baseStorage.get(), 0, storageSize);
            nn::fssystem::save::JournalIntegritySaveDataFileSystem::ExtraData extraData;
            NN_RESULT_DO(nn::fssystem::save::JournalIntegritySaveDataFileSystem::ReadExtraData(
                &extraData,
                subStorage,
                &g_BufferManager,
                nnt::fs::util::GetMacGenerator(),
                nnt::fs::util::SaveDataMinimumVersion
            ));

            std::memcpy(pExtraData, &extraData, sizeExtraData);

            NN_RESULT_SUCCESS;

        };

        *outValue = std::move(info);
    }

}}}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nnt::fs::util::InitializeSaveDataTestHelper();

    nnt::fs::api::LoadAllTests();
    nnt::fs::api::LoadAllSaveDataExtraDataTests();

    nnt::fs::util::InitializeTestLibraryHeap(g_HeapStack, HeapSize);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    save::HierarchicalIntegrityVerificationStorage::SetGenerateRandomFunction(nnt::fs::util::GenerateRandomForSaveDataTest);

    nn::fs::SetEnabledAutoAbort(false);

    nnt::fs::util::String tempRootDirPath;
    nnt::fs::util::String baseFilePath;

    // サイズ計算
    {
        g_ParamDuplex.sizeBlockLevel[0] = BlockSize;
        g_ParamDuplex.sizeBlockLevel[1] = BlockSize;

        g_ParamIntegrity.sizeBlockLevel[0] = BlockSize;
        g_ParamIntegrity.sizeBlockLevel[1] = BlockSize;
        g_ParamIntegrity.sizeBlockLevel[2] = BlockSize;
        g_ParamIntegrity.sizeBlockLevel[3] = BlockSize;

        NNT_ASSERT_RESULT_SUCCESS(save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(&g_TotalSize, BlockSize, CountExpandMax, g_ParamDuplex, g_ParamIntegrity, CountBlock, CountReservedBlock));
    }

#ifdef USE_FILE
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHostRoot());
    tempRootDirPath = nnt::fs::util::CreateUniqueTemporaryDirectory();
    baseFilePath = tempRootDirPath + "/FatBaseFile.bin";
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(baseFilePath.c_str(), g_TotalSize));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&baseFileHandle, baseFilePath.c_str(), nn::fs::OpenMode_Read | nn::fs::OpenMode_Write));
#else
    storageBuffer.reset(new char[(size_t)g_TotalSize]);
    ASSERT_NE(storageBuffer, nullptr);
#endif

    static const size_t BufferPoolSize = 1024 * 1024;
    static NN_ALIGNAS(4096) char g_BufferPool[BufferPoolSize];
    static const size_t WorkBufferSize = nn::fssystem::BufferPoolWorkSize;
    static NN_ALIGNAS(8) char g_WorkBuffer[WorkBufferSize];
    nn::fssystem::InitializeBufferPool(g_BufferPool, BufferPoolSize, g_WorkBuffer, WorkBufferSize);

    SetUpBaseStorage();

    auto ret = RUN_ALL_TESTS();

    TearDownBaseStorage();

    nnt::fs::util::FinalizeSaveDataTestHelper();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(ret);
}
