﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/fs/fs_FileSystemPrivate.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsApi/testFs_Unit_Api.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/fsUtil/testFs_util_adapter.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/fat/fat_FatFileSystem.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fs_FileStorage.h>

#include <nn/fssystem/fs_ConcatenationFileSystem.h>

#define USE_FILE
#define SPECIFY_BIGFILE_FLAG_ALWAYS

using namespace nn::fssystem;
using namespace nn::fs;
using namespace nn::fs::fsa;

#include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>

namespace {
    const int HeapSize = 512 * 1024;
    char g_HeapStack[HeapSize];

#ifdef USE_FILE
    // File Storage version
    const auto StorageSize = 64ULL * 1024 * 1024 * 1024 + 512 * 1024 * 1024;
    nnt::fs::util::TemporaryHostDirectory hostDirectory;
    nn::fs::FileHandle baseFileHandle;
#else
    // Memory Storage version
    const auto StorageSize = 8 * 1024 * 1024;
    std::unique_ptr<char[]> storageBuffer;
#endif


    class BigFileFlagFileSystem : public nnt::fs::util::ThruFs
    {
    public:
        explicit BigFileFlagFileSystem(IFileSystem* pBaseFileSystem) NN_NOEXCEPT
            : ThruFs(pBaseFileSystem)
        {
        }
        virtual Result DoCreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(option);
            return ThruFs::DoCreateFile(path, size, CreateFileOptionFlag_BigFile);
        }
    };



    std::unique_ptr<char[]> g_CacheBuffer;
    std::unique_ptr<IStorage> g_BaseStorage;
    std::unique_ptr<IFileSystem> g_CatFs;

    void SetUpCatFs(std::unique_ptr<IFileSystem>* pOutValue)
    {
        // FatFs を生成
        std::unique_ptr<nn::fat::FatFileSystem> fatFs(new nn::fat::FatFileSystem());
        ASSERT_NE(fatFs, nullptr);

        // キャッシュバッファ・IStorage を割り当て
        ASSERT_NE(g_CacheBuffer, nullptr);
        size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
        NNT_ASSERT_RESULT_SUCCESS(fatFs->Initialize(g_BaseStorage.get(), g_CacheBuffer.get(), cacheBufferSize));

        static bool s_IsFormated = false;
        if( !s_IsFormated )
        {
            // FAT にフォーマット
            NNT_ASSERT_RESULT_SUCCESS(fatFs->Format());
            s_IsFormated = true;
        }

        // fatfs 内部のマウント処理
        NNT_ASSERT_RESULT_SUCCESS(fatFs->Mount());

        // ConcatenationFs を被せる
        g_CatFs.reset(new ConcatenationFileSystem(std::move(fatFs)));

#ifdef SPECIFY_BIGFILE_FLAG_ALWAYS
        // 常に BigFileFlag を指定するレイヤを被せる
        std::unique_ptr<IFileSystem> bigFileFlagFs(new BigFileFlagFileSystem(g_CatFs.get()));
        *pOutValue = std::move(std::move(bigFileFlagFs));
#else
        *pOutValue = std::move(g_CatFs);
#endif
    }

}

namespace nnt { namespace fs { namespace api {

    void GetTestFileSystemInfo(std::unique_ptr<TestFileSystemInfo>* outValue, int index) NN_NOEXCEPT
    {
        NN_UNUSED(index);

        std::unique_ptr<IFileSystem> catFs;
        SetUpCatFs(&catFs);

        std::unique_ptr<TestFileSystemInfo> info(new TestFileSystemInfo(FsApiTestType::Unit));

        info->fileSystem = std::unique_ptr<ITestFileSystem>(new FsApiUnitTestFileSystem(std::move(catFs)));
        info->type = FileSystemType_FatFileSystem;
        info->rootDirPath = "";

        auto& attribute = info->attribute;

        static const int FatNameLengthMax  = 255;     // prfile2 の LONG_NAME_CHARS
        static const int FatPathLengthMax  = 260 - 1; // prfile2 の LONG_NAME_PATH_CHARS
        static const int DriveLetterLength = 2;       // FatFileSystem で付与されるドライブレターの文字数 ("A:")
        static const int InnerFileName     = 3;       // ConcatenationFileSystem 内部で使用されるファイル名 ("/XX")
        attribute.directoryNameLengthMax = FatNameLengthMax;
        attribute.fileNameLengthMax      = FatNameLengthMax;
        attribute.directoryPathLengthMax = FatPathLengthMax - DriveLetterLength;
        attribute.filePathLengthMax      = FatPathLengthMax - DriveLetterLength - InnerFileName;

#ifdef SPECIFY_BIGFILE_FLAG_ALWAYS
        attribute.fileSizeMax = 0x2000000000ULL; // 128GB
#else
        attribute.fileSizeMax = 0xFFFFFFFFULL;
#endif
        attribute.storageSize = StorageSize;

        attribute.fileOpenMax      = 30;
        attribute.directoryOpenMax = 20 - 1; // GetEntryCount() 内部で使うため目減り

        attribute.isSupportedMultiBytePath = false;
        attribute.isSupportedGetFreeSpaceSize = false;
        attribute.isConcatenationFileSystem = true;
        attribute.isSupportedQueryRange = true;
        attribute.isRootExcessFailureResultPathNotFound = true;

#if !defined(USE_FILE)
        attribute.isMemoryStorage = true;
#endif // !defined(USE_FILE)

        attribute.fileSizeAlignment = 1;

        *outValue = std::move(info);
    }
}}}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nnt::fs::api::LoadAllTests();

    nnt::fs::util::InitializeTestLibraryHeap(g_HeapStack, HeapSize);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();

    nn::fs::SetEnabledAutoAbort(false);

    nnt::fs::util::String tempRootDirPath;
    nnt::fs::util::String baseFilePath;


#ifdef USE_FILE
    hostDirectory.Create();
    tempRootDirPath = hostDirectory.GetPath();
    baseFilePath = tempRootDirPath + "/FatBaseFile.bin";
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHostRoot());
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(baseFilePath.c_str(), StorageSize));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&baseFileHandle, baseFilePath.c_str(), nn::fs::OpenMode_Read | nn::fs::OpenMode_Write));
    g_BaseStorage.reset(new FileHandleStorage(baseFileHandle));
#else
    storageBuffer.reset(new char[StorageSize]);
    ASSERT_NE(storageBuffer, nullptr);
    memset(storageBuffer.get(), 0xCD, StorageSize);
    g_BaseStorage.reset(new MemoryStorage(storageBuffer.get(), StorageSize));
#endif

    size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
    g_CacheBuffer.reset(new char[cacheBufferSize]);


    auto ret = RUN_ALL_TESTS();

    g_BaseStorage.reset();
    g_CatFs.reset();

#ifdef USE_FILE
    nn::fs::FlushFile(baseFileHandle);
    nn::fs::CloseFile(baseFileHandle);
    nn::fs::DeleteFile(baseFilePath.c_str());
    nn::fs::UnmountHostRoot();
    hostDirectory.Delete();
#endif

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(ret);
}
