﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/base/testBase_Exit.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsApi/testFs_Unit_Api.h>
#include <nnt/nnt_Argument.h>

#include <nn/crypto.h>
#include <nn/fat/fat_FatFileSystem.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fssystem/fs_AesXtsFileSystem.h>

#include "shim/fs_Library.h"

using namespace nn::fssystem;

#include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>

namespace {
    const int HeapSize = 512 * 1024;
    char g_HeapStack[HeapSize];

    // Memory Storage version
#if defined(FS_TEST_LARGE)
    const auto StorageSize = static_cast<int64_t>(64) * 1024 * 1024 * 1024 + 1024 * 1024;
    const auto BaseStorageSize = StorageSize + 512 * 1024 * 1024;
#else
    const auto BaseStorageSize = 128 * 1024 * 1024;
    const auto StorageSize = 8 * 1024 * 1024;
    std::unique_ptr<char[]> storageBuffer;
#endif // FS_TEST_LARGE
    std::unique_ptr<char[]> cacheBuffer;
    std::unique_ptr<nn::fs::IStorage> g_pBaseStorage;

    void GenerateRandomForAesXtsFileSystemForWin(void* pData, size_t size) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_LESS_EQUAL(size, nn::crypto::HmacSha256Generator::MacSize);

        static auto s_IsInitialized = false;
        static auto s_Now = nn::os::GetSystemTick().GetInt64Value();
        static const char s_Key[] = "AesXtsFileSystem";
        if (s_IsInitialized)
        {
            ++s_Now;
        }
        else
        {
            s_IsInitialized = true;
        }
        char data[nn::crypto::HmacSha256Generator::MacSize];
        nn::crypto::GenerateHmacSha256Mac(
            &data, sizeof(data),
            &s_Now, sizeof(s_Now),
            s_Key, sizeof(s_Key)
        );
        std::memcpy(pData, data, size);
    }

    void SetUpFatFs(std::shared_ptr<nn::fat::FatFileSystem>* pOutValue) NN_NOEXCEPT
    {
        // FatFs を生成
        auto fatFs = nnt::fs::util::AllocateShared<nn::fat::FatFileSystem>();
        ASSERT_NE(fatFs, nullptr);

        // キャッシュバッファ・IStorage を割り当て
        ASSERT_NE(cacheBuffer, nullptr);
        size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
        NNT_ASSERT_RESULT_SUCCESS(fatFs->Initialize(g_pBaseStorage.get(), cacheBuffer.get(), cacheBufferSize));

        static bool s_IsFormated = false;
        if( !s_IsFormated )
        {
            // FAT にフォーマット
            fat::FatFormatParam param =
            {
#if defined(FS_TEST_LARGE)
                true,
#else
                false,
#endif // FS_TEST_LARGE
                0,
                nn::fs::ResultUnknown()
            };
            NNT_ASSERT_RESULT_SUCCESS(fatFs->Format(&param));
            s_IsFormated = true;
        }

        // fatfs 内部のマウント処理
        NNT_ASSERT_RESULT_SUCCESS(fatFs->Mount());

        *pOutValue = std::move(fatFs);
    }

#if defined(FS_TEST_LARGE)
    class NoCheckAllocator : public nn::MemoryResource
    {
    private:
        void* do_allocate(std::size_t bytes, std::size_t alignment) NN_NOEXCEPT
        {
            NN_UNUSED(alignment);
            return std::malloc(bytes);
        }

        void do_deallocate(void* p, std::size_t bytes, std::size_t alignment) NN_NOEXCEPT
        {
            NN_UNUSED(bytes);
            NN_UNUSED(alignment);
            std::free(p);
        }

        virtual bool do_is_equal(const MemoryResource& other) const NN_NOEXCEPT
        {
            return this == &other;
        }
    };

    NoCheckAllocator g_Allocator;
#endif // FS_TEST_LARGE
}

namespace nnt { namespace fs { namespace api {

    void GetTestFileSystemInfo(std::unique_ptr<TestFileSystemInfo>* outValue, const int index) NN_NOEXCEPT
    {
        NN_UNUSED(index);

        std::unique_ptr<nn::fssystem::AesXtsFileSystem> aesXtsFs(new nn::fssystem::AesXtsFileSystem());
        {
            // オリジナルのFileSystem
            std::shared_ptr<nn::fat::FatFileSystem> fatFs;
            SetUpFatFs(&fatFs);

            // 決め打ちの適当な鍵を使う
            static const char encryptionKeyGenerationKey[nn::fssystem::AesXtsFileSystem::EncryptionKeyGenerationKeySize] =
                { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 };
            static const char macKey[nn::fssystem::AesXtsFileSystem::MacKeySize] =
                { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 };

            aesXtsFs->Initialize(
                fatFs,
                encryptionKeyGenerationKey, sizeof(encryptionKeyGenerationKey),
                macKey, sizeof(macKey),
                GenerateRandomForAesXtsFileSystemForWin,
                nn::fssystem::AesXtsFileSystem::AesBlockSize);
        }

        std::unique_ptr<TestFileSystemInfo> info(new TestFileSystemInfo(FsApiTestType::Unit));

        info->fileSystem = std::unique_ptr<ITestFileSystem>(new FsApiUnitTestFileSystem(std::move(aesXtsFs)));
        info->type = FileSystemType_AesXtsFileSystem;
        info->rootDirPath = "";

        auto& attribute = info->attribute;

        static const int FatNameLengthMax  = 255;     // prfile2 の LONG_NAME_CHARS
        static const int FatPathLengthMax  = 260 - 1; // prfile2 の LONG_NAME_PATH_CHARS
        static const int DriveLetterLength = 2;       // FatFileSystem で付与されるドライブレターの文字数 ("A:")
        attribute.directoryNameLengthMax = FatNameLengthMax;
        attribute.fileNameLengthMax      = FatNameLengthMax;
        attribute.directoryPathLengthMax = FatPathLengthMax - DriveLetterLength;
        attribute.filePathLengthMax      = FatPathLengthMax - DriveLetterLength;

#if defined(FS_TEST_LARGE)
        attribute.fileSizeMax = StorageSize;
#else
        attribute.fileSizeMax = 0xFFFFFFFF;
#endif
        attribute.storageSize = StorageSize;

        attribute.fileOpenMax      = 30;
        attribute.directoryOpenMax = 20;

        attribute.isSupportedMultiBytePath = false;
        attribute.isSupportedGetFreeSpaceSize = false;
        attribute.isConcatenationFileSystem = false;
        attribute.isSupportedQueryRange = true;
        attribute.isMemoryStorage = true;
        attribute.isRootExcessFailureResultPathNotFound = true;

        attribute.fileSizeAlignment = 1;

        *outValue = std::move(info);
    }
}}}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

#if defined(FS_TEST_LARGE)
    if( !nn::fat::FatFileSystem::IsExFatSupported() )
    {
        nnt::Exit(1);
    }
#endif // FS_TEST_LARGE

    ::testing::InitGoogleTest(&argc, argv);

    nnt::fs::api::LoadAllTests();

    nnt::fs::util::InitializeTestLibraryHeap(g_HeapStack, HeapSize);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();

    nn::fs::detail::InitializeFileSystemLibrary();

#if defined(FS_TEST_LARGE)
    nnt::fs::util::VirtualMemoryStorage::SetAllocator(&g_Allocator);
    nn::fat::FatFileSystem::SetAllocatorForFat(&g_Allocator);
    g_pBaseStorage.reset(new nnt::fs::util::VirtualMemoryStorage(BaseStorageSize));
#else
    storageBuffer.reset(new char[BaseStorageSize]);
    ASSERT_NE(storageBuffer, nullptr);
    memset(storageBuffer.get(), 0xCD, BaseStorageSize);
    g_pBaseStorage.reset(new nn::fs::MemoryStorage(storageBuffer.get(), BaseStorageSize));
#endif // FS_TEST_LARGE

    size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
    cacheBuffer.reset(new char[cacheBufferSize]);

    auto ret = RUN_ALL_TESTS();

    g_pBaseStorage.reset(nullptr);

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(ret);
}
