﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/util/util_StringUtil.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util_performance.h>
#include <nnt/fsUtil/testFs_util_storagePerformance.autogen.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_SdCardForDebug.h>

namespace nnt { namespace fs { namespace util {

nn::os::Mutex g_MutexPerformanceResults(false);

/**
* @brief 与えられた値の倍数でのランダムな値を配列に設定します。
* @param[in/out] array[]    オフセット値格納用の配列
* @param[in] arrayCount     ファイルの分割数
* @param[in]     unitSize   バッファサイズ
*/
void CreateRandomArray(int array[], const int arrayCount, const int unitSize) NN_NOEXCEPT
{
    for(int i = 0; i < arrayCount; i++)
    {
        array[i] = i * unitSize;
    }
    // 乱数を生成して配列をシャッフル
    static int s_Seed = 12345;
    std::shuffle(array, array + arrayCount, std::mt19937(s_Seed++));
}

/**
* @brief パフォーマンス計測の構造体オブジェクト vector<PerformanceResult> を push_back します。
* @brief 構造体メンバー bufferSize が一致する要素がある場合は、その中の extensionValue を push_back し、
* @brief なければ、パフォーマンス計測の構造体オブジェクト自身を push_back します。
*/
void AddPerformanceResults(Vector<PerformanceResult>& perfObject, const int bufferSize, const int64_t totalSize, const int64_t elapsedTimeUsec) NN_NOEXCEPT
{
    int done = 0;
    if(perfObject.size() == 0)
    {
        perfObject.push_back(PerformanceResult(bufferSize, totalSize, elapsedTimeUsec));
    }
    else
    {
        for(size_t i = 0 ; i < perfObject.size(); i++)
        {
            // bufferSize 値が同じ場合は、その配下（extensionValue）を push_back します。
            if(perfObject[i].bufferSize == bufferSize)
            {
                perfObject[i].extensionValue.push_back(PerformanceResult(bufferSize, totalSize, elapsedTimeUsec));
                done = 1;
                break;
            }
        }
        // 同じ bufferSize 値がない場合は、本体（perfObject）を push_back します。
        if(done == 0)
        {
            perfObject.push_back(PerformanceResult(bufferSize, totalSize, elapsedTimeUsec));
        }
    }
}

/**
* @brief マルチスレッド用
* @brief パフォーマンス計測の構造体オブジェクト vector<PerformanceResult> を push_back 用関数。
* @brief elapsedTimeUsec 値の大きい方のみを保持します。
* @brief
*/
void AddPerformanceResultsMultiThread(Vector<PerformanceResult>& perfObject, const int bufferSize, const int64_t totalSize, const int64_t elapsedTimeUsec, const int threadCount) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_MutexPerformanceResults);

    static int threadCountId = 0;
    if(threadCountId == 0)
    {
        // マルチスレッドの初回ループ時は push_back を行います。
        threadCountId = threadCount;
        AddPerformanceResults(perfObject, bufferSize, totalSize, elapsedTimeUsec);
    }
    else
    {
        for(size_t i = 0 ; i < perfObject.size(); i++)
        {
            if(perfObject[i].bufferSize == bufferSize)
            {
                // 既に他のスレッドで push_back が行われている状態では、
                // elapsedTimeUsec 値を比較して、大きい方のみを保持します。
                size_t extnum = perfObject[i].extensionValue.size();
                if(extnum == 0)
                {
                    size_t num = perfObject.size() - 1;
                    perfObject[num].elapsedTimeUsec = std::max(perfObject[num].elapsedTimeUsec, elapsedTimeUsec);
                }
                else
                {
                    extnum--;
                    perfObject[i].extensionValue[extnum].elapsedTimeUsec = std::max(perfObject[i].extensionValue[extnum].elapsedTimeUsec, elapsedTimeUsec);
                    break;
                }
            }
        }
    }
    threadCountId--;
}

/**
* @brief Calculates the speed of a data transfer
* @param[in] totalSize      Size of data transfer in bytes
* @param[in] elapsedTime    Duration of data transfer in usec
* @return                   Data transfer speed in MiB/s
*/
double CalculateDataVelocity(int64_t totalSize, int64_t elapsedTime) NN_NOEXCEPT
{
    double mebibytes = (double) totalSize / (1024 * 1024);
    double seconds = (double) elapsedTime / (1000 * 1000);

    return mebibytes / seconds;
}

/**
* @brief パフォーマンス計測結果を TeamCity が解釈できるフォーマットで書き込みます。
* @brief シーケンシャル、ランダムアクセスの順で実行するテスト用です。
* @param [in] prefix     ログに表示する名前
* @param [in] results    パフォーマンス計測データ
*/
void WriteTeamCityPerformanceResults(const char* prefix, Vector<PerformanceResult>& results) NN_NOEXCEPT
{
    for(auto result : results)
    {
        NN_LOG("##teamcity[buildStatisticValue key='%s_%d_seqVelocity(MiB/s)' value='%.5lf']\n",
            prefix, result.bufferSize,
            CalculateDataVelocity(result.totalSize, result.elapsedTimeUsec));

        NN_LOG("##teamcity[buildStatisticValue key='%s_%d_seqElapsedTime(usec)' value='%lld']\n",
            prefix, result.bufferSize, result.elapsedTimeUsec);

        if(!result.extensionValue.empty())
        {
            auto &ext = result.extensionValue[0];
            NN_LOG("##teamcity[buildStatisticValue key='%s_%d_rndVelocity(MiB/s)' value='%.5lf']\n",
                prefix, result.bufferSize,
                CalculateDataVelocity(result.totalSize, ext.elapsedTimeUsec));

            NN_LOG("##teamcity[buildStatisticValue key='%s_%d_rndElapsedTime(usec)' value='%lld']\n",
                prefix, result.bufferSize, ext.elapsedTimeUsec);
        }
    }
}

/**
* @brief 構造体 PerformanceResult のログを出力します。
* @param[in] results  ログ出力する構造体
*/
void DumpPerformanceResults(Vector<PerformanceResult>& results) NN_NOEXCEPT
{

    int64_t totalSizeflag = 0;
    if(results.size() != 0)
    {
        int64_t totalSize = 0;
        totalSize = results[0].totalSize;
        // 最初のスレッド分の全ての bufferSize のケースでの totalSize が同一の場合を確認。
        if(results[0].extensionValue.size() != 0)
        {
            for(auto result : results)
            {
                if(totalSize != result.totalSize)
                {
                    totalSizeflag = -1;
                    break;
                }
                totalSize = result.totalSize;
            }
            // totalSize が同一の場合、各スレッドの totalSize を 1行で表示する。
            // （各スレッドに totalSize 欄を追加すると1行の文字数が長くなりすぎるため）
            // 但し、最初のスレッド分の totalSize しか確認していないため、以降のスレッド分の totalSize の際については未対応。
            if(totalSizeflag != -1){
                NN_LOG("                                      TotalSize[bytes] : %11lld", results[0].totalSize);
                for(auto ext : results[0].extensionValue)
                {
                    NN_LOG("   TotalSize[bytes] : %11lld", ext.totalSize);
                }
                NN_LOG("\n");
            }
        }
    }

    NN_LOG("BufferSize[bytes]  TotalSize[bytes]   Velocity[MB/s]   Elapsed[usec]");
    if(results.size() != 0)
    {
        if(results[0].extensionValue.size() != 0)
        {
            // extensionValue.size() 分、表題を追加します。
            for(size_t i = 0 ; i <= results[0].extensionValue.size() - 1; i++)
            {
                NN_LOG("   Velocity[MB/s]   Elapsed[usec]");
            }
        }
    }
    NN_LOG("\n");

    for(auto result : results)
    {
        // totalSize が同一のため、1行で表示した場合は、totalSize は表示しない。
        if(totalSizeflag != -1){
            NN_LOG("        %9d                 -      %11.5lf       %9lld",
                result.bufferSize,
                (double)result.totalSize / 1024 / 1024 / ((double)result.elapsedTimeUsec / 1000 / 1000),
                result.elapsedTimeUsec);
        }
        else
        {
            NN_LOG("        %9d       %11lld      %11.5lf       %9lld",
                result.bufferSize,
                result.totalSize,
                (double)result.totalSize / 1024 / 1024 / ((double)result.elapsedTimeUsec / 1000 / 1000),
                result.elapsedTimeUsec);
        }

        // extensionValue.size() 分、値を出力します。
        for(auto ext : result.extensionValue)
        {
            NN_LOG("      %11.5lf       %9lld",
                (double)result.totalSize / 1024 / 1024 / ((double)ext.elapsedTimeUsec / 1000 / 1000),
                ext.elapsedTimeUsec);
        }
        NN_LOG("\n");
    }
}

/**
* @brief FsTest が想定しているパフォーマンスコンフィグを設定します。
*/
void SetFsTestPerformanceConfiguration()
{
// Windows向け以外の場合、OEライブラリを利用する
#if !defined(NN_BUILD_CONFIG_OS_WIN32)
    nn::oe::Initialize();
    nn::oe::SetPerformanceConfiguration(nn::oe::PerformanceMode::PerformanceMode_Normal, nn::oe::PerformanceConfiguration_Cpu1020MhzGpu307MhzEmc1331Mhz);
    nn::oe::SetPerformanceConfiguration(nn::oe::PerformanceMode::PerformanceMode_Boost, nn::oe::PerformanceConfiguration_Cpu1020MhzGpu307MhzEmc1331Mhz);
#endif
}

/**
* @brief シーケンシャルリードの計測部
* @param[in] filePath  リードを行う対象ファイルパス
* @param[in] argv      内部で生成するバッファのサイズ
*/
void ReadSequentialCore(Vector<PerformanceResult>& perfObject, const String filePath, const int bufferSize, const int64_t fileSize, int64_t offsetDifference) NN_NOEXCEPT
{
    ASSERT_EQ(0, (fileSize - offsetDifference) % bufferSize);
    int64_t totalSize = fileSize - offsetDifference;

    nn::fs::FileHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handle, filePath.c_str(), nn::fs::OpenMode_Read));

    std::unique_ptr<char[]> readBuffer(new char[bufferSize]);
    ASSERT_NE(nullptr, readBuffer.get());

    TimeCount timeCount;
    for(int64_t offset = 0; offset < totalSize; offset += bufferSize)
    {
        NNT_FS_SCOPED_TRACE_SAFE("buffer size: %d, offset count: %d\n", bufferSize, offset);
        timeCount.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(
            handle, offset + offsetDifference, readBuffer.get(), bufferSize));
        timeCount.StopTime();
    }

    // パフォーマンス計測結果 push_back 処理（シングルスレッド用）
    AddPerformanceResults(perfObject, bufferSize, totalSize, timeCount.GetTotalTime() / 1000);

    nn::fs::CloseFile(handle);
}

/**
* @brief ランダムリードの計測部
*/
void ReadShuffleCore(Vector<PerformanceResult>& perfObject, const String filePath, const int bufferSize, const int64_t fileSize, int64_t offsetDifference) NN_NOEXCEPT
{
    ASSERT_EQ(0, (fileSize - offsetDifference) % bufferSize);
    int64_t totalSize = fileSize - offsetDifference;

    nn::fs::FileHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handle, filePath.c_str(), nn::fs::OpenMode_Read));

    // オフセット値格納用の配列を作成
    int arrayNum = static_cast<int>(totalSize / bufferSize);
    std::unique_ptr<int[]> offsetArray(new int[arrayNum]);
    CreateRandomArray(offsetArray.get(), arrayNum, bufferSize);

    std::unique_ptr<char[]> readBuffer(new char[bufferSize]);

    TimeCount timeCount;
    for(int i = 0; i < arrayNum; i++)
    {
        NNT_FS_SCOPED_TRACE_SAFE("buffer size: %d, offset count: %d\n", bufferSize, offsetArray[i]);
        timeCount.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(
            handle, offsetArray[i] + offsetDifference, readBuffer.get(), bufferSize));
        timeCount.StopTime();
    }

    // パフォーマンス計測結果 push_back 処理（シングルスレッド用）
    AddPerformanceResults(perfObject, bufferSize, totalSize, timeCount.GetTotalTime() / 1000);

    nn::fs::CloseFile(handle);
}

nn::Result GetFileSize(int64_t* pOutValue, const char* filePath) NN_NOEXCEPT
{
    nn::fs::FileHandle file;
    NN_RESULT_DO(nn::fs::OpenFile(&file, filePath, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(file);
    };
    return nn::fs::GetFileSize(pOutValue, file);
}

/**
* @brief パッチのパフォーマンステストとして ROM 内のファイルのシーケンシャル読み込みを計測します。
* @pre MountRom が成功している
*/
void ReadSequentialForPatchedRom(
    Vector<PerformanceResult>* pPerformanceResults,
    const String& filePath,
    int bufferSizeMin,
    int bufferSizeMax,
    int divisionCountMax
) NN_NOEXCEPT
{
    ASSERT_NE(nullptr, pPerformanceResults);

    int64_t originalFileSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(GetFileSize(&originalFileSize, filePath.c_str()));
    for( int bufferSize = bufferSizeMin;
        bufferSize <= bufferSizeMax && bufferSize <= originalFileSize; bufferSize *= 2 )
    {
        const int64_t fileSize
            = std::min(originalFileSize, static_cast<int64_t>(bufferSize) * divisionCountMax);
        ReadSequentialCore(*pPerformanceResults, filePath, bufferSize, fileSize, 0);
    }
}

/**
* @brief パッチのパフォーマンステストとして ROM 内のファイルのランダムな箇所からの読み込みを計測します。
* @pre MountRom が成功している
*/
void ReadShuffleForPatchedRom(
    Vector<PerformanceResult>* pPerformanceResults,
    const String& filePath,
    int bufferSizeMin,
    int bufferSizeMax,
    int divisionCountMax
) NN_NOEXCEPT
{
    ASSERT_NE(nullptr, pPerformanceResults);

    int64_t originalFileSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(GetFileSize(&originalFileSize, filePath.c_str()));

    for( int bufferSize = bufferSizeMin;
        bufferSize <= bufferSizeMax && bufferSize <= originalFileSize; bufferSize *= 2 )
    {
        const int64_t fileSize
            = std::min(originalFileSize, static_cast<int64_t>(bufferSize) * divisionCountMax);
        ReadShuffleCore(*pPerformanceResults, filePath, bufferSize, fileSize, 0);
    }
}

/**
* @brief パッチのパフォーマンステストとしてシーケンシャルリード、ランダムリードの速度を計測します。
* @param[in] fileName           計測で読み込むファイル名（マウント名を含まないパス）
* @param[in] bufferSizeMin      最小バッファサイズ
* @param[in] bufferSizeMax      最大バッファサイズ
* @param[in] divisionCountMax   バッファサイズごとに読み込む最大回数
*/
void PerformanceReadTestForPatchedRom(
    const char* fileName,
    int bufferSizeMin,
    int bufferSizeMax,
    int divisionCountMax
) NN_NOEXCEPT
{
    size_t mountRomCacheBufferSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheBufferSize));
    auto mountRomCacheBuffer = nnt::fs::util::AllocateBuffer(mountRomCacheBufferSize);

    String mountName = "perftest";
    const String filePath = mountName + ":/" + fileName;
    Vector<PerformanceResult> performanceResults;

    {
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::MountRom(
                mountName.c_str(), mountRomCacheBuffer.get(), mountRomCacheBufferSize));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(mountName.c_str());
        };

        ReadSequentialForPatchedRom(
            &performanceResults, filePath, bufferSizeMin, bufferSizeMax, divisionCountMax);
        ReadShuffleForPatchedRom(
            &performanceResults, filePath, bufferSizeMin, bufferSizeMax, divisionCountMax);

    }

    NN_LOG("Read Rom Performance Test\n");
    NN_LOG("                                                          Sequential                           Random\n");
    nnt::fs::util::DumpPerformanceResults(performanceResults);
    NN_LOG("\n\n");
}

/**
* @brief 与えられた targetSpeed がその状況における下限速度を満たしているかを判定します。
* @param[in] targetSpeed        計測されたストレージの速度
* @param[in] storageSpeedSample 下限速度の元となるサンプル速度
* @param[in] chunkSize          計測したチャンクサイズ
* @param[in] isRand             シーケンスかランダムのアクセスかどうか(シーケンスかランダムの 2 択であること前提)
* @param[in] isVsSlower         Slower と他の速度エミュレーションを比較する場合は true, 同じモード同士で比較する場合は false
*/
bool CheckPerformanceThreashold(double targetSpeed, const StorageSpeedSample& storageSpeedSample, int chunkSize, bool isRand, bool isVsSlower) NN_NOEXCEPT
{
    const double MinimumSpeedRate = 0.85;
    const double MinimumSpeedRateForVsSlower = 0.95;

    NN_ASSERT_GREATER_EQUAL(chunkSize, 1024);
    const int chunkSizeIndex = static_cast<int>(std::log10(static_cast<double>(chunkSize)) / std::log10(2.0) - 10);// 1024 スタート
    const double storageSpeed = storageSpeedSample.data[isRand][chunkSizeIndex];
    const double minimumSpeed = storageSpeed * (isVsSlower ? MinimumSpeedRateForVsSlower : MinimumSpeedRate);
    if( !isVsSlower && targetSpeed <= minimumSpeed )
    {
        NN_LOG("--- below lower limit!! ---\n");
        NN_LOG("storageTypeName = %s\n", storageSpeedSample.storageTypeName);
        NN_LOG("chunkSizeIndex = %d\n", chunkSizeIndex);
        NN_LOG("isRand = %d\n", isRand);
        NN_LOG("speed %lf > %lf \n", targetSpeed, minimumSpeed);
        return false;
    }
    return true;
}

bool CheckPerformanceResult(const char* prefix, const Vector<PerformanceResult>& results, bool isSlowerVsOff) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    NN_UNUSED(prefix);
    NN_UNUSED(results);
    NN_UNUSED(isSlowerVsOff);
    return true;
#else
    const int TestNameMax = 64;
    NN_LOG("=== %s ===\n", prefix);
    for( const auto& storageSpeedSample : StorageSpeedSampleTable )
    {
        if( (strncmp(prefix, storageSpeedSample.storageTypeName, strlen(storageSpeedSample.storageTypeName)) == 0)
            && (strnlen(prefix, TestNameMax) == strnlen(storageSpeedSample.storageTypeName, TestNameMax)) )
        {
            for(const auto& result : results)
            {
                if(!CheckPerformanceThreashold(CalculateDataVelocity(result.totalSize, result.elapsedTimeUsec), storageSpeedSample, result.bufferSize, false, isSlowerVsOff))
                {
                    return false;
                }
                if(!result.extensionValue.empty())
                {
                    auto &ext = result.extensionValue[0];
                    if(!CheckPerformanceThreashold(CalculateDataVelocity(result.totalSize, ext.elapsedTimeUsec), storageSpeedSample, result.bufferSize, true, isSlowerVsOff))
                    {
                        return false;
                    }
                }
            }
            break;
        }
    }
    return true;
#endif
}

nn::fs::PriorityRaw ConvertStringToPriorityRaw(const char* pPriority) NN_NOEXCEPT
{
    nn::fs::PriorityRaw priority = nn::fs::PriorityRaw_Normal;

    if( std::strcmp(pPriority, "Realtime") == 0 )
    {
        priority = nn::fs::PriorityRaw_Realtime;
    }
    else if( std::strcmp(pPriority, "Normal") == 0 )
    {
        priority = nn::fs::PriorityRaw_Normal;
    }
    else if( std::strcmp(pPriority, "Low") == 0 )
    {
        priority = nn::fs::PriorityRaw_Low;
    }
    else if( std::strcmp(pPriority, "Background") == 0 )
    {
        priority = nn::fs::PriorityRaw_Background;
    }
    else if( std::strcmp(pPriority, "Default") == 0 )
    {
        priority = nn::fs::GetPriorityRawOnCurrentThread();
    }
    else
    {
        NN_ABORT("Unknown priority %s\n", pPriority);
    }

    return priority;
}

const char* ConvertPriorityRawToString(nn::fs::PriorityRaw priority) NN_NOEXCEPT
{
    switch( priority )
    {
    case nn::fs::PriorityRaw_Low:
        return "Low";
    case nn::fs::PriorityRaw_Background:
        return "Background";
    case nn::fs::PriorityRaw_Realtime:
        return "Realtime";
    case nn::fs::PriorityRaw_Normal:
        return "Normal";
    default:
        NN_ABORT("Unknown priority %d\n", priority);
        return "Unknown";
    }
}

namespace {
    std::atomic_bool g_IsBackgroundAccessThreadAlive(false);
    nn::os::ThreadType g_BackgroundAccessThread;
    char NN_OS_ALIGNAS_THREAD_STACK g_BackgroundAccessThreadStack[16 * 1024];
    nn::os::LightEvent g_BackgroundAccessThreadLaunchEvent(nn::os::EventClearMode_AutoClear);

    void BackgroundAccessThread(void* pArgument) NN_NOEXCEPT
    {
        auto pBackgroundAccessThreadParameter = reinterpret_cast<BackgroundAccessThreadParameter*>(pArgument);

        static const char* MountName = "bg";
        static const char* TestFileName = "bg:/BackgroundAccessThread.bin";

        auto buffer = nnt::fs::util::AllocateBuffer(pBackgroundAccessThreadParameter->accessSize);
        if( !buffer )
        {
            NN_LOG("buffer allocation failed at %s %s(%d)\n", NN_CURRENT_FUNCTION_NAME, __FILE__, __LINE__);
            return;
        }

        nn::Result resultMount;
        switch( pBackgroundAccessThreadParameter->fileSystemType )
        {
        case BackgroundAccessThreadFileSystemType::BisFs:
            resultMount = nn::fs::MountBis(MountName, nn::fs::BisPartitionId::User);
            break;
        case BackgroundAccessThreadFileSystemType::SdCard:
            resultMount = nn::fs::MountSdCardForDebug(MountName);
            break;
        default:
            return;
        }
        if( resultMount.IsFailure() )
        {
            NN_LOG("mount failed at %s %s(%d)\n", NN_CURRENT_FUNCTION_NAME, __FILE__, __LINE__);
            return;
        }
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName);
        };
        if( nnt::fs::util::DeleteFileOrDirectoryIfExists(TestFileName).IsFailure() )
        {
            NN_LOG("%s delete failed at %s %s(%d)\n", TestFileName, NN_CURRENT_FUNCTION_NAME, __FILE__, __LINE__);
            return;
        }
        if( nn::fs::CreateFile(TestFileName, pBackgroundAccessThreadParameter->storageSize).IsFailure() )
        {
            NN_LOG("%s create failed at %s %s(%d)\n", TestFileName, NN_CURRENT_FUNCTION_NAME, __FILE__, __LINE__);
            return;
        }
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::DeleteFile(TestFileName);
        };

        nn::fs::FileHandle file;
        if( nn::fs::OpenFile(&file, TestFileName, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write).IsFailure() )
        {
            NN_LOG("%s open failed at %s %s(%d)\n", TestFileName, NN_CURRENT_FUNCTION_NAME, __FILE__, __LINE__);
            return;
        }
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(file);
        };

        nn::fs::SetPriorityRawOnCurrentThread(pBackgroundAccessThreadParameter->accessPriority);

        g_BackgroundAccessThreadLaunchEvent.Signal();

        int64_t offset = 0;
        while( g_IsBackgroundAccessThreadAlive )
        {
            nn::Result result;

            switch( pBackgroundAccessThreadParameter->accessType )
            {
            case BackgroundAccessThreadAccessType::Read:
                result = nn::fs::ReadFile(file, offset, buffer.get(), pBackgroundAccessThreadParameter->accessSize);
                break;
            case BackgroundAccessThreadAccessType::Write:
                result = nn::fs::WriteFile(file, offset, buffer.get(), pBackgroundAccessThreadParameter->accessSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
                break;
            default:
                FAIL();
            }

            if( result.IsFailure() )
            {
                NN_LOG("%s read failed at %s %s(%d)\n", TestFileName, NN_CURRENT_FUNCTION_NAME, __FILE__, __LINE__);
                break;
            }

            if( pBackgroundAccessThreadParameter->intervalMilliSeconds > 0 )
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pBackgroundAccessThreadParameter->intervalMilliSeconds));
            }
        }
    }
}

void ParseBackgroundAccessThreadParameter(BackgroundAccessThreadParameter* pParameter, int* pArgc, char** pArgv) NN_NOEXCEPT
{
    pParameter->isEnabled = false;
    pParameter->fileSystemType = BackgroundAccessThreadFileSystemType::BisFs;
    pParameter->threadPriority = nn::os::DefaultThreadPriority;
    pParameter->accessPriority = nn::fs::PriorityRaw_Normal;
    pParameter->accessType = BackgroundAccessThreadAccessType::Read;
    pParameter->accessSize = 16 * 1024;
    pParameter->storageSize = 256 * 1024;
    pParameter->intervalMilliSeconds = 0;

    for( int i = 0; i < *pArgc; )
    {
        if( nn::util::Strncmp(pArgv[i], "--bg_", 5) == 0 )
        {
            int shiftWidth = 0;

            const char* pOption = pArgv[i] + 5;
            ++i;
            ++shiftWidth;

            pParameter->isEnabled = true;

            if( std::strcmp(pOption, "fs-type") == 0 )
            {
                if( i >= *pArgc )
                {
                    NN_LOG("filesystem type not specified\n");
                    FAIL();
                }

                pOption = pArgv[i];
                ++i;
                ++shiftWidth;
                if( std::strcmp(pOption, "BisFs") == 0 )
                {
                    pParameter->fileSystemType = BackgroundAccessThreadFileSystemType::BisFs;
                }
                else if( std::strcmp(pOption, "SdCard") == 0 )
                {
                    pParameter->fileSystemType = BackgroundAccessThreadFileSystemType::SdCard;
                }
                else
                {
                    NN_LOG("unknown filesystem type: %s\n", pOption);
                    FAIL();
                }
            }
            else if( std::strcmp(pOption, "thread-priority") == 0 )
            {
                if( i >= *pArgc )
                {
                    NN_LOG("thread priority not specified\n");
                    FAIL();
                }

                pParameter->threadPriority = nn::os::DefaultThreadPriority + std::atoi(pArgv[i]);
                ++i;
                ++shiftWidth;
            }
            else if( std::strcmp(pOption, "access-priority") == 0 )
            {
                if( i >= *pArgc )
                {
                    NN_LOG("access priority not specified\n");
                    FAIL();
                }

                pParameter->accessPriority = ConvertStringToPriorityRaw(pArgv[i]);
                ++i;
                ++shiftWidth;
            }
            else if( std::strcmp(pOption, "access-size") == 0 )
            {
                if( i >= *pArgc )
                {
                    NN_LOG("access size not specified\n");
                    FAIL();
                }

                int64_t value = std::strtoll(pArgv[i], nullptr, 10);
                pParameter->accessSize = static_cast<size_t>(value);
                ++i;
                ++shiftWidth;
            }
            else if( std::strcmp(pOption, "storage-size") == 0 )
            {
                if( i >= *pArgc )
                {
                    NN_LOG("storage size not specified\n");
                    FAIL();
                }

                int64_t value = std::strtoll(pArgv[i], nullptr, 10);
                pParameter->storageSize = value;
                ++i;
                ++shiftWidth;
            }
            else if( std::strcmp(pOption, "interval") == 0 )
            {
                if( i >= *pArgc )
                {
                    NN_LOG("interval not specified\n");
                    FAIL();
                }

                pParameter->intervalMilliSeconds = std::atoi(pArgv[i]);
                ++i;
                ++shiftWidth;
            }
            else if( std::strcmp(pOption, "access-type") == 0 )
            {
                if( i >= *pArgc )
                {
                    NN_LOG("access type not specified\n");
                    FAIL();
                }

                pOption = pArgv[i];
                ++i;
                ++shiftWidth;
                if( std::strcmp(pOption, "Read") == 0 )
                {
                    pParameter->accessType = BackgroundAccessThreadAccessType::Read;
                }
                else if( std::strcmp(pOption, "Write") == 0 )
                {
                    pParameter->accessType = BackgroundAccessThreadAccessType::Write;
                }
                else
                {
                    NN_LOG("unknown access type: %s\n", pOption);
                    FAIL();
                }
            }
            else
            {
                NN_LOG("unknown option: %s", pOption);
                FAIL();
            }

            std::memmove(pArgv + i - shiftWidth, pArgv + i, sizeof(pArgv[0]) * (*pArgc - i));
            i -= shiftWidth;
            *pArgc -= shiftWidth;
        }
        else
        {
            ++i;
        }
    }
} // NOLINT(impl/function_size)

void RunBackgroundAccessThread(const BackgroundAccessThreadParameter& parameter) NN_NOEXCEPT
{
    ASSERT_FALSE(g_IsBackgroundAccessThreadAlive);

    static BackgroundAccessThreadParameter s_Parameter;
    s_Parameter = parameter;

    NNT_ASSERT_RESULT_SUCCESS(
        nn::os::CreateThread(
            &g_BackgroundAccessThread,
            BackgroundAccessThread,
            &s_Parameter,
            g_BackgroundAccessThreadStack,
            sizeof(g_BackgroundAccessThreadStack),
            parameter.threadPriority,
            1
        )
    );
    nn::os::SetThreadCoreMask(&g_BackgroundAccessThread, 1, 1 << 1);

    g_IsBackgroundAccessThreadAlive = true;
    nn::os::StartThread(&g_BackgroundAccessThread);

    g_BackgroundAccessThreadLaunchEvent.Wait();
}

void StopBackgroundAccessThread() NN_NOEXCEPT
{
    ASSERT_TRUE(g_IsBackgroundAccessThreadAlive);
    g_IsBackgroundAccessThreadAlive = false;
    nn::os::WaitThread(&g_BackgroundAccessThread);
    nn::os::DestroyThread(&g_BackgroundAccessThread);
}

}}}
