﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/detail/fs_CommonMountName.h>
#include <nn/fs/fsa/fs_Registrar.h>

#include <nnt/fsUtil/testFs_util_fixture_Mount.h>
#include <nnt/fsUtil/mock/testFs_util_MockFileSystem.h>

namespace nnt { namespace fs { namespace util {

void LoadMountTest() NN_NOEXCEPT
{
}

namespace
{
    const char* TestMountName = "mount";
}

#if defined(NN_BUILD_CONFIG_COMPILER_VC)
void PrintTo(const MountTestParameter& parameter, ::std::ostream* os) NN_NOEXCEPT
#else
void PrintTo(const MountTestParameter& parameter, ::nnt::testing::detail::StringStream* os) NN_NOEXCEPT
#endif
{
    *os << parameter.mountName;
}

nn::Result WithMountFunction::MountByName(const char* name) NN_NOEXCEPT
{
    auto parameter = GetParam();

    NN_ABORT_UNLESS_NOT_NULL(parameter.mountByName);

    NN_RESULT_DO(parameter.mountByName(name));
    NN_RESULT_SUCCESS;
}

nn::Result WithMountFunction::MountAndUnmountByNameSafely(const char* name) NN_NOEXCEPT
{
    NN_RESULT_DO(MountByName(name));
    nn::fs::Unmount(name);
    NN_RESULT_SUCCESS;
}

nn::Result WithMountFunction::MountByPath(const char* path) NN_NOEXCEPT
{
    auto parameter = GetParam();

    NN_ABORT_UNLESS_NOT_NULL(parameter.mountByPath);

    NN_RESULT_DO(parameter.mountByPath(TestMountName, path));
    NN_RESULT_SUCCESS;
}

nn::Result WithMountFunction::MountAndUnmountByPathSafely(const char* path) NN_NOEXCEPT
{
    NN_RESULT_DO(MountByPath(path));
    nn::fs::Unmount(TestMountName);
    NN_RESULT_SUCCESS;
}

const MountTestAttribute WithMountFunction::GetAttribute() NN_NOEXCEPT
{
    if( GetParam().getMountTestAttribute == nullptr )
    {
        MountTestAttribute attribute = {};
        return attribute;
    }
    else
    {
        return GetParam().getMountTestAttribute();
    }
}

void WithMountFunction::SetUp() NN_NOEXCEPT
{
    if( GetAttribute().setUp != nullptr )
    {
        GetAttribute().setUp();
    }
}

void WithMountFunction::TearDown() NN_NOEXCEPT
{
    if( GetAttribute().tearDowm != nullptr )
    {
        GetAttribute().tearDowm();
    }
}

TEST_P(MountBasic, ValidMountName)
{
    NNT_FS_UTIL_SKIP_TEST_UNLESS(!GetAttribute().isTestSkipRequired);
    NNT_FS_UTIL_SKIP_TEST_UNLESS(GetParam().mountByName != nullptr);

    SetUpWithParam();
    NN_UTIL_SCOPE_EXIT
    {
        TearDownWithParam();
    };

    // InvalidMountName に対する Success パターンをテストする

    // '@' 以外の文字で始まる。
    // → @ 以外の記号は許可
    NNT_EXPECT_RESULT_SUCCESS(MountAndUnmountByNameSafely("$test"));
    // → @ が先頭でないなら許可
    NNT_EXPECT_RESULT_SUCCESS(MountAndUnmountByNameSafely("test@"));

    if( GetAttribute().isReservedMountNameSupported )
    {
        // 一部のファイルシステムは '@' を許可している。
        // (他のファイルシステムで予約されたものであっても)
        NNT_EXPECT_RESULT_SUCCESS(MountAndUnmountByNameSafely("@test"));
        NNT_EXPECT_RESULT_SUCCESS(MountAndUnmountByNameSafely(nn::fs::detail::BisUserPartitionMountName));
        NNT_EXPECT_RESULT_SUCCESS(MountAndUnmountByNameSafely(nn::fs::detail::HostRootFileSystemMountName));
        NNT_EXPECT_RESULT_SUCCESS(MountAndUnmountByNameSafely(nn::fs::detail::SdCardFileSystemMountName));
        NNT_EXPECT_RESULT_SUCCESS(MountAndUnmountByNameSafely(nn::fs::detail::GameCardFileSystemMountName));
        NNT_EXPECT_RESULT_SUCCESS(MountAndUnmountByNameSafely(nn::fs::detail::ContentStorageUserMountName));
        NNT_EXPECT_RESULT_SUCCESS(MountAndUnmountByNameSafely(nn::fs::detail::RegisteredUpdatePartitionMountName));
    }

    // ':' および '/' を含まない。
    // → それ以外の記号 (↑とは違う記号で)
    NNT_EXPECT_RESULT_SUCCESS(MountAndUnmountByNameSafely("test\\"));
    NNT_EXPECT_RESULT_SUCCESS(MountAndUnmountByNameSafely("test;"));

    // アルファベット一文字でない。
    // → それ以外の文字なら1文字でも許可
    NNT_EXPECT_RESULT_SUCCESS(MountAndUnmountByNameSafely("_"));

    // 終端の NULL 文字を含めず 1 バイト以上 nn::fs::MountNameLengthMax バイト以下の文字列である。
    // → nn::fs::MountNameLengthMax は許可
    {
        char mountNameMax[nn::fs::MountNameLengthMax + 1];
        std::memset(mountNameMax, 'a', nn::fs::MountNameLengthMax);
        mountNameMax[nn::fs::MountNameLengthMax] = '\0';

        NNT_EXPECT_RESULT_SUCCESS(MountAndUnmountByNameSafely(mountNameMax));
    }

    // UTF-8 エンコーディング
    NNT_EXPECT_RESULT_SUCCESS(MountAndUnmountByNameSafely("\xCE\xBA\xE1\xBD\xB9\xCF\x83\xCE\xBC\xCE\xB5"));
}

TEST_P(MountFailure, AlreadyExist)
{
    NNT_FS_UTIL_SKIP_TEST_UNLESS(!GetAttribute().isTestSkipRequired);
    NNT_FS_UTIL_SKIP_TEST_UNLESS(GetParam().mountByName != nullptr);

    SetUpWithParam();
    NN_UTIL_SCOPE_EXIT
    {
        TearDownWithParam();
    };

    static const char* MountName = "test";

    // マウントしている他のファイルシステムのマウント名に使用されていない。に違反

    std::unique_ptr<mock::fsa::MockFileSystem> fileSystem(new mock::fsa::MockFileSystem);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::fsa::Register(MountName, std::move(fileSystem)));

    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultMountNameAlreadyExists, MountAndUnmountByNameSafely(MountName));

    nn::fs::fsa::Unregister(MountName);
}

TEST_P(MountFailure, MountNameIsNullptr)
{
    NNT_FS_UTIL_SKIP_TEST_UNLESS(!GetAttribute().isTestSkipRequired);
    NNT_FS_UTIL_SKIP_TEST_UNLESS(GetParam().mountByName != nullptr);

    SetUpWithParam();
    NN_UTIL_SCOPE_EXIT
    {
        TearDownWithParam();
    };

    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidMountName, MountAndUnmountByNameSafely(nullptr));
}

TEST_P(MountFailure, InvalidMountName)
{
    NNT_FS_UTIL_SKIP_TEST_UNLESS(!GetAttribute().isTestSkipRequired);
    NNT_FS_UTIL_SKIP_TEST_UNLESS(GetParam().mountByName != nullptr);

    SetUpWithParam();
    NN_UTIL_SCOPE_EXIT
    {
        TearDownWithParam();
    };

    {
        char tooLongMountName[nn::fs::MountNameLengthMax + 2];
        std::memset(tooLongMountName, 'a', nn::fs::MountNameLengthMax + 1);
        tooLongMountName[nn::fs::MountNameLengthMax + 1] = '\0';

        // 終端の NULL 文字を含めず 1 バイト以上 nn::fs::MountNameLengthMax バイト以下の文字列である。に違反
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidMountName, MountAndUnmountByNameSafely(""));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidMountName, MountAndUnmountByNameSafely(tooLongMountName));
    }

    if( !GetAttribute().isReservedMountNameSupported )
    {
        // '@' 以外の文字で始まる。に違反
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidMountName, MountAndUnmountByNameSafely("@test"));
    }

    // ':' および '/' を含まない。に違反
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidMountName, MountAndUnmountByNameSafely("test:"));
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidMountName, MountAndUnmountByNameSafely("test/"));

    // アルファベット一文字でない。に違反
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidMountName, MountAndUnmountByNameSafely("D"));
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidMountName, MountAndUnmountByNameSafely("d"));

    // UTF-8 エンコーディング、に違反
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidMountName, MountAndUnmountByNameSafely("\x80"));
}

TEST_P(MountFailure, TooLongPath)
{
    NNT_FS_UTIL_SKIP_TEST_UNLESS(!GetAttribute().isTestSkipRequired);
    NNT_FS_UTIL_SKIP_TEST_UNLESS(GetParam().mountByPath != nullptr);

    const nnt::fs::util::String longPath(nn::fs::EntryNameLengthMax + 1, 'a');

    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultTooLongPath,
        MountAndUnmountByPathSafely(longPath.c_str()));
}

TEST_P(MountPathDeathTest, TooLongPath)
{
    NNT_FS_UTIL_SKIP_TEST_UNLESS(!GetAttribute().isTestSkipRequired);
    NNT_FS_UTIL_SKIP_TEST_UNLESS(GetParam().mountByPath != nullptr);

    nn::fs::SetEnabledAutoAbort(true);
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::SetEnabledAutoAbort(false);
    };

    const nnt::fs::util::String longPath(nn::fs::EntryNameLengthMax + 1, 'a');

    if( GetAttribute().isNotAbortedByTooLongPath )
    {
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultTooLongPath,
            MountAndUnmountByPathSafely(longPath.c_str()));
    }
    else
    {
        EXPECT_DEATH_IF_SUPPORTED(
            MountAndUnmountByPathSafely(longPath.c_str()), "");
    }
}

}}}
