﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util_DynamicModule.h>

namespace nnt { namespace fs { namespace util {

ScopedDynamicModuleLoader::ScopedDynamicModuleLoader() NN_NOEXCEPT
    : m_MountCacheBuffer(nullptr, &nnt::fs::util::DeleterBuffer)
    , m_NrrFileBuffer(nullptr, &Deallocate)
    , m_NroFileBuffer(nullptr, &Deallocate)
    , m_BssBuffer(nullptr, &Deallocate)
    , m_IsRoInitialized(false)
    , m_IsModuleInfoRegistered(false)
    , m_IsModuleLoaded(false)
{
}

ScopedDynamicModuleLoader::~ScopedDynamicModuleLoader() NN_NOEXCEPT
{
    if( m_IsModuleLoaded )
    {
        nn::ro::UnloadModule(&m_Module);
    }

    if( m_IsModuleInfoRegistered )
    {
        nn::ro::UnregisterModuleInfo(&m_Info);
    }

    if( m_IsRoInitialized )
    {
        nn::ro::Finalize();
    }

    if( !m_MountName.empty() )
    {
        nn::fs::Unmount(m_MountName.c_str());
    }
}

nn::Result ScopedDynamicModuleLoader::Load(
    const char* mountName, const char* nrrPath, const char* nroPath) NN_NOEXCEPT
{
    // Rom をマウント
    size_t cacheBufferSize = 0;
    NN_RESULT_DO(nn::fs::QueryMountRomCacheSize(&cacheBufferSize));

    m_MountCacheBuffer = nnt::fs::util::AllocateBuffer(cacheBufferSize);
    NN_ABORT_UNLESS_NOT_NULL(m_MountCacheBuffer);

    NN_RESULT_DO(
        nn::fs::MountRom(mountName, m_MountCacheBuffer.get(), cacheBufferSize));
    m_MountName = mountName;

    nn::ro::Initialize();
    m_IsRoInitialized = true;

    // nrr を登録
    m_NrrFileBuffer = LoadFile(nrrPath);
    NN_RESULT_DO(nn::ro::RegisterModuleInfo(&m_Info, m_NrrFileBuffer.get()));
    m_IsModuleInfoRegistered = true;

    // nro を読み込む
    m_NroFileBuffer = LoadFile(nroPath);

    // 動的モジュールのロードに必要なバッファを確保
    size_t bssBufferSize = 0;
    NN_RESULT_DO(nn::ro::GetBufferSize(&bssBufferSize, m_NroFileBuffer.get()));
    m_BssBuffer = Allocate(bssBufferSize);

    // モジュールをロード
    NN_RESULT_DO(
        nn::ro::LoadModule(
            &m_Module,
            m_NroFileBuffer.get(),
            m_BssBuffer.get(),
            bssBufferSize,
            nn::ro::BindFlag_Lazy));
    m_IsModuleLoaded = true;

    NN_RESULT_SUCCESS;
}

void ScopedDynamicModuleLoader::Deallocate(void* ptr) NN_NOEXCEPT
{
    free(ptr);
}

ScopedDynamicModuleLoader::Buffer ScopedDynamicModuleLoader::Allocate(size_t size) NN_NOEXCEPT
{
    Buffer buffer(aligned_alloc(nn::os::MemoryPageSize, size), Deallocate);

    if( buffer != nullptr )
    {
        memset(buffer.get(), 0, size);
    }

    return buffer;
}

ScopedDynamicModuleLoader::Buffer ScopedDynamicModuleLoader::LoadFile(
    const char* filePath) NN_NOEXCEPT
{
    nn::fs::FileHandle file;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&file, filePath, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(file);
    };

    int64_t fileSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, file));

    auto buffer = Allocate(fileSize);
    NN_ABORT_UNLESS_NOT_NULL(buffer);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(file, 0, buffer.get(), fileSize));

    return buffer;
}


}}}
