﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nnt/nntest.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

namespace nnt { namespace fs { namespace api {
    void LoadPointerArgumentRoTests()
    {
        return;
    }

    namespace {
        const OpenMode OpenFileModes[] =
        {
            OpenMode_Read,
            OpenMode_Write,
            static_cast<OpenMode>(OpenMode_Read  | OpenMode_Write),
            static_cast<OpenMode>(OpenMode_Read  | OpenMode_AllowAppend),
            static_cast<OpenMode>(OpenMode_Write | OpenMode_AllowAppend),
            static_cast<OpenMode>(OpenMode_Read  | OpenMode_Write | OpenMode_AllowAppend)
        };

        const OpenDirectoryMode OpenDirectoryModes[] =
        {
            OpenDirectoryMode_Directory,
            OpenDirectoryMode_File,
            OpenDirectoryMode_All
        };
    }

    class PointerArgumentRoNull : public GetFileSystemTestFixture
    {
    protected:
        PointerArgumentRoNull()
        {
            CheckGlobalNewDeleteFlagTestFixture::DisableCheck();
            CheckMemoryLeakFixture::DisableCheck();
        }

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            GetFileSystemTestFixture::SetUp();
            m_FileName = GetTestRootPath().append("/PointerArgumentRo/test.file");
            m_DirectoryName = GetTestRootPath().append("/PointerArgumentRo/test");
            m_NewFileName = GetTestRootPath().append("/PointerArgumentRo/testNew.file");
            m_NewDirectoryName = GetTestRootPath().append("/PointerArgumentRo/testNew");
        }
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            GetFileSystemTestFixture::TearDown();
        }

        const String& GetFileName() NN_NOEXCEPT
        {
            return m_FileName;
        }

        const String& GetDirectoryName() NN_NOEXCEPT
        {
            return m_DirectoryName;
        }

        const String& GetNewFileName() NN_NOEXCEPT
        {
            return m_NewFileName;
        }

        const String& GetNewDirectoryName() NN_NOEXCEPT
        {
            return m_NewDirectoryName;
        }

    private:
        String m_FileName;
        String m_DirectoryName;
        String m_NewFileName;
        String m_NewDirectoryName;
    };

    TEST_F(PointerArgumentRoNull, GetEntryType)
    {
        DirectoryEntryType directoryEntryType;

        NNT_EXPECT_RESULT_FAILURE(
            ResultNullptrArgument, GetFs().GetEntryType(nullptr, GetFileName().c_str()));
        NNT_EXPECT_RESULT_FAILURE(
            ResultInvalidPath,     GetFs().GetEntryType(&directoryEntryType,  nullptr));
    }

    TEST_F(PointerArgumentRoNull, GetFreeSpaceSize)
    {
        NNT_EXPECT_RESULT_FAILURE(
            ResultNullptrArgument, GetFs().GetFreeSpaceSize(nullptr, GetDirectoryName().c_str()));

        int64_t size = 0;
        NNT_EXPECT_RESULT_FAILURE(
            ResultInvalidPath, GetFs().GetFreeSpaceSize(&size, nullptr));
    }

    TEST_F(PointerArgumentRoNull, GetTotalSpaceSize)
    {
        NNT_EXPECT_RESULT_FAILURE(
            ResultNullptrArgument, GetFs().GetTotalSpaceSize(nullptr, GetDirectoryName().c_str()));

        int64_t size = 0;
        NNT_EXPECT_RESULT_FAILURE(
            ResultInvalidPath, GetFs().GetTotalSpaceSize(&size, nullptr));
    }

    TEST_F(PointerArgumentRoNull, CreateFile)
    {
        NNT_EXPECT_RESULT_FAILURE(ResultInvalidPath, GetFs().CreateFile(nullptr, 0));
    }

    TEST_F(PointerArgumentRoNull, DeleteFile)
    {
        NNT_EXPECT_RESULT_FAILURE(ResultInvalidPath, GetFs().DeleteFile(nullptr));
    }

    TEST_F(PointerArgumentRoNull, RenameFile)
    {
        NNT_EXPECT_RESULT_FAILURE(
            ResultInvalidPath, GetFs().RenameFile(nullptr, GetNewFileName().c_str()));
        NNT_EXPECT_RESULT_FAILURE(
            ResultInvalidPath, GetFs().RenameFile(GetFileName().c_str(), nullptr));
    }

    TEST_F(PointerArgumentRoNull, OpenFile)
    {
        for( auto openMode : OpenFileModes )
        {
            std::unique_ptr<ITestFile> file;
            NNT_EXPECT_RESULT_FAILURE(
                ResultNullptrArgument, GetFs().OpenFile(nullptr, GetFileName().c_str(), openMode));
            NNT_EXPECT_RESULT_FAILURE(
                ResultInvalidPath, GetFs().OpenFile(&file, nullptr, openMode));
        }
    }

    TEST_F(PointerArgumentRoNull, ReadFile)
    {
        std::unique_ptr<ITestFile> file;
        const int FileSize = 32;
        char buffer[FileSize + 1] = {0};
        size_t readSize = FileSize;

        NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, GetFileName().c_str(), OpenMode_Read));

        NNT_EXPECT_RESULT_FAILURE(
            ResultNullptrArgument, file->Read(nullptr, 0, buffer, FileSize, ReadOption()));
        NNT_EXPECT_RESULT_FAILURE(
            ResultNullptrArgument, file->Read(nullptr, 0, buffer, FileSize));

        NNT_EXPECT_RESULT_FAILURE(
            ResultNullptrArgument, file->Read(0, nullptr, FileSize, ReadOption()));
        NNT_EXPECT_RESULT_FAILURE(
            ResultNullptrArgument, file->Read(0, nullptr, FileSize));
        NNT_EXPECT_RESULT_FAILURE(
            ResultNullptrArgument, file->Read(&readSize, 0, nullptr, FileSize, ReadOption()));
        NNT_EXPECT_RESULT_FAILURE(
            ResultNullptrArgument, file->Read(&readSize, 0, nullptr, FileSize));
    }

    TEST_F(PointerArgumentRoNull, WriteFile)
    {
        std::unique_ptr<ITestFile> file;
        const int FileSize = 32;
        WriteOption writeOption = {};

        NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, GetFileName().c_str(), OpenMode_Read));

        NNT_EXPECT_RESULT_FAILURE(
            ResultNullptrArgument, file->Write(0, nullptr, FileSize, writeOption));
    }

    TEST_F(PointerArgumentRoNull, GetFileSize)
    {
        std::unique_ptr<ITestFile> file;

        NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, GetFileName().c_str(), OpenMode_Read));

        NNT_EXPECT_RESULT_FAILURE(ResultNullptrArgument, file->GetSize(nullptr));
    }

    TEST_F(PointerArgumentRoNull, CreateDirectory)
    {
        NNT_EXPECT_RESULT_FAILURE(ResultInvalidPath, GetFs().CreateDirectory(nullptr));
    }

    TEST_F(PointerArgumentRoNull, DeleteDirectory)
    {
        NNT_EXPECT_RESULT_FAILURE(ResultInvalidPath, GetFs().DeleteDirectory(nullptr));
    }

    TEST_F(PointerArgumentRoNull, DeleteDirectoryRecursively)
    {
        NNT_EXPECT_RESULT_FAILURE(ResultInvalidPath, GetFs().DeleteDirectoryRecursively(nullptr));
    }

    TEST_F(PointerArgumentRoNull, RenameDirectory)
    {
        NNT_EXPECT_RESULT_FAILURE(
            ResultInvalidPath, GetFs().RenameDirectory(nullptr, GetNewDirectoryName().c_str()));
        NNT_EXPECT_RESULT_FAILURE(
            ResultInvalidPath, GetFs().RenameDirectory(GetDirectoryName().c_str(), nullptr));
    }

    TEST_F(PointerArgumentRoNull, OpenDirectory)
    {
        for( auto openDirecotryMode : OpenDirectoryModes )
        {
            std::unique_ptr<ITestDirectory> directory;
            NNT_EXPECT_RESULT_FAILURE(
                ResultNullptrArgument,
                GetFs().OpenDirectory(nullptr, GetDirectoryName().c_str(), openDirecotryMode));
            NNT_EXPECT_RESULT_FAILURE(
                ResultInvalidPath,
                GetFs().OpenDirectory(&directory, nullptr, openDirecotryMode));
        }
    }

    TEST_F(PointerArgumentRoNull, ReadDirectory)
    {
        for( auto openDirecotryMode : OpenDirectoryModes )
        {
            std::unique_ptr<ITestDirectory> directory;
            int64_t entryCount;
            DirectoryEntry directoryEntry[2];

            util::InvalidateVariable(directoryEntry, 2);

            NNT_ASSERT_RESULT_SUCCESS(
                GetFs().OpenDirectory(&directory, GetDirectoryName().c_str(), openDirecotryMode));

            NNT_EXPECT_RESULT_FAILURE(
                ResultNullptrArgument, directory->Read(nullptr, directoryEntry, 1));
            NNT_EXPECT_RESULT_FAILURE(
                ResultNullptrArgument, directory->Read(&entryCount, nullptr, 1));
        }
    }

    TEST_F(PointerArgumentRoNull, GetEntryCount)
    {
        for( auto openDirecotryMode : OpenDirectoryModes )
        {
            std::unique_ptr<ITestDirectory> directory;
            NNT_ASSERT_RESULT_SUCCESS(
                GetFs().OpenDirectory(&directory, GetDirectoryName().c_str(), openDirecotryMode));

            NNT_EXPECT_RESULT_FAILURE(ResultNullptrArgument, directory->GetEntryCount(nullptr));
        }
    }

    //!< @brief OperateRange の QueryRange で NULL を渡すと失敗することを確認する
    TEST_F(PointerArgumentRoNull, OperateRange_QueryRange)
    {
        auto fileName = GetTestRootPath().append("/OperateRangeRo/test.file");

        // ファイルオープン
        std::unique_ptr<ITestFile> file;

        NNT_ASSERT_RESULT_SUCCESS(
            GetFs().OpenFile(
                &file,
                fileName.c_str(),
                nn::fs::OpenMode::OpenMode_Read
            )
        );

        int64_t size;
        NNT_ASSERT_RESULT_SUCCESS(file->GetSize(&size));

        if( GetFsAttribute()->isSupportedQueryRange )
        {
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultNullptrArgument,
                file->QueryRange(nullptr, 0, size));
        }
        else
        {
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultUnsupportedOperation,
                file->QueryRange(nullptr, 0, size));
        }
    }

}}}
