﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <algorithm>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/util/util_FormatString.h>
#include <nnt/nntest.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

namespace {
    const char Utf8String[] = {static_cast<char>(-29), static_cast<char>(-127), static_cast<char>(-126), '\0'}; // "あ\0" in utf-8
}

namespace nnt { namespace fs { namespace api {
    void LoadPathArgumentRoTests()
    {
        return;
    }

    class PathArgumentRoLengthMax : public GetFileSystemTestFixture
    {
    public:
        static String GenerateEntryName(String directoryPath, int length, String entryPrefix)
        {
            String entryPath = entryPrefix;
            int left = static_cast<int>(length - entryPath.size() - directoryPath.size());
            while (left > 0)
            {
                char tmp[5];
                int appendSize = std::min(left, 4);
                nn::util::SNPrintf(tmp, appendSize + 1, "long");
                entryPath.append(tmp);
                left -= appendSize;
            }
            return entryPath;
        }

        void CheckIfOneEntryExist(
            const char* directoryName, const char* entryName, bool isDirectory)
        {
            const int pathLengthMax = isDirectory ? m_DirectoryPathLengthMax : m_FilePathLengthMax;
            std::unique_ptr<ITestDirectory> directory;
            NNT_ASSERT_RESULT_SUCCESS(
                GetFs().OpenDirectory(
                    &directory,
                    directoryName,
                    isDirectory ? OpenDirectoryMode_Directory : OpenDirectoryMode_File));
            int64_t readNum = 0;
            const int EntryBufferLength = 1;
            DirectoryEntry dirEntry[EntryBufferLength];
            NNT_EXPECT_RESULT_SUCCESS(directory->Read(&readNum, dirEntry, EntryBufferLength));
            EXPECT_EQ(1, readNum);
            if (readNum > 0)
            {
                EXPECT_EQ(
                    strnlen(dirEntry[0].name, pathLengthMax), strnlen(entryName, pathLengthMax));
                EXPECT_TRUE(strncmp(dirEntry[0].name, entryName, pathLengthMax) == 0);
            }
        }

        int m_FilePathLengthMax;
        int m_DirectoryPathLengthMax;
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            GetFileSystemTestFixture::SetUp();

            // TODO: フルパス長の観点のテスト
            m_FilePathLengthMax      = GetFsAttribute()->fileNameLengthMax;
            m_DirectoryPathLengthMax = GetFsAttribute()->directoryNameLengthMax;
        }
    };

    class PathArgumentRoLengthExcess : public PathArgumentRoLengthMax
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            PathArgumentRoLengthMax::SetUp();
            m_FilePathLengthMax += 1;
            m_DirectoryPathLengthMax += 1;
        }
    };

    class PathArgumentRoRootExcessBase : public nnt::fs::api::GetFileSystemTestFixture
    {
    public:
        virtual ~PathArgumentRoRootExcessBase() NN_NOEXCEPT NN_OVERRIDE
        {
        }

    protected:
        void DoTest() NN_NOEXCEPT;

        virtual const nnt::fs::util::String& GetRootPath() NN_NOEXCEPT = 0;
    };

    class PathArgumentRoRootExcess : public PathArgumentRoRootExcessBase
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            nnt::fs::api::PathArgumentRoRootExcessBase::SetUp();
            m_RootPath = GetTestRootPath();
            if( m_RootPath[0] == '/' || m_RootPath[0] == '\\' )
            {
                m_RootPath.clear();
            }
            else
            {
                auto pos = m_RootPath.find(':');
                if( nnt::fs::util::String::npos != pos )
                {
                    m_RootPath = m_RootPath.substr(0, pos + 1);
                }
            }
        }

        virtual const nnt::fs::util::String& GetRootPath() NN_NOEXCEPT NN_OVERRIDE
        {
            return m_RootPath;
        }

    private:
        nnt::fs::util::String m_RootPath;
    };

    class PathArgumentRoRootExcessSubDirectory : public PathArgumentRoRootExcessBase
    {
    public:
        virtual ~PathArgumentRoRootExcessSubDirectory() NN_NOEXCEPT NN_OVERRIDE
        {
        }

    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            nnt::fs::api::PathArgumentRoRootExcessBase::SetUp();
            m_SubDirectoryPath = GetTestRootPath() + "/PathArgumentRo/RootExcess";

            size_t firstIndex = m_SubDirectoryPath.find(':');
            if( nnt::fs::util::String::npos == firstIndex )
            {
                firstIndex = 0;
            }
            else
            {
                ++firstIndex;
            }

            m_RootPath = m_SubDirectoryPath;
            for( auto pos = nnt::fs::util::String::npos; ; --pos )
            {
                m_RootPath += "/..";
                pos = m_SubDirectoryPath.find_last_of("\\/", pos);
                if( pos == firstIndex || pos == nnt::fs::util::String::npos )
                {
                    break;
                }
            }

            nn::fs::DirectoryEntryType directoryEntryType;
            NNT_ASSERT_RESULT_SUCCESS(GetFs().GetEntryType(&directoryEntryType, m_RootPath.c_str()));
        }

        virtual const nnt::fs::util::String& GetRootPath() NN_NOEXCEPT NN_OVERRIDE
        {
            return m_RootPath;
        }

    private:
        nnt::fs::util::String m_SubDirectoryPath;
        nnt::fs::util::String m_RootPath;
    };

    typedef PathArgumentRoLengthMax PathArgumentRoMultiBytePath;

#if 0
    TEST_F(PathArgumentRoLengthMax, OpenFile)
    {
        String rootPath = GetTestRootPath().append("/PathArgumentRo/");
        String fileName = GenerateEntryName(rootPath, m_FilePathLengthMax, String("file_"));
        String filePath = rootPath + fileName;
        std::unique_ptr<ITestFile> file;
        NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenFile(&file, filePath.c_str(), OpenMode_Read));
        file.reset(nullptr);
    }

    TEST_F(PathArgumentRoLengthMax, OpenDirectory)
    {
        String rootPath = GetTestRootPath().append("/PathArgumentRo/");
        String directoryName = GenerateEntryName(rootPath, m_DirectoryPathLengthMax, String("dir_"));
        String directoryPath = rootPath + directoryName;
        std::unique_ptr<ITestDirectory> dir;
        NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenDirectory(&dir, directoryPath.c_str(), OpenDirectoryMode_File));
        dir.reset(nullptr);
    }

    TEST_F(PathArgumentRoLengthMax, GetEntryType)
    {
        String rootPath = GetTestRootPath().append("/PathArgumentRo/");
        {
            String fileName = GenerateEntryName(rootPath, m_FilePathLengthMax, String("file_"));
            String filePath = rootPath + fileName;

            DirectoryEntryType directoryEntryType;
            NNT_EXPECT_RESULT_SUCCESS(GetFs().GetEntryType(&directoryEntryType, filePath.c_str()));
            EXPECT_EQ(directoryEntryType, DirectoryEntryType_File);
        }
        {
            String directoryName = GenerateEntryName(rootPath, m_DirectoryPathLengthMax, String("dir_"));
            String directoryPath = rootPath + directoryName;
            DirectoryEntryType directoryEntryType;
            NNT_EXPECT_RESULT_SUCCESS(GetFs().GetEntryType(&directoryEntryType, directoryPath.c_str()));
        }
    }
#endif
    TEST_F(PathArgumentRoMultiBytePath, OpenFile)
    {
        NNT_FS_UTIL_SKIP_TEST_UNLESS(GetFsAttribute()->isSupportedMultiBytePath);
        String rootPath = GetTestRootPath().append("/PathArgumentRo/");
        String fileName = String(Utf8String) + ".file";
        String filePath = rootPath + fileName;
        std::unique_ptr<ITestFile> file;
        NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenFile(&file, filePath.c_str(), OpenMode_Read));
    }

    TEST_F(PathArgumentRoMultiBytePath, OpenDirectory)
    {
        NNT_FS_UTIL_SKIP_TEST_UNLESS(GetFsAttribute()->isSupportedMultiBytePath);
        String rootPath = GetTestRootPath().append("/PathArgumentRo/");
        String directoryName = String(Utf8String);
        String directoryPath = rootPath + directoryName;
        std::unique_ptr<ITestDirectory> directory;
        NNT_EXPECT_RESULT_SUCCESS(
            GetFs().OpenDirectory(&directory, directoryPath.c_str(), OpenDirectoryMode_File));
    }

    TEST_F(PathArgumentRoMultiBytePath, GetEntryType)
    {
        NNT_FS_UTIL_SKIP_TEST_UNLESS(GetFsAttribute()->isSupportedMultiBytePath);
        String rootPath = GetTestRootPath().append("/PathArgumentRo/");
        {
            String fileName = String(Utf8String) + ".file";
            String filePath = rootPath + fileName;

            DirectoryEntryType directoryEntryType;
            NNT_EXPECT_RESULT_SUCCESS(GetFs().GetEntryType(&directoryEntryType, filePath.c_str()));
            EXPECT_EQ(directoryEntryType, DirectoryEntryType_File);
        }
        {
            String directoryName = String(Utf8String);
            String directoryPath = rootPath + directoryName;
            DirectoryEntryType directoryEntryType;
            NNT_EXPECT_RESULT_SUCCESS(
                GetFs().GetEntryType(&directoryEntryType, directoryPath.c_str()));
            EXPECT_EQ(directoryEntryType, DirectoryEntryType_Directory);
        }
    }

    void PathArgumentRoRootExcessBase::DoTest() NN_NOEXCEPT
    {
        auto rootPath = GetRootPath();
        NN_LOG("Fs Root Path : '%s'\n", rootPath.c_str());

        auto mode = nn::fs::OpenDirectoryMode_All;
        if( GetFsType() == nnt::fs::api::FileSystemType_HostFileSystem && rootPath.length() > 1 && rootPath[1] == ':' )
        {
            {
                std::unique_ptr<ITestDirectory> pDir;
                NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenDirectory(&pDir, (rootPath + "/../").c_str(), mode));
            }
            {
                std::unique_ptr<ITestDirectory> pDir;
                NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenDirectory(&pDir, (rootPath + "/../../").c_str(), mode));
            }
            {
                std::unique_ptr<ITestDirectory> pDir;
                NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenDirectory(&pDir, (rootPath + "/../../../").c_str(), mode));
            }
        }
        else if( GetFsAttribute()->isRootExcessFailureResultPathNotFound )
        {
            std::unique_ptr<ITestDirectory> pDir;
            NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, GetFs().OpenDirectory(&pDir, (rootPath + "/../").c_str(), mode));
            NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, GetFs().OpenDirectory(&pDir, (rootPath + "/../../").c_str(), mode));
            NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, GetFs().OpenDirectory(&pDir, (rootPath + "/../../../").c_str(), mode));
        }
        else
        {
            std::unique_ptr<ITestDirectory> pDir;
            NNT_EXPECT_RESULT_FAILURE(ResultDirectoryUnobtainable, GetFs().OpenDirectory(&pDir, (rootPath + "/../").c_str(), mode));
            NNT_EXPECT_RESULT_FAILURE(ResultDirectoryUnobtainable, GetFs().OpenDirectory(&pDir, (rootPath + "/../../").c_str(), mode));
            NNT_EXPECT_RESULT_FAILURE(ResultDirectoryUnobtainable, GetFs().OpenDirectory(&pDir, (rootPath + "/../../../").c_str(), mode));
        }
    }

    //! @brief ルートの上位ディレクトリが OpenDirectory で見えないこと
    TEST_F(PathArgumentRoRootExcess, OpenDirectory)
    {
        DoTest();
    }

    //! @brief ルートの上位ディレクトリが OpenDirectory で見えないこと
    TEST_F(PathArgumentRoRootExcessSubDirectory, OpenDirectory)
    {
        DoTest();
    }
}}}
