﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/fsApi/testFs_Api.h>

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

namespace nnt { namespace fs { namespace api {
    void LoadWriteOptionTests() NN_NOEXCEPT
    {
        return;
    }

    class WriteOptionTest : public CleanupFileSystemTestFixture
    {
    protected:

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            CleanupFileSystemTestFixture::SetUp();
            m_FileName = GetTestRootPath().append("/test.file");
        }
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            CleanupFileSystemTestFixture::TearDown();
        }

        const String& GetFileName() NN_NOEXCEPT
        {
            return m_FileName;
        }
    private:
        String m_FileName;
    };

    //!< @brief WriteOption のフラグ指定・未指定時の挙動が正しく動作していること
    TEST_F(WriteOptionTest, WriteFile)
    {
        const int FileSize = 32;
        const int BufferSize = 32;
        auto writeBuffer = AllocateBuffer(BufferSize);
        auto readBuffer = AllocateBuffer(BufferSize);

        FillBufferWithRandomValue(writeBuffer.get(), BufferSize);

        {
            // WriteOption のフラグ(0)指定時の挙動が正しく動作していること
            const WriteOption option = WriteOption::MakeValue(0);
            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(GetFileName().c_str(), FileSize));

            std::unique_ptr<ITestFile> file;
            NNT_ASSERT_RESULT_SUCCESS(
                GetFs().OpenFile(
                    &file,
                    GetFileName().c_str(),
                    static_cast<OpenMode>(OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend)));

            NNT_ASSERT_RESULT_SUCCESS(file->Write(0, writeBuffer.get(), FileSize, option));
            NNT_EXPECT_RESULT_SUCCESS(file->Flush());

            size_t readSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(
                file->Read(&readSize, 0, readBuffer.get(), BufferSize, ReadOption()));
            ASSERT_EQ(readSize, BufferSize);
            NNT_FS_UTIL_ASSERT_MEMCMPEQ(writeBuffer.get(), readBuffer.get(), BufferSize);

            file.reset(nullptr);
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(GetFileName().c_str()));
        }

        {
            // WriteOption()オプションとWriteOption のフラグ(0)指定オプションが一致していること
            const WriteOption option = WriteOption::MakeValue(0);
            const WriteOption compareOption = WriteOption();
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(&option, &compareOption, sizeof(WriteOption));
        }

        {
            // WriteOption のフラグ(WriteOptionFlag_Flush)指定時の挙動が正しく動作していること
            const WriteOption option = WriteOption::MakeValue(WriteOptionFlag_Flush);
            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(GetFileName().c_str(), FileSize));

            std::unique_ptr<ITestFile> file;
            NNT_ASSERT_RESULT_SUCCESS(
                GetFs().OpenFile(
                    &file,
                    GetFileName().c_str(),
                    static_cast<OpenMode>(OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend)));

            NNT_ASSERT_RESULT_SUCCESS(file->Write(0, writeBuffer.get(), FileSize, option));

            size_t readSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(
                file->Read(&readSize, 0, readBuffer.get(), BufferSize, ReadOption()));
            ASSERT_EQ(readSize, BufferSize);
            NNT_FS_UTIL_ASSERT_MEMCMPEQ(writeBuffer.get(), readBuffer.get(), BufferSize);

            file.reset(nullptr);
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(GetFileName().c_str()));
        }

        {
            // 0 バイト書き込み時の WriteOption のフラグ(WriteOptionFlag_Flush)指定時の挙動が正しく動作していること
            const WriteOption optionZero = WriteOption::MakeValue(0);
            const WriteOption optionFlush = WriteOption::MakeValue(WriteOptionFlag_Flush);
            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(GetFileName().c_str(), FileSize));

            std::unique_ptr<ITestFile> file;
            NNT_ASSERT_RESULT_SUCCESS(
                GetFs().OpenFile(
                    &file,
                    GetFileName().c_str(),
                    static_cast<OpenMode>(OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend)));

            NNT_ASSERT_RESULT_SUCCESS(file->Write(0, writeBuffer.get(), FileSize, optionZero));
            NNT_ASSERT_RESULT_SUCCESS(file->Write(0, writeBuffer.get(), 0, optionFlush));

            size_t readSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(
                file->Read(&readSize, 0, readBuffer.get(), BufferSize, ReadOption()));
            ASSERT_EQ(readSize, BufferSize);
            NNT_FS_UTIL_ASSERT_MEMCMPEQ(writeBuffer.get(), readBuffer.get(), BufferSize);

            file.reset(nullptr);
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(GetFileName().c_str()));
        }

    }

}}}
