﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/fsApi/testFs_Api.h>

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

namespace {
    const int FileSize = 32;

    const OpenMode OpenFileModes[] =
    {
        OpenMode_Read,
        OpenMode_Write,
        static_cast<OpenMode>(OpenMode_Read | OpenMode_Write),
        static_cast<OpenMode>(OpenMode_Read | OpenMode_AllowAppend),
        static_cast<OpenMode>(OpenMode_Write | OpenMode_AllowAppend),
        static_cast<OpenMode>(OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend)
    };

    const OpenDirectoryMode OpenDirectoryModes[] =
    {
        OpenDirectoryMode_Directory,
        OpenDirectoryMode_File,
        OpenDirectoryMode_All
    };
}

namespace nnt { namespace fs { namespace api {
    void LoadTargetExistenceTests() NN_NOEXCEPT
    {
        return;
    }

    class TargetExistenceFile : public CleanupFileSystemTestFixture
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            CleanupFileSystemTestFixture::SetUp();
            m_ExistFileName = GetTestRootPath().append("/test.file");
            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(m_ExistFileName.c_str(), FileSize));
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(m_ExistFileName.c_str()));
            CleanupFileSystemTestFixture::TearDown();
        }

        const String& GetExistFileName() NN_NOEXCEPT
        {
            return m_ExistFileName;
        }
    private:
        String m_ExistFileName;
    };

    class TargetExistenceDirectory : public CleanupFileSystemTestFixture
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            CleanupFileSystemTestFixture::SetUp();
            m_ExistDirectoryName = GetTestRootPath().append("/testDir");
            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(m_ExistDirectoryName.c_str()));
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            NNT_ASSERT_RESULT_SUCCESS(GetFs().DeleteDirectory(m_ExistDirectoryName.c_str()));
            CleanupFileSystemTestFixture::TearDown();
        }

        const String& GetExistDirectoryName() NN_NOEXCEPT
        {
            return m_ExistDirectoryName;
        }
    private:
        String m_ExistDirectoryName;
    };

    class TargetExistenceFileInDirectory : public CleanupFileSystemTestFixture
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            CleanupFileSystemTestFixture::SetUp();
            m_ExistDirectoryName = GetTestRootPath().append("/testDir");
            m_ExistFileName = GetTestRootPath().append("/testDir/test.file");
            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(m_ExistDirectoryName.c_str()));
            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(m_ExistFileName.c_str(), FileSize));
        }
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(m_ExistFileName.c_str()));
            NNT_ASSERT_RESULT_SUCCESS(GetFs().DeleteDirectory(m_ExistDirectoryName.c_str()));
            CleanupFileSystemTestFixture::TearDown();
        }

        const String& GetExistDirectoryName() NN_NOEXCEPT
        {
            return m_ExistDirectoryName;
        }
    private:
        String m_ExistFileName;
        String m_ExistDirectoryName;
    };

    class TargetExistenceNonExistent : public CleanupFileSystemTestFixture
    {
    };

    class TargetExistenceInvalidNotDir : public TargetExistenceFile
    {
    };

    class TargetExistenceInvalidNotFile : public TargetExistenceDirectory
    {
    };

    //! @brief 同一パスに同名ファイルを作成し、ResultPathAlreadyExists が返却されることを確認する
    TEST_F(TargetExistenceFile, CreateFile)
    {
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathAlreadyExists, GetFs().CreateFile(GetExistFileName().c_str(), FileSize));
    }

    //! @brief 同一パスに同名ディレクトリを作成し、ResultPathAlreadyExists が返却されることを確認する
    TEST_F(TargetExistenceDirectory, CreateDirectory)
    {
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathAlreadyExists, GetFs().CreateDirectory(GetExistDirectoryName().c_str()));
    }

    //! @brief 同一パスに別名ファイルを作成し、既存ファイルにリネームした際に ResultPathAlreadyExists が返却されることを確認する
    TEST_F(TargetExistenceFile, RenameFile)
    {
        String fileName = GetTestRootPath().append("/rename.file");
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(fileName.c_str(), FileSize));
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathAlreadyExists,
            GetFs().RenameFile(fileName.c_str(), GetExistFileName().c_str()));
        NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(fileName.c_str()));
    }

    //! @brief 同一パスに別名ディレクトリを作成し、既存ディレクトリにリネームした際に ResultPathAlreadyExists が返却されることを確認する
    TEST_F(TargetExistenceDirectory, RenameDirectory)
    {
        String directoryName = GetTestRootPath().append("/rename");
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(directoryName.c_str()));
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathAlreadyExists,
            GetFs().RenameDirectory(directoryName.c_str(), GetExistDirectoryName().c_str()));
        NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteDirectory(directoryName.c_str()));
    }

    //! @brief 既存のファイルパスでディレクトリを作成し、ResultPathAlreadyExists が返却されることを確認する
    TEST_F(TargetExistenceFile, CreateDirectory)
    {
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathAlreadyExists, GetFs().CreateDirectory(GetExistFileName().c_str()));
    }

    //! @brief 既存のディレクトリパスでファイルを作成し、ResultPathAlreadyExists が返却されることを確認する
    TEST_F(TargetExistenceDirectory, CreateFile)
    {
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathAlreadyExists, GetFs().CreateFile(GetExistDirectoryName().c_str(), FileSize));
    }

    //! @brief ディレクトリを作成し既存のファイルパスにリネームした際に ResultPathAlreadyExists が返却されることを確認する
    TEST_F(TargetExistenceFile, RenameDirectory)
    {
        String directoryName = GetTestRootPath().append("/rename");
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(directoryName.c_str()));
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathAlreadyExists,
            GetFs().RenameDirectory(directoryName.c_str(), GetExistFileName().c_str()));
        NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteDirectory(directoryName.c_str()));
    }

    //! @brief ファイルを作成し既存のディレクトリパスにリネームした際に ResultPathAlreadyExists が返却されることを確認する
    TEST_F(TargetExistenceDirectory, RenameFile)
    {
        String fileName = GetTestRootPath().append("/rename.file");
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(fileName.c_str(), FileSize));
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathAlreadyExists,
            GetFs().RenameFile(fileName.c_str(), GetExistDirectoryName().c_str()));
        NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(fileName.c_str()));
    }

    //! @brief ディレクトリを作成し、その中にファイルを作成し、ディレクトリを削除。ResultDirectoryNotEmpty が返却されることを確認する
    TEST_F(TargetExistenceFileInDirectory, DeleteDirectory)
    {
        NNT_EXPECT_RESULT_FAILURE(
            ResultDirectoryNotEmpty, GetFs().DeleteDirectory(GetExistDirectoryName().c_str()));
    }

    //! @brief 存在しないパスに対して、ファイルを作成し、ResultPathNotFound が返却されることを確認する
    TEST_F(TargetExistenceNonExistent, CreateFile)
    {
        String nonExistentFileName = GetTestRootPath().append("/noexist/test.file");
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathNotFound, GetFs().CreateFile(nonExistentFileName.c_str(), FileSize));
    }

    //! @brief 存在しないパスに対して、ディレクトリを作成し、ResultPathNotFound が返却されることを確認する
    TEST_F(TargetExistenceNonExistent, CreateDirectory)
    {
        String nonExistentDirectoryName = GetTestRootPath().append("/noexist/test");
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathNotFound, GetFs().CreateDirectory(nonExistentDirectoryName.c_str()));
    }

    //! @brief 存在しないファイルを削除し、ResultPathNotFound が返却されることを確認する
    TEST_F(TargetExistenceNonExistent, DeleteFile)
    {
        String nonExistentFileName = GetTestRootPath().append("/noexist.file");
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathNotFound, GetFs().DeleteFile(nonExistentFileName.c_str()));
    }

    //! @brief 存在しないディレクトリを削除し、ResultPathNotFound が返却されることを確認する
    TEST_F(TargetExistenceNonExistent, DeleteDirectory)
    {
        String nonExistentDirectoryName = GetTestRootPath().append("/noexist");
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathNotFound, GetFs().DeleteDirectory(nonExistentDirectoryName.c_str()));
    }

    //! @brief 存在しないディレクトリを再帰的に削除し、ResultPathNotFound が返却されることを確認する
    TEST_F(TargetExistenceNonExistent, DeleteDirectoryRecursively)
    {
        String nonExistentDirectoryName = GetTestRootPath().append("/noexist");
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathNotFound, GetFs().DeleteDirectoryRecursively(nonExistentDirectoryName.c_str()));
    }

    //! @brief 存在しないディレクトリの EntryType を取得し、ResultPathNotFound が返却されることを確認する
    TEST_F(TargetExistenceNonExistent, GetEntryType)
    {
        String nonExistentEntryName = GetTestRootPath().append("/noexist");
        DirectoryEntryType directoryEntryType;
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathNotFound,
            GetFs().GetEntryType(&directoryEntryType, nonExistentEntryName.c_str()));
    }

    //! @brief 存在しないファイルに対してリネームし、ResultPathNotFound が返却されることを確認する
    //!        存在しないファイルに対して既存のファイル名にリネームし、ResultPathNotFound が返却されることを確認する
    TEST_F(TargetExistenceNonExistent, RenameFile)
    {
        String nonExistentFileName = GetTestRootPath().append("/noexist.file");
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathNotFound,
            GetFs().RenameFile(nonExistentFileName.c_str(), nonExistentFileName.c_str()));

        String fileName = GetTestRootPath().append("/test.file");
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(fileName.c_str(), FileSize));
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathNotFound, GetFs().RenameFile(nonExistentFileName.c_str(), fileName.c_str()));
        NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(fileName.c_str()));
    }

    //! @brief 存在しないディレクトリに対してリネームし、ResultPathNotFound が返却されることを確認する
    //!        存在しないディレクトリに対して既存のディレクトリ名にリネームし、ResultPathNotFound が返却されることを確認する
    TEST_F(TargetExistenceNonExistent, RenameDirectory)
    {
        String nonExistentDirectoryName = GetTestRootPath().append("/noexist");
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathNotFound,
            GetFs().RenameDirectory(
                nonExistentDirectoryName.c_str(), nonExistentDirectoryName.c_str()));

        String directoryName = GetTestRootPath().append("/test");
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(directoryName.c_str()));
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathNotFound,
            GetFs().RenameDirectory(nonExistentDirectoryName.c_str(), directoryName.c_str()));
        NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteDirectory(directoryName.c_str()));
    }

    //! @brief 存在しないファイルをそれぞれのモードで開き、ResultPathNotFound が返却されることを確認する
    TEST_F(TargetExistenceNonExistent, OpenFile)
    {
        String nonExistentFileName = GetTestRootPath().append("/noexist.file");
        for( auto openMode : OpenFileModes )
        {
            std::unique_ptr<ITestFile> file;
            NNT_EXPECT_RESULT_FAILURE(
                ResultPathNotFound, GetFs().OpenFile(&file, nonExistentFileName.c_str(), openMode));
        }
    }

    //! @brief 存在しないディレクトリをそれぞれのモードで開き、ResultPathNotFound が返却されることを確認する
    TEST_F(TargetExistenceNonExistent, OpenDirectory)
    {
        String nonExistentDirectoryName = GetTestRootPath().append("/noexist");
        for( auto openDirectoryMode : OpenDirectoryModes )
        {
            std::unique_ptr<ITestDirectory> directory;
            NNT_EXPECT_RESULT_FAILURE(
                ResultPathNotFound,
                GetFs().OpenDirectory(
                    &directory, nonExistentDirectoryName.c_str(), openDirectoryMode));
        }
    }

    //! @brief 作成済のファイルをディレクトリとしてパスを作成し、ファイルを作成、ResultPathNotFound が返却されることを確認する
    TEST_F(TargetExistenceInvalidNotDir, CreateFile)
    {
        String invalidFileName = GetExistFileName();
        invalidFileName.append("/test.file");
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathNotFound, GetFs().CreateFile(invalidFileName.c_str(), FileSize));
    }

    //! @brief 作成済のファイルをディレクトリとしてパスを作成し、ディレクトリを作成、ResultPathNotFound が返却されることを確認する
    TEST_F(TargetExistenceInvalidNotDir, CreateDirectory)
    {
        String invalidDirName = GetExistFileName();
        invalidDirName.append("/test");
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathNotFound, GetFs().CreateDirectory(invalidDirName.c_str()));
    }

    //! @brief 作成済のディレクトリをファイルとして削除し、ResultPathNotFound が返却されることを確認する
    TEST_F(TargetExistenceInvalidNotFile, DeleteFile)
    {
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathNotFound, GetFs().DeleteFile(GetExistDirectoryName().c_str()));
    }

    //! @brief 作成済のファイルをディレクトリとして削除し、ResultPathNotFound が返却されることを確認する
    TEST_F(TargetExistenceInvalidNotDir, DeleteDirectory)
    {
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathNotFound, GetFs().DeleteDirectory(GetExistFileName().c_str()));
    }

    //! @brief 作成済のファイルをディレクトリとして再帰的に削除し、ResultPathNotFound が返却されることを確認する
    TEST_F(TargetExistenceInvalidNotDir, DeleteDirectoryRecursively)
    {
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathNotFound, GetFs().DeleteDirectoryRecursively(GetExistFileName().c_str()));
    }

    //! @brief 作成済のファイルをディレクトリとしてパスを作成し、EntryTypeを取得、ResultPathNotFound が返却されることを確認する
    TEST_F(TargetExistenceInvalidNotDir, GetEntryType)
    {
        String invalidEntryName = GetExistFileName();
        invalidEntryName.append("/test");
        DirectoryEntryType directoryEntryType;
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathNotFound,
            GetFs().GetEntryType(&directoryEntryType, invalidEntryName.c_str()));
    }

    //! @brief 作成済のディレクトリをファイルとしてリネームし、ResultPathNotFound が返却されることを確認する
    TEST_F(TargetExistenceInvalidNotFile, RenameFile)
    {
        String renamedFileName = GetTestRootPath().append("/rename.file");
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathNotFound,
            GetFs().RenameFile(GetExistDirectoryName().c_str(), renamedFileName.c_str()));
    }

    //! @brief 作成済のファイルをディレクトリとしてリネームし、ResultPathNotFound が返却されることを確認する
    TEST_F(TargetExistenceInvalidNotDir, RenameDirectory)
    {
        String renamedDirName = GetTestRootPath().append("/rename");
        NNT_EXPECT_RESULT_FAILURE(
            ResultPathNotFound,
            GetFs().RenameDirectory(GetExistFileName().c_str(), renamedDirName.c_str()));
    }

    //! @brief 作成済のディレクトリをファイルとしてそれぞれのモードで開き、ResultPathNotFound が返却されることを確認する
    TEST_F(TargetExistenceInvalidNotFile, OpenFile)
    {
        for( auto openMode : OpenFileModes )
        {
            std::unique_ptr<ITestFile> file;
            NNT_EXPECT_RESULT_FAILURE(
                ResultPathNotFound,
                GetFs().OpenFile(&file, GetExistDirectoryName().c_str(), openMode));
        }
    }

    //! @brief 作成済のファイルをディレクトリとしてそれぞれのモードで開き、ResultPathNotFound が返却されることを確認する
    TEST_F(TargetExistenceInvalidNotDir, OpenDirectory)
    {
        for( auto openDirectoryMode : OpenDirectoryModes )
        {
            std::unique_ptr<ITestDirectory> directory;
            NNT_EXPECT_RESULT_FAILURE(
                ResultPathNotFound,
                GetFs().OpenDirectory(&directory, GetExistFileName().c_str(), openDirectoryMode));
        }
    }
}}}
