﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_ResultPrivate.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt/fsApi/testFs_Api.h>

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

namespace nnt { namespace fs { namespace api {
    void LoadPointerArgumentTests() NN_NOEXCEPT
    {
        return;
    }

    namespace {
        const OpenMode OpenFileModes[] =
        {
            OpenMode_Read,
            OpenMode_Write,
            static_cast<OpenMode>(OpenMode_Read  | OpenMode_Write),
            static_cast<OpenMode>(OpenMode_Read  | OpenMode_AllowAppend),
            static_cast<OpenMode>(OpenMode_Write | OpenMode_AllowAppend),
            static_cast<OpenMode>(OpenMode_Read  | OpenMode_Write | OpenMode_AllowAppend)
        };

        const OpenDirectoryMode OpenDirectoryModes[] =
        {
            OpenDirectoryMode_Directory,
            OpenDirectoryMode_File,
            OpenDirectoryMode_All
        };
    }

    class PointerArgumentNull : public CleanupFileSystemTestFixture
    {
    protected:
        PointerArgumentNull() NN_NOEXCEPT
        {
            CheckGlobalNewDeleteFlagTestFixture::DisableCheck();
            CheckMemoryLeakFixture::DisableCheck();
        }

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            CleanupFileSystemTestFixture::SetUp();
            m_FileName = GetTestRootPath().append("/test.file");
            m_DirectoryName = GetTestRootPath().append("/test");
            m_NewFileName = GetTestRootPath().append("/testNew.file");
            m_NewDirectoryName = GetTestRootPath().append("/testNew");
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            CleanupFileSystemTestFixture::TearDown();
        }

        const String& GetFileName() NN_NOEXCEPT
        {
            return m_FileName;
        }

        const String& GetDirectoryName() NN_NOEXCEPT
        {
            return m_DirectoryName;
        }

        const String& GetNewFileName() NN_NOEXCEPT
        {
            return m_NewFileName;
        }

        const String& GetNewDirectoryName() NN_NOEXCEPT
        {
            return m_NewDirectoryName;
        }

    private:
        String m_FileName;
        String m_DirectoryName;
        String m_NewFileName;
        String m_NewDirectoryName;
    };

    //! @brief ファイルに対して GetEntryType の引数にNULLを与え、失敗すること
    TEST_F(PointerArgumentNull, GetEntryType)
    {
        DirectoryEntryType directoryEntryType;
        const int FileSize = 32;

        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(GetFileName().c_str(), FileSize));
        NN_UTIL_SCOPE_EXIT
        {
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(GetFileName().c_str()));
        };

        NNT_EXPECT_RESULT_FAILURE(
            ResultNullptrArgument, GetFs().GetEntryType(nullptr, GetFileName().c_str()));
        NNT_EXPECT_RESULT_FAILURE(
            ResultInvalidPath,     GetFs().GetEntryType(&directoryEntryType,  nullptr));
    }

    //! @brief GetFreeSpaceSize の引数にNULLを与え、失敗すること
    TEST_F(PointerArgumentNull, GetFreeSpaceSize)
    {
        NNT_EXPECT_RESULT_FAILURE(
            ResultNullptrArgument, GetFs().GetFreeSpaceSize(nullptr, GetDirectoryName().c_str()));

        int64_t size = 0;
        NNT_EXPECT_RESULT_FAILURE(
            ResultInvalidPath, GetFs().GetFreeSpaceSize(&size, nullptr));
    }

    //! @brief GetTotalSpaceSize の引数にNULLを与え、失敗すること
    TEST_F(PointerArgumentNull, GetTotalSpaceSize)
    {
        NNT_EXPECT_RESULT_FAILURE(
            ResultNullptrArgument, GetFs().GetTotalSpaceSize(nullptr, GetDirectoryName().c_str()));

        int64_t size = 0;
        NNT_EXPECT_RESULT_FAILURE(
            ResultInvalidPath, GetFs().GetTotalSpaceSize(&size, nullptr));
    }

    //! @brief CreateFile の引数にNULLを与え、失敗すること
    TEST_F(PointerArgumentNull, CreateFile)
    {
        NNT_EXPECT_RESULT_FAILURE(ResultInvalidPath, GetFs().CreateFile(nullptr, 0));
    }

    //! @brief DeleteFile の引数にNULLを与え、失敗すること
    TEST_F(PointerArgumentNull, DeleteFile)
    {
        NNT_EXPECT_RESULT_FAILURE(ResultInvalidPath, GetFs().DeleteFile(nullptr));
    }

    //! @brief RenameFile の引数にNULLを与え、失敗すること
    TEST_F(PointerArgumentNull, RenameFile)
    {
        const int FileSize = 32;
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(GetFileName().c_str(), FileSize));
        NN_UTIL_SCOPE_EXIT
        {
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(GetFileName().c_str()));
        };

        NNT_EXPECT_RESULT_FAILURE(
            ResultInvalidPath, GetFs().RenameFile(nullptr, GetNewFileName().c_str()));
        NNT_EXPECT_RESULT_FAILURE(
            ResultInvalidPath, GetFs().RenameFile(GetFileName().c_str(), nullptr));
    }

    //! @brief OpenFile の引数にNULLを与え、失敗すること
    TEST_F(PointerArgumentNull, OpenFile)
    {
        std::unique_ptr<ITestFile> file;
        const int FileSize = 32;

        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(GetFileName().c_str(), FileSize));
        NN_UTIL_SCOPE_EXIT
        {
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(GetFileName().c_str()));
        };

        for( auto openMode : OpenFileModes )
        {
            NNT_EXPECT_RESULT_FAILURE(
                ResultNullptrArgument, GetFs().OpenFile(nullptr, GetFileName().c_str(), openMode));
            NNT_EXPECT_RESULT_FAILURE(
                ResultInvalidPath, GetFs().OpenFile(&file, nullptr, openMode));
        }
    }

    //! @brief ファイルの読み込みをする際の引数にNULLを与え、失敗すること
    TEST_F(PointerArgumentNull, ReadFile)
    {
        const int FileSize = 32;
        char buffer[FileSize + 1] = {0};
        size_t readSize = FileSize;

        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(GetFileName().c_str(), FileSize));
        NN_UTIL_SCOPE_EXIT
        {
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(GetFileName().c_str()));
        };

        for( auto openMode : OpenFileModes )
        {
            std::unique_ptr<ITestFile> file;
            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, GetFileName().c_str(), openMode));

            NNT_EXPECT_RESULT_FAILURE(
                ResultNullptrArgument, file->Read(nullptr, 0, buffer, FileSize, ReadOption()));
            NNT_EXPECT_RESULT_FAILURE(
                ResultNullptrArgument, file->Read(nullptr, 0, buffer, FileSize));

            NNT_EXPECT_RESULT_FAILURE(
                ResultNullptrArgument, file->Read(0, nullptr, FileSize));
            NNT_EXPECT_RESULT_FAILURE(
                ResultNullptrArgument, file->Read(0, nullptr, FileSize, ReadOption()));
            NNT_EXPECT_RESULT_FAILURE(
                ResultNullptrArgument, file->Read(&readSize, 0, nullptr, FileSize, ReadOption()));
            NNT_EXPECT_RESULT_FAILURE(
                ResultNullptrArgument, file->Read(&readSize, 0, nullptr, FileSize));
        }
    }

    //! @brief ファイルの書き込みをする際の引数にNULLを与え、失敗すること
    TEST_F(PointerArgumentNull, WriteFile)
    {
        const int FileSize = 32;

        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(GetFileName().c_str(), FileSize));
        NN_UTIL_SCOPE_EXIT
        {
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(GetFileName().c_str()));
        };

        for( auto openMode : OpenFileModes )
        {
            std::unique_ptr<ITestFile> file;
            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, GetFileName().c_str(), openMode));

            NNT_EXPECT_RESULT_FAILURE(
                ResultNullptrArgument, file->Write(0, nullptr, FileSize, WriteOption()));
        }
    }

    //! @brief ファイルに対して GetSize の引数にNULLを与え、失敗すること
    TEST_F(PointerArgumentNull, GetSize)
    {
        const int FileSize = 32;

        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(GetFileName().c_str(), FileSize));
        NN_UTIL_SCOPE_EXIT
        {
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(GetFileName().c_str()));
        };

        for( auto openMode : OpenFileModes )
        {
            std::unique_ptr<ITestFile> file;
            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, GetFileName().c_str(), openMode));
            NNT_EXPECT_RESULT_FAILURE(ResultNullptrArgument, file->GetSize(nullptr));
        }
    }

    //! @brief CreateDirectory の引数にNULLを与え、失敗すること
    TEST_F(PointerArgumentNull, CreateDirectory)
    {
        NNT_EXPECT_RESULT_FAILURE(ResultInvalidPath, GetFs().CreateDirectory(nullptr));
    }

    //! @brief DeleteDirectory の引数にNULLを与え、失敗すること
    TEST_F(PointerArgumentNull, DeleteDirectory)
    {
        NNT_EXPECT_RESULT_FAILURE(ResultInvalidPath, GetFs().DeleteDirectory(nullptr));
    }

    //! @brief DeleteDirectoryRecursively の引数にNULLを与え、失敗すること
    TEST_F(PointerArgumentNull, DeleteDirectoryRecursively)
    {
        NNT_EXPECT_RESULT_FAILURE(ResultInvalidPath, GetFs().DeleteDirectoryRecursively(nullptr));
    }

    //! @brief RenameDirectory の引数にNULLを与え、失敗すること
    TEST_F(PointerArgumentNull, RenameDirectory)
    {
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(GetDirectoryName().c_str()));
        NN_UTIL_SCOPE_EXIT
        {
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteDirectory(GetDirectoryName().c_str()));
        };

        NNT_EXPECT_RESULT_FAILURE(
            ResultInvalidPath, GetFs().RenameDirectory(nullptr, GetNewDirectoryName().c_str()));
        NNT_EXPECT_RESULT_FAILURE(
            ResultInvalidPath, GetFs().RenameDirectory(GetDirectoryName().c_str(), nullptr));
    }

    //! @brief OpenDirectory の引数にNULLを与え、失敗すること
    TEST_F(PointerArgumentNull, OpenDirectory)
    {
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(GetDirectoryName().c_str()));
        NN_UTIL_SCOPE_EXIT
        {
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteDirectory(GetDirectoryName().c_str()));
        };

        for( auto openDirecotryMode : OpenDirectoryModes )
        {
            std::unique_ptr<ITestDirectory> directory;
            NNT_EXPECT_RESULT_FAILURE(
                ResultNullptrArgument,
                GetFs().OpenDirectory(nullptr, GetDirectoryName().c_str(), openDirecotryMode));
            NNT_EXPECT_RESULT_FAILURE(
                ResultInvalidPath,
                GetFs().OpenDirectory(&directory, nullptr, openDirecotryMode));
        }
    }

    //! @brief ディレクトリに対して ReadDirectory の引数にNULLを与え、失敗すること
    TEST_F(PointerArgumentNull, ReadDirectory)
    {
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(GetDirectoryName().c_str()));
        NN_UTIL_SCOPE_EXIT
        {
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteDirectory(GetDirectoryName().c_str()));
        };

        for( auto openDirecotryMode : OpenDirectoryModes )
        {
            std::unique_ptr<ITestDirectory> directory;
            int64_t entryCount;
            DirectoryEntry directoryEntry[2];

            util::InvalidateVariable(directoryEntry, 2);
            NNT_ASSERT_RESULT_SUCCESS(
                GetFs().OpenDirectory(&directory, GetDirectoryName().c_str(), openDirecotryMode));

            NNT_EXPECT_RESULT_FAILURE(
                ResultNullptrArgument, directory->Read(nullptr, directoryEntry, 1));
            NNT_EXPECT_RESULT_FAILURE(
                ResultNullptrArgument, directory->Read(&entryCount, nullptr, 1));
        }
    }

    //! @brief ディレクトリに対して GetEntryCount の引数にNULLを与え、失敗すること
    TEST_F(PointerArgumentNull, GetEntryCount)
    {
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(GetDirectoryName().c_str()));
        NN_UTIL_SCOPE_EXIT
        {
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteDirectory(GetDirectoryName().c_str()));
        };

        for( auto openDirecotryMode : OpenDirectoryModes )
        {
            std::unique_ptr<ITestDirectory> directory;

            NNT_ASSERT_RESULT_SUCCESS(
                GetFs().OpenDirectory(&directory, GetDirectoryName().c_str(), openDirecotryMode));

            NNT_EXPECT_RESULT_FAILURE(ResultNullptrArgument, directory->GetEntryCount(nullptr));
        }
    }

    //!< @brief QueryRange で NULL を渡すと失敗することを確認する
    TEST_F(PointerArgumentNull, OperateRange_QueryRange)
    {
        const auto FileSize = 1024;

        // ファイル作成
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(GetFileName().c_str(), FileSize));
        NN_UTIL_SCOPE_EXIT
        {
            // ファイルシステムによっては失敗を記憶しているために DeleteFile に失敗する
            (void) GetFs().DeleteFile(GetFileName().c_str());
        };

        // ファイルオープン
        std::unique_ptr<ITestFile> file;

        NNT_ASSERT_RESULT_SUCCESS(
            GetFs().OpenFile(
                &file,
                GetFileName().c_str(),
                static_cast<nn::fs::OpenMode>(
                    nn::fs::OpenMode::OpenMode_Read |
                    nn::fs::OpenMode::OpenMode_Write |
                    nn::fs::OpenMode::OpenMode_AllowAppend)
            )
        );

        if( GetFsAttribute()->isSupportedQueryRange )
        {
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultNullptrArgument,
                file->QueryRange(nullptr, 0, FileSize));
        }
        else
        {
            auto result = file->QueryRange(nullptr, 0, FileSize);
            if( !nn::fs::ResultUnsupportedOperation::Includes(result)
                && !nn::fs::ResultNullptrArgument::Includes(result) )
            {
                NN_LOG("Unexpected result: 0x%08X\n", result.GetInnerValueForDebug());
                FAIL();
            }
        }
    }

}}}
