﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/util/util_FormatString.h>
#include <nnt/fsApi/testFs_Api.h>

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

namespace {
    const int FileSize = 32;
    const char Utf8String1[] = {
        static_cast<char>(-29), static_cast<char>(-127), static_cast<char>(-126), '\0'
    }; // "あ\0" in utf-8
    const char Utf8String2[] = {
        static_cast<char>(-29), static_cast<char>(-127), static_cast<char>(-124), '\0'
    }; // "い\0" in utf-8
}

namespace nnt { namespace fs { namespace api {
    void LoadPathArgumentTests() NN_NOEXCEPT
    {
        return;
    }

    class PathArgumentFixtureBase : public nnt::fs::api::GetFileSystemTestFixture
    {
    protected:
        nnt::fs::util::String GetTestRootPath() NN_NOEXCEPT
        {
            testDirPath = GetFileSystemTestFixture::GetTestRootPath() + "/FsTest";
            return testDirPath;
        }

        static void SetUpTestCase() NN_NOEXCEPT
        {
            std::unique_ptr<TestFileSystemInfo> fileSystemInfo;
            GetTestFileSystemInfo(&fileSystemInfo, 0);
            String rootDirectoryPath = nnt::fs::util::String(fileSystemInfo->rootDirPath).append("/FsTest/");
            NNT_ASSERT_RESULT_SUCCESS(fileSystemInfo->fileSystem->CreateDirectory(rootDirectoryPath.c_str()));
            if( fileSystemInfo->attribute.isSaveFileSystem )
            {
                NNT_ASSERT_RESULT_SUCCESS(fileSystemInfo->fileSystem->Commit());
            }
        }

        static void TearDownTestCase() NN_NOEXCEPT
        {
            std::unique_ptr<TestFileSystemInfo> fileSystemInfo;
            GetTestFileSystemInfo(&fileSystemInfo, 0);
            const String rootDirectoryPath = nnt::fs::util::String(fileSystemInfo->rootDirPath).append("/FsTest/");
            NNT_ASSERT_RESULT_SUCCESS(fileSystemInfo->fileSystem->DeleteDirectoryRecursively(rootDirectoryPath.c_str()));
            if( fileSystemInfo->attribute.isSaveFileSystem )
            {
                NNT_ASSERT_RESULT_SUCCESS(fileSystemInfo->fileSystem->Commit());
            }
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            if( GetFsAttribute()->isSaveFileSystem )
            {
                NNT_ASSERT_RESULT_SUCCESS(GetFs().Commit());
            }
            GetFileSystemTestFixture::TearDown();
        }

        //! @brief 指定したディレクトリ内に存在するエントリが指定したエントリだけかテスト
        void CheckIfOneEntryExist(
            const char* directoryName, const char* entryName, bool isDirectory) NN_NOEXCEPT
        {
            const int pathLengthMax = isDirectory ?
                GetFsAttribute()->directoryPathLengthMax : GetFsAttribute()->filePathLengthMax;
            std::unique_ptr<ITestDirectory> directory;
            NNT_ASSERT_RESULT_SUCCESS(
                GetFs().OpenDirectory(
                    &directory,
                    directoryName,
                    isDirectory ? OpenDirectoryMode_Directory : OpenDirectoryMode_File));
            int64_t readCount = 0;
            const int EntryBufferLength = 1;
            DirectoryEntry directoryEntry[EntryBufferLength];
            NNT_EXPECT_RESULT_SUCCESS(
                directory->Read(&readCount, directoryEntry, EntryBufferLength));
            EXPECT_EQ(1, readCount);
            if (readCount > 0)
            {
                EXPECT_EQ(
                    strnlen(entryName, pathLengthMax),
                    strnlen(directoryEntry[0].name, pathLengthMax));
                EXPECT_TRUE(strncmp(directoryEntry[0].name, entryName, pathLengthMax) == 0);
            }
        }
    private:
        nnt::fs::util::String testDirPath;
    };

    class PathArgumentPathLength : public PathArgumentFixtureBase
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            PathArgumentFixtureBase::SetUp();

            if( !g_IsInitialized )
            {
                const auto filePathLengthMax = GetFsAttribute()->filePathLengthMax;
                const auto directoryPathLengthMax = GetFsAttribute()->directoryPathLengthMax;
                const auto pathLengthMax = std::min(filePathLengthMax, directoryPathLengthMax);

                // パスの最大長の一歩手前まで親ディレクトリを作成
                const String rootDirectoryPath = GetTestRootPath().append("/");
                g_ParentDirectoryPath = rootDirectoryPath;
                g_ParentDirectoryPathLength = g_ParentDirectoryPath.length();
                g_AppendDirectoryName = GetFsAttribute()->isSupportedMultiBytePath
                    ? String(Utf8String1).append("/") : String("a/");
                g_AppendDirectoryNameLength = 0;
                if( GetFsType() == FileSystemType_HostFileSystem )
                {
                    g_AppendDirectoryNameLength = 2;
                }
                else
                {
                    g_AppendDirectoryNameLength = g_AppendDirectoryName.length();
                }
                int fileNameLength = 1;

                while( static_cast<int>(g_ParentDirectoryPathLength)
                    <= pathLengthMax - static_cast<int>(g_AppendDirectoryNameLength) - fileNameLength )
                {
                    g_ParentDirectoryPath.append(g_AppendDirectoryName);
                    g_ParentDirectoryPathLength += g_AppendDirectoryNameLength;
                    NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(g_ParentDirectoryPath.c_str()));
                }
                g_IsInitialized = true;
            }
        }

        static void TearDownTestCase() NN_NOEXCEPT
        {
            {
                std::unique_ptr<TestFileSystemInfo> fileSystemInfo;
                GetTestFileSystemInfo(&fileSystemInfo, 0);

                // 親ディレクトリを削除
                const String rootDirectoryPath = nnt::fs::util::String(fileSystemInfo->rootDirPath).append("/FsTest/");
                const auto rootDirectoryPathLength = rootDirectoryPath.length();
                while( g_ParentDirectoryPathLength > rootDirectoryPathLength )
                {
                    NNT_ASSERT_RESULT_SUCCESS(
                        fileSystemInfo->fileSystem->DeleteDirectoryRecursively(g_ParentDirectoryPath.c_str()));
                    g_ParentDirectoryPathLength -= g_AppendDirectoryNameLength;
                    g_ParentDirectoryPath = g_ParentDirectoryPath.substr(
                            0, g_ParentDirectoryPath.length() - g_AppendDirectoryName.length());
                }
                ASSERT_EQ(g_ParentDirectoryPath, rootDirectoryPath);
                if( fileSystemInfo->attribute.isSaveFileSystem )
                {
                    NNT_ASSERT_RESULT_SUCCESS(fileSystemInfo->fileSystem->Commit());
                }
            }
            g_IsInitialized = false;
            PathArgumentFixtureBase::TearDownTestCase();
        }

        const String& GetParentDirectoryPath() const NN_NOEXCEPT
        {
            return g_ParentDirectoryPath;
        }

        size_t GetParentDirectoryPathLength() const NN_NOEXCEPT
        {
            return g_ParentDirectoryPathLength;
        }

    private:
        static String g_ParentDirectoryPath;
        static size_t g_ParentDirectoryPathLength;
        static String g_AppendDirectoryName;
        static size_t g_AppendDirectoryNameLength;
        static bool g_IsInitialized;
    };

    class PathArgumentRootExcessBase : public nnt::fs::api::CleanupFileSystemTestFixture
    {
    public:
        virtual ~PathArgumentRootExcessBase() NN_NOEXCEPT NN_OVERRIDE
        {
        }

    protected:
        void DoTest() NN_NOEXCEPT;

        virtual const nnt::fs::util::String& GetRootPath() NN_NOEXCEPT = 0;
    };

    class PathArgumentRootExcess : public PathArgumentRootExcessBase
    {
    public:
        virtual ~PathArgumentRootExcess() NN_NOEXCEPT NN_OVERRIDE
        {
        }

    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            nnt::fs::api::PathArgumentRootExcessBase::SetUp();
            m_RootPath = GetTestRootPath();
            if( m_RootPath[0] == '/' || m_RootPath[0] == '\\' )
            {
                m_RootPath.clear();
            }
            else
            {
                auto pos = m_RootPath.find(':');
                if( nnt::fs::util::String::npos != pos )
                {
                    m_RootPath = m_RootPath.substr(0, pos + 1);
                }
            }
        }

        virtual const nnt::fs::util::String& GetRootPath() NN_NOEXCEPT NN_OVERRIDE
        {
            return m_RootPath;
        }

    private:
        nnt::fs::util::String m_RootPath;
    };

    class PathArgumentRootExcessSubDirectory : public PathArgumentRootExcessBase
    {
    public:
        virtual ~PathArgumentRootExcessSubDirectory() NN_NOEXCEPT NN_OVERRIDE
        {
        }

    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            nnt::fs::api::PathArgumentRootExcessBase::SetUp();
            m_SubDirectoryPath = GetTestRootPath() + "/RootExcessSubDirectory";
            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(m_SubDirectoryPath.c_str()));
            if( GetFsAttribute()->isSaveFileSystem )
            {
                NNT_ASSERT_RESULT_SUCCESS(GetFs().Commit());
            }

            size_t firstIndex = m_SubDirectoryPath.find(':');
            if( nnt::fs::util::String::npos == firstIndex )
            {
                firstIndex = 0;
            }
            else
            {
                ++firstIndex;
            }

            m_RootPath = m_SubDirectoryPath;
            for( auto pos = nnt::fs::util::String::npos; ; --pos )
            {
                m_RootPath += "/..";
                pos = m_SubDirectoryPath.find_last_of("\\/", pos);
                if( pos == firstIndex || pos == nnt::fs::util::String::npos )
                {
                    break;
                }
            }

            nn::fs::DirectoryEntryType directoryEntryType;
            NNT_ASSERT_RESULT_SUCCESS(GetFs().GetEntryType(&directoryEntryType, m_RootPath.c_str()));
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            NNT_ASSERT_RESULT_SUCCESS(GetFs().DeleteDirectoryRecursively(m_SubDirectoryPath.c_str()));
            if( GetFsAttribute()->isSaveFileSystem )
            {
                NNT_ASSERT_RESULT_SUCCESS(GetFs().Commit());
            }
            nnt::fs::api::PathArgumentRootExcessBase::TearDown();
        }

        virtual const nnt::fs::util::String& GetRootPath() NN_NOEXCEPT NN_OVERRIDE
        {
            return m_RootPath;
        }

    private:
        nnt::fs::util::String m_SubDirectoryPath;
        nnt::fs::util::String m_RootPath;
    };

    String PathArgumentPathLength::g_ParentDirectoryPath;
    size_t PathArgumentPathLength::g_ParentDirectoryPathLength;
    String PathArgumentPathLength::g_AppendDirectoryName;
    size_t PathArgumentPathLength::g_AppendDirectoryNameLength;
    bool PathArgumentPathLength::g_IsInitialized = false;

    //! @brief ファイルパスの最大長をテスト
    TEST_F(PathArgumentPathLength, FilePathLength)
    {
        const auto filePathLengthMax = GetFsAttribute()->filePathLengthMax;
        const auto directoryPathLengthMax = GetFsAttribute()->directoryPathLengthMax;

        // パスの最大長ぴったりとなるようにファイル名を設定
        const int parentDirectoryPathLength = static_cast<int>(GetParentDirectoryPathLength());
        const int fileNameLength = filePathLengthMax - parentDirectoryPathLength;
        String fileName1;
        String fileName2;
        for( int i = 0; i < fileNameLength; ++i )
        {
            fileName1.append("b");
            fileName2.append("a");
        }
        const String& parentDirectoryPath = GetParentDirectoryPath();
        const String filePath1 = parentDirectoryPath + fileName1;
        const String filePath2 = parentDirectoryPath + fileName2;

        // パスの最大長をちょうど超えるファイル名を設定
        String filePathExcess = filePath1 + String("a");
        if( GetFsAttribute()->isConcatenationFileSystem )
        {
            filePathExcess = parentDirectoryPath;
            for( int i = 0; i < directoryPathLengthMax + 1 - parentDirectoryPathLength; ++i )
            {
                filePathExcess.append("a");
            }
        }

        // CreateFile が最大長を超えるパスで失敗するか
        NNT_EXPECT_RESULT_FAILURE(
            ResultTooLongPath, GetFs().CreateFile(filePathExcess.c_str(), FileSize));

        // CreateFile が最大長のパスで成功するか
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(filePath1.c_str(), FileSize));
        CheckIfOneEntryExist(parentDirectoryPath.c_str(), fileName1.c_str(), false);

        // OpenFile が最大長のパスで成功するか
        {
            std::unique_ptr<ITestFile> file;
            NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenFile(&file, filePath1.c_str(), OpenMode_Read));
            file.reset();
        }

        // GetEntryType が最大長のパスで成功するか
        {
            DirectoryEntryType directoryEntryType;
            NNT_ASSERT_RESULT_SUCCESS(GetFs().GetEntryType(&directoryEntryType, filePath1.c_str()));
            EXPECT_EQ(DirectoryEntryType_File, directoryEntryType);
        }

        // RenameFile が最大長を超えるパスで失敗するか
        NNT_EXPECT_RESULT_FAILURE(
            ResultTooLongPath, GetFs().RenameFile(filePath1.c_str(), filePathExcess.c_str()));

        // RenameFile が最大長のパスで成功するか
        NNT_ASSERT_RESULT_SUCCESS(GetFs().RenameFile(filePath1.c_str(), filePath2.c_str()));
        CheckIfOneEntryExist(parentDirectoryPath.c_str(), fileName2.c_str(), false);

        // DeleteFile が最大長のパスで成功するか
        NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(filePath2.c_str()));
    }

    //! @brief ディレクトリパスの最大長をテスト
    TEST_F(PathArgumentPathLength, DirectoryPathLength)
    {
        const auto directoryPathLengthMax = GetFsAttribute()->directoryPathLengthMax;

        // パスの最大長ぴったりとなるようにディレクトリ名を設定
        const int parentDirectoryPathLength = static_cast<int>(GetParentDirectoryPathLength());
        const int directoryNameLength = directoryPathLengthMax - parentDirectoryPathLength;
        String directoryName1;
        String directoryName2;
        for( int i = 0; i < directoryNameLength; ++i )
        {
            directoryName1.append("b");
            directoryName2.append("a");
        }
        const String& parentDirectoryPath = GetParentDirectoryPath();
        const String directoryPath1 = parentDirectoryPath + directoryName1;
        const String directoryPath2 = parentDirectoryPath + directoryName2;

        // パスの最大長をちょうど超えるディレクトリ名を設定
        const String directoryPathExcess = directoryPath1 + String("a");

        // CreateDirectory が最大長を超えるパスで失敗するか
        NNT_EXPECT_RESULT_FAILURE(
            ResultTooLongPath, GetFs().CreateDirectory(directoryPathExcess.c_str()));

        // CreateDirectory が最大長のパスで成功するか
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(directoryPath1.c_str()));
        CheckIfOneEntryExist(parentDirectoryPath.c_str(), directoryName1.c_str(), true);

        // OpenDirectory が最大長のパスで成功するか
        {
            std::unique_ptr<ITestDirectory> directory;
            NNT_EXPECT_RESULT_SUCCESS(
                GetFs().OpenDirectory(&directory, directoryPath1.c_str(), OpenDirectoryMode_File));
        }

        // GetEntryType が最大長のパスで成功するか
        {
            DirectoryEntryType directoryEntryType;
            NNT_ASSERT_RESULT_SUCCESS(
                GetFs().GetEntryType(&directoryEntryType, directoryPath1.c_str()));
            EXPECT_EQ(DirectoryEntryType_Directory, directoryEntryType);
        }

        // RenameDirectory のリネーム先を設定
        String renameDirectoryName = directoryName2;
        String renameDirectoryPathExcess = directoryPathExcess;
#if defined(NN_BUILD_CONFIG_OS_WIN32)
        if( GetFsType() == nnt::fs::api::FileSystemType_HostFileSystem )
        {
            // directoryPathLengthMax は ::CreateDirectoryW（ヌル文字含むパス最大長: 248）に合わせて設定されているので
            // RenameDirectory で使用される ::MoveFile（ヌル文字含むパス最大長: 260）に合わせてパスを設定する
            static const int CreateDirectoryWPathLengthMax = 248;
            static const int MoveFilePathLengthMax = 260;
            const int renameDirectoryPathLength
                = directoryPathLengthMax - CreateDirectoryWPathLengthMax + MoveFilePathLengthMax;
            renameDirectoryName = "";
            for( int i = 0; i < renameDirectoryPathLength - parentDirectoryPathLength; ++i )
            {
                renameDirectoryName.append("a");
            }
            renameDirectoryPathExcess = parentDirectoryPath + renameDirectoryName + String("a");
        }
#endif
        const String renameDirectoryPath = parentDirectoryPath + renameDirectoryName;

        // RenameDirectory が最大長を超えるパスで失敗するか
        NNT_ASSERT_RESULT_FAILURE(
            ResultTooLongPath,
            GetFs().RenameDirectory(
                directoryPath1.c_str(), renameDirectoryPathExcess.c_str()));

        // RenameDirectory が最大長のパスで成功するか
        NNT_ASSERT_RESULT_SUCCESS(
            GetFs().RenameDirectory(directoryPath1.c_str(), renameDirectoryPath.c_str()));
        CheckIfOneEntryExist(parentDirectoryPath.c_str(), renameDirectoryName.c_str(), true);

        // DeleteDirectory が最大長のパスで成功するか
        NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteDirectory(renameDirectoryPath.c_str()));
    }

    class PathArgumentNameLength : public PathArgumentFixtureBase
    {
    protected:
        //! @brief 指定した長さで指定した UTF-8 1 文字が連続した名前を生成
        //!        指定した文字だけで指定した長さにできない場合、端数を "a" で埋める
        const String GenerateName(int length, const String& appendChar) NN_NOEXCEPT
        {
            const int firstCharByteLength = CalculateUtf8FirstCharByteLength(appendChar);
            NN_ASSERT_EQUAL(appendChar.length(), static_cast<size_t>(firstCharByteLength));
            const int appendStringLength = GetFsType() == FileSystemType_HostFileSystem ?
                1 : static_cast<int>(appendChar.length());

            String name;
            int nameLength = 0;
            for( ; nameLength + appendStringLength <= length; nameLength += appendStringLength )
            {
                name.append(appendChar);
            }
            for( ; nameLength < length; ++nameLength )
            {
                name.append("a");
            }
            return name;
        }
    private:
        //! @brief 与えられた UTF-8 文字の 1 文字目が何バイトか求める
        int CalculateUtf8FirstCharByteLength(const String& str) NN_NOEXCEPT
        {
            NN_ASSERT_GREATER(str.length(), static_cast<size_t>(0));

            const uint8_t first = static_cast<uint8_t>(str.at(0));
            NN_ASSERT((first < 0x80 || 0xbf < first) && first < 0xfe, "");

            if( first <= 0x7f )
            {
                return 1;
            }
            else if( first <= 0xdf )
            {
                return 2;
            }
            else if( first <= 0xef )
            {
                return 3;
            }
            else if( first <= 0xf7 )
            {
                return 4;
            }
            else if( first <= 0xfb )
            {
                return 5;
            }
            else
            {
                return 6;
            }
        }
    };

    //! @brief ファイル名の最大長をテスト
    TEST_F(PathArgumentNameLength, FileNameLength)
    {
        const String rootPath = GetTestRootPath().append("/");

        // ファイル名の最大長ぴったりになるようにファイル名を設定
        const auto fileNameLengthMax = std::min(
            GetFsAttribute()->fileNameLengthMax,
            GetFsAttribute()->filePathLengthMax - static_cast<int>(rootPath.length()));

        String appendString1;
        String appendString2;
        if( GetFsAttribute()->isSupportedMultiBytePath )
        {
            appendString1 = String(Utf8String1);
            appendString2 = String(Utf8String2);
        }
        else
        {
            appendString1 = String("a");
            appendString2 = String("b");
        }

        const String fileName1 = GenerateName(fileNameLengthMax, appendString1);
        const String fileName2 = GenerateName(fileNameLengthMax, appendString2);

        const String filePath1 = rootPath + fileName1;
        const String filePath2 = rootPath + fileName2;

        // ファイル名の最大長をちょうど超えるようにファイル名を設定
        String filePathExcess = filePath1 + String("a");
        if( GetFsAttribute()->isConcatenationFileSystem )
        {
            const auto directoryNameLengthMax = std::min(
                GetFsAttribute()->directoryNameLengthMax,
                GetFsAttribute()->directoryPathLengthMax
                    - static_cast<int>(GetTestRootPath().length()));
            filePathExcess = rootPath + GenerateName(directoryNameLengthMax, appendString2);
        }

        // CreateFile が最大長を超える名前で失敗するか
        NNT_EXPECT_RESULT_FAILURE(
            ResultTooLongPath, GetFs().CreateFile(filePathExcess.c_str(), FileSize));

        // CreateFile が最大長の名前で成功するか
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(filePath1.c_str(), FileSize));
        CheckIfOneEntryExist(rootPath.c_str(), fileName1.c_str(), false);

        // OpenFile が最大長の名前で成功するか
        {
            std::unique_ptr<ITestFile> file;
            NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenFile(&file, filePath1.c_str(), OpenMode_Read));
            file.reset();
        }

        // GetEntryType が最大長の名前で成功するか
        {
            DirectoryEntryType directoryEntryType;
            NNT_ASSERT_RESULT_SUCCESS(GetFs().GetEntryType(&directoryEntryType, filePath1.c_str()));
            EXPECT_EQ(DirectoryEntryType_File, directoryEntryType);
        }

        // RenameFile が最大長を超える名前で失敗するか
        NNT_EXPECT_RESULT_FAILURE(
            ResultTooLongPath, GetFs().RenameFile(filePath1.c_str(), filePathExcess.c_str()));

        // RenameFile が最大長の名前で成功するか
        NNT_ASSERT_RESULT_SUCCESS(GetFs().RenameFile(filePath1.c_str(), filePath2.c_str()));
        CheckIfOneEntryExist(rootPath.c_str(), fileName2.c_str(), false);

        // DeleteFile が最大長の名前で成功するか
        NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(filePath2.c_str()));
    } // NOLINT(impl/function_size)

    //! @brief ディレクトリ名の最大長をテスト
    TEST_F(PathArgumentNameLength, DirectoryNameLength)
    {
        const auto directoryPathLengthMax = GetFsAttribute()->directoryPathLengthMax;
        const String rootPath = GetTestRootPath().append("/");

        // ディレクトリ名の最大長ぴったりになるようにディレクトリ名を設定
        const auto directoryNameLengthMax = std::min(
            GetFsAttribute()->directoryNameLengthMax,
            directoryPathLengthMax - static_cast<int>(rootPath.length()));

        String appendString1;
        String appendString2;
        if( GetFsAttribute()->isSupportedMultiBytePath )
        {
            appendString1 = String(Utf8String1);
            appendString2 = String(Utf8String2);
        }
        else
        {
            appendString1 = String("a");
            appendString2 = String("b");
        }

        const String directoryName1 = GenerateName(directoryNameLengthMax, appendString1);
        const String directoryName2 = GenerateName(directoryNameLengthMax, appendString2);

        const String directoryPath1 = rootPath + directoryName1;
        const String fdirectoryath2 = rootPath + directoryName2;

        // ディレクトリ名の最大長をちょうど超えるようにディレクトリ名を設定
        const String directoryPathExcess = directoryPath1 + String("a");

        // CreateDirectory が最大長を超える名前で失敗するか
        NNT_EXPECT_RESULT_FAILURE(
            ResultTooLongPath, GetFs().CreateDirectory(directoryPathExcess.c_str()));

        // CreateDirectory が最大長の名前で成功するか
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(directoryPath1.c_str()));
        CheckIfOneEntryExist(rootPath.c_str(), directoryName1.c_str(), true);

        // OpenDirectory が最大長の名前で成功するか
        {
            std::unique_ptr<ITestDirectory> directory;
            NNT_EXPECT_RESULT_SUCCESS(
                GetFs().OpenDirectory(&directory, directoryPath1.c_str(), OpenDirectoryMode_File));
        }

        // GetEntryType が最大長の名前で成功するか
        {
            DirectoryEntryType directoryEntryType;
            NNT_ASSERT_RESULT_SUCCESS(
                GetFs().GetEntryType(&directoryEntryType, directoryPath1.c_str()));
            EXPECT_EQ(DirectoryEntryType_Directory, directoryEntryType);
        }

        // RenameDirectory のリネーム先を設定
        String renameDirectoryName = directoryName2;
        String renameDirectoryPathExcess = directoryPathExcess;
#if defined(NN_BUILD_CONFIG_OS_WIN32)
        if( GetFsType() == nnt::fs::api::FileSystemType_HostFileSystem )
        {
            // directoryPathLengthMax は ::CreateDirectoryW（ヌル文字含むパス最大長: 248）に合わせて設定されているので
            // RenameDirectory で使用される ::MoveFile（ヌル文字含むパス最大長: 260）に合わせてパスを設定する
            static const int CreateDirectoryWPathLengthMax = 248;
            static const int MoveFilePathLengthMax = 260;
            const int renameDirectoryPathLength
                = directoryPathLengthMax - CreateDirectoryWPathLengthMax + MoveFilePathLengthMax;
            const int renameDirectoryNameLength
                = renameDirectoryPathLength - static_cast<int>(rootPath.length());
            if( directoryNameLengthMax != renameDirectoryNameLength )
            {
                renameDirectoryName = GenerateName(renameDirectoryNameLength, appendString2);
            }
            renameDirectoryPathExcess = rootPath + renameDirectoryName + String("a");
        }
#endif
        const String renameDirectoryPath = rootPath + renameDirectoryName;

        // RenameDirectory が最大長を超える名前で失敗するか
        NNT_ASSERT_RESULT_FAILURE(
            ResultTooLongPath,
            GetFs().RenameDirectory(
                directoryPath1.c_str(), renameDirectoryPathExcess.c_str()));

        // RenameDirectory が最大長の名前で成功するか
        NNT_ASSERT_RESULT_SUCCESS(
            GetFs().RenameDirectory(directoryPath1.c_str(), renameDirectoryPath.c_str()));
        CheckIfOneEntryExist(rootPath.c_str(), renameDirectoryName.c_str(), true);

        // DeleteDirectory が最大長の名前で成功するか
        NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteDirectory(renameDirectoryPath.c_str()));
    } // NOLINT(impl/function_size)

    //! @brief パスの最大長とエントリの最大長が異なる場合をテスト
    TEST_F(PathArgumentPathLength, EntryLength)
    {
        const auto rootPath = GetTestRootPath();

        const int64_t filePathLengthMax
            = GetFsAttribute()->filePathLengthMax - (rootPath.size() + 1);
        if( GetFsAttribute()->fileNameLengthMax + 3 < filePathLengthMax )
        {
            {
                auto path = rootPath;

                path.append("/");
                path.append(GetFsAttribute()->fileNameLengthMax, 'a');
                NNT_EXPECT_RESULT_SUCCESS(GetFs().CreateFile(path.c_str(), 0));

                path.append("a");
                NNT_EXPECT_RESULT_FAILURE(
                    nn::fs::ResultTooLongPath,
                    GetFs().CreateFile(path.c_str(), 0));
            }

            {
                auto path = rootPath;

                path.append("/b");
                NNT_EXPECT_RESULT_SUCCESS(GetFs().CreateDirectory(path.c_str()));

                path.append("/");
                path.append(GetFsAttribute()->fileNameLengthMax, 'b');
                NNT_EXPECT_RESULT_SUCCESS(GetFs().CreateFile(path.c_str(), 0));

                path.append("b");
                NNT_EXPECT_RESULT_FAILURE(
                    nn::fs::ResultTooLongPath,
                    GetFs().CreateFile(path.c_str(), 0));
            }
        }

        const int64_t directoryPathLengthMax
            = GetFsAttribute()->directoryPathLengthMax - (rootPath.size() + 1);
        if( GetFsAttribute()->directoryNameLengthMax + 3 < directoryPathLengthMax )
        {
            {
                auto path = rootPath;

                path.append("/");
                path.append(GetFsAttribute()->directoryNameLengthMax, 'c');
                NNT_EXPECT_RESULT_SUCCESS(GetFs().CreateDirectory(path.c_str()));

                path.append("c");
                NNT_EXPECT_RESULT_FAILURE(
                    nn::fs::ResultTooLongPath,
                    GetFs().CreateFile(path.c_str(), 0));
            }

            {
                auto path = rootPath;

                path.append("/d");
                NNT_EXPECT_RESULT_SUCCESS(GetFs().CreateDirectory(path.c_str()));

                path.append("/");
                path.append(GetFsAttribute()->fileNameLengthMax, 'd');
                NNT_EXPECT_RESULT_SUCCESS(GetFs().CreateDirectory(path.c_str()));

                path.append("d");
                NNT_EXPECT_RESULT_FAILURE(
                    nn::fs::ResultTooLongPath,
                    GetFs().CreateDirectory(path.c_str()));
            }
        }
    }

    void PathArgumentRootExcessBase::DoTest() NN_NOEXCEPT
    {
        auto rootPath = GetRootPath();
        NN_LOG("Fs Root Path : '%s'\n", rootPath.c_str());

        auto mode = nn::fs::OpenDirectoryMode_All;

        if( GetFsType() == nnt::fs::api::FileSystemType_HostFileSystem && rootPath.length() > 1 && rootPath[1] == ':' )
        {
            {
                std::unique_ptr<ITestDirectory> pDir;
                NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenDirectory(&pDir, (rootPath + "/../").c_str(), mode));
            }
            {
                std::unique_ptr<ITestDirectory> pDir;
                NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenDirectory(&pDir, (rootPath + "/../../").c_str(), mode));
            }
            {
                std::unique_ptr<ITestDirectory> pDir;
                NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenDirectory(&pDir, (rootPath + "/../../../").c_str(), mode));
            }
        }
        else if( GetFsAttribute()->isRootExcessFailureResultPathNotFound )
        {
            std::unique_ptr<ITestDirectory> pDir;
            NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, GetFs().OpenDirectory(&pDir, (rootPath + "/../").c_str(), mode));
            NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, GetFs().OpenDirectory(&pDir, (rootPath + "/../../").c_str(), mode));
            NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, GetFs().OpenDirectory(&pDir, (rootPath + "/../../../").c_str(), mode));
        }
        else
        {
            std::unique_ptr<ITestDirectory> pDir;
            NNT_EXPECT_RESULT_FAILURE(ResultDirectoryUnobtainable, GetFs().OpenDirectory(&pDir, (rootPath + "/../").c_str(), mode));
            NNT_EXPECT_RESULT_FAILURE(ResultDirectoryUnobtainable, GetFs().OpenDirectory(&pDir, (rootPath + "/../../").c_str(), mode));
            NNT_EXPECT_RESULT_FAILURE(ResultDirectoryUnobtainable, GetFs().OpenDirectory(&pDir, (rootPath + "/../../../").c_str(), mode));
        }
    }

    //! @brief ルートの上位ディレクトリが OpenDirectory で見えないこと
    TEST_F(PathArgumentRootExcess, OpenDirectory)
    {
        DoTest();
    }

    //! @brief ルートの上位ディレクトリが OpenDirectory で見えないこと
    TEST_F(PathArgumentRootExcessSubDirectory, OpenDirectory)
    {
        DoTest();
    }
}}}
