﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/fsApi/testFs_Api.h>

namespace nnt { namespace fs { namespace api {
    void LoadCommitTests() NN_NOEXCEPT
    {
        return;
    }

    typedef GetFileSystemTestFixture CommitTest;

    //! @brief 全ての FS で Commit が成功することを確認
    TEST_F(CommitTest, Commit)
    {
        NNT_FS_UTIL_SKIP_TEST_UNLESS(GetFsAttribute()->isCommitSupported);
        NNT_EXPECT_RESULT_SUCCESS(GetFs().Commit());
    }

    class CommitWithFile : public CleanupFileSystemTestFixture
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            CleanupFileSystemTestFixture::SetUp();
            m_FileName = GetTestRootPath().append("/test.file");
            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(m_FileName.c_str(), 0));
            if( GetFsAttribute()->isSaveFileSystem )
            {
                NNT_ASSERT_RESULT_SUCCESS(GetFs().Commit());
            }
        }
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(m_FileName.c_str()));
            CleanupFileSystemTestFixture::TearDown();
        }

        const nnt::fs::util::String& GetFileName() NN_NOEXCEPT
        {
            return m_FileName;
        }

    private:
        nnt::fs::util::String m_FileName;
    };

    class CommitWithDirectory : public CleanupFileSystemTestFixture
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            CleanupFileSystemTestFixture::SetUp();
            m_DirectoryName = GetTestRootPath().append("/dir");
            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(m_DirectoryName.c_str()));
            if( GetFsAttribute()->isSaveFileSystem )
            {
                NNT_ASSERT_RESULT_SUCCESS(GetFs().Commit());
            }
        }
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteDirectory(m_DirectoryName.c_str()));
            CleanupFileSystemTestFixture::TearDown();
        }

        const nnt::fs::util::String& GetDirectoryName() NN_NOEXCEPT
        {
            return m_DirectoryName;
        }

    private:
        nnt::fs::util::String m_DirectoryName;
    };

    //! @brief OpenMode_Write 以外でファイルオープンしたまま Commit は可
    TEST_F(CommitWithFile, CommitWithOpenFile)
    {
        NNT_FS_UTIL_SKIP_TEST_UNLESS(GetFsAttribute()->isCommitSupported);

        std::unique_ptr<ITestFile> pFile;
        NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&pFile, GetFileName().c_str(), nn::fs::OpenMode_Read));
        NNT_EXPECT_RESULT_SUCCESS(GetFs().Commit());
    }

    //! @brief ディレクトリオープンしたまま Commit は可
    //         todo: テストケース分け
    TEST_F(CommitWithDirectory, CommitWithOpenDirectory)
    {
        NNT_FS_UTIL_SKIP_TEST_UNLESS(GetFsAttribute()->isCommitSupported);

        auto filePath = GetDirectoryName() + "/file";
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(filePath.c_str(), 0, 0));
        NNT_ASSERT_RESULT_SUCCESS(GetFs().Commit());
        NN_UTIL_SCOPE_EXIT
        {
            NNT_ASSERT_RESULT_SUCCESS(GetFs().DeleteFile(filePath.c_str()));
            NNT_ASSERT_RESULT_SUCCESS(GetFs().Commit());
        };

        const nn::fs::OpenDirectoryMode Modes[] =
        {
            nn::fs::OpenDirectoryMode_All,
            nn::fs::OpenDirectoryMode_Directory,
            nn::fs::OpenDirectoryMode_File,
        };

        const char* const OpenPath[] =
        {
            "/",
            "/dir",
        };

        for( auto path : OpenPath )
        {
            for( auto mode : Modes )
            {
                auto directoryPath = GetTestRootPath() + path;

                std::unique_ptr<ITestDirectory> pDir;
                NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenDirectory(&pDir, directoryPath.c_str(), mode));

                NNT_EXPECT_RESULT_SUCCESS(GetFs().Commit());

                pDir.reset();

                NNT_ASSERT_RESULT_SUCCESS(GetFs().Commit());

                RemountFs();
            }
        }
    }
}}}
