﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_ScopeExit.h>
#include <nnt/fsApi/testFs_Api.h>

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

/**
* @file
* @brief    ファイルシステム関数、アライメント観点を定義します。
*/
namespace {
    // 書き込み・読み込みを比較するサイズ
    const size_t CompareSize = 1024;

    // リライト用バッファ
    uint8_t g_WriteBuffer[2 * CompareSize];
    uint8_t g_ReadBuffer[2 * CompareSize];

    typedef nn::Result(*ReadFunction)(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size);

    nn::Result ReadFunction0(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(pTestFile->Read(offset, buffer, size, nn::fs::ReadOption()));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction1(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(pTestFile->Read(offset, buffer, size));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction2(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        size_t sizeRead;
        NN_RESULT_DO(pTestFile->Read(&sizeRead, offset, buffer, size, nn::fs::ReadOption()));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction3(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        size_t sizeRead;
        NN_RESULT_DO(pTestFile->Read(&sizeRead, offset, buffer, size));
        NN_RESULT_SUCCESS;
    }

    bool IsReadOverload(ReadFunction read) NN_NOEXCEPT
    {
        return read != ReadFunction2;
    }

    const ReadFunction ReadFunctions[] =
    {
        ReadFunction0,
        ReadFunction1,
        ReadFunction2,
        ReadFunction3
    };
}

namespace nnt { namespace fs { namespace api {
    void LoadBufferAddressAlignmentTests() NN_NOEXCEPT
    {
        return;
    }

    /**
     * @brief   BufferAddressAlignment で利用するテストフィクスチャです。
     */
    class BufferAddressAlignment : public CleanupFileSystemTestFixture, public ::testing::WithParamInterface<ReadFunction>
    {
    protected:

        /**
         * @brief テスト開始時に毎回呼び出される関数です。
         */
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            CleanupFileSystemTestFixture::SetUp();
        }

        /**
         * @brief テスト終了時に毎回呼び出される関数です。
         */
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            CleanupFileSystemTestFixture::TearDown();
        }

        /**
         * @brief バッファの初期値領域を確認する関数です。
         *   @param[in]  pBuffer                初期値を確認するバッファの先頭ポインタを指定します
         *   @param[in]  bufferSize             指定しているバッファのサイズ（バイト単位）を指定します
         *   @param[in]  offsetSize             バッファをオフセットするサイズ（バイト単位）を指定します
         *
         *   @return     指定バッファの比較結果が返ります。
         *   @retval     true                   指定バッファには初期値のみ含まれています。
         *   @retval     false                  指定バッファには初期値以外の内容が含まれています。
         */
        bool IsValidInitializedBound(
            const uint8_t* pBuffer, const size_t bufferSize, const size_t offsetSize) NN_NOEXCEPT
        {
            return nnt::fs::util::IsValidInitializedBound(pBuffer, bufferSize, offsetSize, CompareSize);
        }
    };

    //!< @brief 2048バイトのファイルを作成し0～64バイト範囲でオフセットしつつ、1024バイト単位の書き込み／読み込みが正しくできること
    TEST_P(BufferAddressAlignment, ReadFile_WriteFile)
    {
        auto read = GetParam();
        NNT_FS_UTIL_SKIP_TEST_UNLESS(!IsReadOverload(read) || GetFsAttribute()->isReadOverloadsSupported);

        std::unique_ptr<ITestFile> file;
        String fileName = GetTestRootPath().append("/test.file");

        // 0～64までのオフセット用MAX値
        const size_t OffsetSize = 64;

        FillBufferWithRandomValue(g_WriteBuffer, sizeof(g_WriteBuffer));
        for (size_t changePos = 0; changePos <= OffsetSize; ++changePos)
        {
            NNT_FS_SCOPED_TRACE("offset count: %d, file path: %s\n", changePos, fileName.c_str());

            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(fileName.c_str(), sizeof(g_WriteBuffer)));
            NN_UTIL_SCOPE_EXIT
            {
                NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(fileName.c_str()));
            };

            NNT_ASSERT_RESULT_SUCCESS(
                GetFs().OpenFile(
                    &file,
                    fileName.c_str(),
                    static_cast<OpenMode>(OpenMode_Read | OpenMode_Write)));
            NN_UTIL_SCOPE_EXIT
            {
                file.reset(nullptr);
            };

            NNT_EXPECT_RESULT_SUCCESS(
                file->Write(0, &g_WriteBuffer[changePos], CompareSize, WriteOption()));
            NNT_EXPECT_RESULT_SUCCESS(file->Flush());

            InvalidateVariable(g_ReadBuffer, sizeof(g_ReadBuffer));
            NNT_EXPECT_RESULT_SUCCESS(
                read(file.get(), 0, &g_ReadBuffer[changePos], CompareSize));

            NNT_FS_UTIL_EXPECT_MEMCMPEQ(
                &g_WriteBuffer[changePos], &g_ReadBuffer[changePos], CompareSize);

            EXPECT_TRUE(IsValidInitializedBound(g_ReadBuffer, sizeof(g_ReadBuffer), changePos));
        }
    }

    INSTANTIATE_TEST_CASE_P(WithReadOverloads,
        BufferAddressAlignment,
        ::testing::ValuesIn(ReadFunctions));

}}}
