﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testFriends_Common.h"

namespace
{
    nn::account::Uid s_Users[nn::account::UserCountMax] = {};
    nn::account::NetworkServiceAccountId s_AccountIds[nn::account::UserCountMax] = {};
    int s_UserCount = 0;
}

class FriendsUserSetting : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::account::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());

        nnt::friends::LoadAccounts(&s_UserCount, s_Users, s_AccountIds, NN_ARRAY_SIZE(s_Users));
        NN_ABORT_UNLESS_GREATER_EQUAL(s_UserCount, 1);

        nn::friends::SetOption(nn::friends::OptionAdmin_CheckUserStatus, 0);
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
    }
};

TEST_F(FriendsUserSetting, Sync)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nn::friends::AsyncContext context;
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::SyncUserSetting(&context, s_Users[0]));

    nn::os::SystemEvent completionEvent;
    NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

    completionEvent.Wait();

    NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

    nn::friends::UserSetting setting;

    NNT_ASSERT_RESULT_SUCCESS(setting.Initialize(s_Users[0]));

    nnt::friends::Dump(setting);
}

TEST_F(FriendsUserSetting, FriendCode)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nn::friends::UserSetting setting;

    NNT_ASSERT_RESULT_SUCCESS(setting.Initialize(s_Users[0]));

    nn::friends::AsyncContext context;
    NNT_ASSERT_RESULT_SUCCESS(setting.IssueFriendCode(&context));

    nn::os::SystemEvent completionEvent;
    NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

    completionEvent.Wait();

    if (context.GetResult().IsFailure())
    {
        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultDisallowFriendCodeIssue, context.GetResult());
    }

    nnt::friends::Dump(setting);
}

TEST_F(FriendsUserSetting, ExtraPermission)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nn::friends::UserSetting setting;

    NNT_ASSERT_RESULT_SUCCESS(setting.Initialize(s_Users[0]));

    // 未設定
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(setting.ChangePlayLogPermission(&context, nn::friends::ExtraPermission_Unconfigured));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultInvalidArgument, context.GetResult());
    }

    nn::friends::ExtraPermission permissions[] =
    {
        nn::friends::ExtraPermission_Self,
        nn::friends::ExtraPermission_FavoriteFriends,
        nn::friends::ExtraPermission_Friends,
        nn::friends::ExtraPermission_FriendsOfFriend,
        nn::friends::ExtraPermission_Everyone,
    };

    for (int i = 0; i < NN_ARRAY_SIZE(permissions); i++)
    {
        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(setting.ChangePlayLogPermission(&context, permissions[i]));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            EXPECT_EQ(setting.GetPlayLogPermission(), permissions[i]);
        }
        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(setting.ClearPlayLog(&context));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            EXPECT_EQ(setting.GetPlayLogPermission(), permissions[i]);
        }
    }
}
