﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testFriends_Common.h"

namespace
{
    nn::account::Uid s_Users[nn::account::UserCountMax] = {};
    nn::account::NetworkServiceAccountId s_AccountIds[nn::account::UserCountMax] = {};
    int s_UserCount = 0;
}

class FriendsRelationship : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::account::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());

        nnt::friends::LoadAccounts(&s_UserCount, s_Users, s_AccountIds, NN_ARRAY_SIZE(s_Users));
        NN_ABORT_UNLESS_GREATER_EQUAL(s_UserCount, 3);

        nn::friends::SetOption(nn::friends::OptionAdmin_CheckUserStatus, 0);
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
    }
};

TEST_F(FriendsRelationship, CleanupRelationship)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nnt::friends::CleanupRelationship(s_Users[0]);
}

TEST_F(FriendsRelationship, NotFound)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nn::account::NetworkServiceAccountId accontId = {};
    nn::friends::Relationship relationship = {};

    nn::friends::AsyncContext context;
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetRelationship(&context, &relationship, s_Users[0], accontId));

    nn::os::SystemEvent completionEvent;
    NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

    completionEvent.Wait();

    NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultNetworkServiceAccountNotExistsGeneral, context.GetResult());
}

TEST_F(FriendsRelationship, Stranger)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nn::friends::Relationship relationship = {};

    nn::friends::AsyncContext context;
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetRelationship(&context, &relationship, s_Users[0], s_AccountIds[1]));

    nn::os::SystemEvent completionEvent;
    NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

    completionEvent.Wait();

    NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

    EXPECT_FALSE(relationship.isFriend);
    EXPECT_FALSE(relationship.isFriendOfFriend);
    EXPECT_FALSE(relationship.isBlocked);
    EXPECT_FALSE(relationship.isRequestPending);
}

TEST_F(FriendsRelationship, RequestPending)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[0], s_AccountIds[1],
            nn::friends::RequestType_FriendCode));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        nn::friends::Relationship relationship = {};

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetRelationship(&context, &relationship, s_Users[0], s_AccountIds[1]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        EXPECT_FALSE(relationship.isFriend);
        EXPECT_FALSE(relationship.isFriendOfFriend);
        EXPECT_FALSE(relationship.isBlocked);
        EXPECT_TRUE(relationship.isRequestPending);
    }
}

TEST_F(FriendsRelationship, Friend)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[1], s_AccountIds[0],
            nn::friends::RequestType_FriendCode));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        nn::friends::Relationship relationship = {};

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetRelationship(&context, &relationship, s_Users[0], s_AccountIds[1]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        EXPECT_TRUE(relationship.isFriend);
        EXPECT_FALSE(relationship.isFriendOfFriend);
        EXPECT_FALSE(relationship.isBlocked);
        EXPECT_FALSE(relationship.isRequestPending);
    }

    // 同期してキャッシュが作成されることを保証する。
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SyncFriendList(&context, s_Users[0]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }

    nn::friends::Friend f;
    int count;
    nn::friends::FriendFilter filter = {};

    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendList(&count, &f, s_Users[0], 0, 1, filter));
    EXPECT_EQ(count, 1);
    EXPECT_EQ(f.GetAccountId(), s_AccountIds[1]);
}

TEST_F(FriendsRelationship, FriendOfFriend)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    // 1 -> 2
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[1], s_AccountIds[2],
            nn::friends::RequestType_FriendCode));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    // 2 -> 1
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[2], s_AccountIds[1],
            nn::friends::RequestType_FriendCode));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    // 2 は 0 の友達の友達
    {
        nn::friends::Relationship relationship = {};

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetRelationship(&context, &relationship, s_Users[0], s_AccountIds[2]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        EXPECT_FALSE(relationship.isFriend);
        EXPECT_TRUE(relationship.isFriendOfFriend);
        EXPECT_FALSE(relationship.isBlocked);
        EXPECT_FALSE(relationship.isRequestPending);
    }
}

TEST_F(FriendsRelationship, BlockFriend)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::BlockUser(&context, s_Users[0], s_AccountIds[1],
            nn::friends::BlockReason_BadFriend));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        nn::friends::Relationship relationship = {};

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetRelationship(&context, &relationship, s_Users[0], s_AccountIds[1]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        EXPECT_FALSE(relationship.isFriend);
        EXPECT_FALSE(relationship.isFriendOfFriend);
        EXPECT_TRUE(relationship.isBlocked);
        EXPECT_FALSE(relationship.isRequestPending);
    }

    nn::friends::Friend f;
    int count;
    nn::friends::FriendFilter filter = {};

    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendList(&count, &f, s_Users[0], 0, 1, filter));
    EXPECT_EQ(count, 0);
}

TEST_F(FriendsRelationship, SendFriendRequestToBlocker)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nn::friends::FriendRequest sentRequest;

    // ブロックされたユーザーからブロックしたユーザーにフレンド申請するのは問題ない。
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[1], s_AccountIds[0],
            nn::friends::RequestType_FriendCode));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        int count;

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count, &sentRequest, s_Users[1], 0, 1,
            nn::friends::RequestListType_Sent));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        EXPECT_EQ(count, 1);
    }
    {
        nn::friends::FriendRequest receivedRequest;
        int count;

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count, &receivedRequest, s_Users[0], 0, 1,
            nn::friends::RequestListType_Received));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        // 相手をブロックしているので、受信したフレンド申請件数は 0 のはず。
        EXPECT_EQ(count, 0);
    }
    // ブロックしたユーザーからのフレンド申請は ID を知っていても承認できない。
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::AcceptFriendRequest(&context, s_Users[0], sentRequest.GetRequestId()));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultRequestNotFound, context.GetResult());
    }
}

TEST_F(FriendsRelationship, SendFriendRequestToBlockee)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    // ブロックしたユーザーがブロックされたユーザーにフレンド申請はできない。
    nn::friends::AsyncContext context;
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[0], s_AccountIds[1],
        nn::friends::RequestType_FriendCode));

    nn::os::SystemEvent completionEvent;
    NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

    completionEvent.Wait();

    NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultBlockedByMe, context.GetResult());
}

TEST_F(FriendsRelationship, UnblockAndReblock)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    // ブロック解除
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::UnblockUser(&context, s_Users[0], s_AccountIds[1]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        nn::friends::FriendRequest request;
        int count;

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count, &request, s_Users[0], 0, 1,
            nn::friends::RequestListType_Received));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        // ブロックした相手からのフレンド申請が復活する。
        EXPECT_EQ(count, 1);
    }

    // 再ブロック
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::BlockUser(&context, s_Users[0], s_AccountIds[1],
            nn::friends::BlockReason_BadFriend));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        nn::friends::FriendRequest request;
        int count;

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count, &request, s_Users[0], 0, 1,
            nn::friends::RequestListType_Received));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        // また見えなくなる。
        EXPECT_EQ(count, 0);
    }
}
