﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testFriends_Common.h"

namespace
{
    nn::account::Uid s_Users[nn::account::UserCountMax] = {};
    nn::account::NetworkServiceAccountId s_AccountIds[nn::account::UserCountMax] = {};
    int s_UserCount = 0;
}

class FriendsPlayHistory : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::account::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());

        nnt::friends::LoadAccounts(&s_UserCount, s_Users, s_AccountIds, NN_ARRAY_SIZE(s_Users));
        NN_ABORT_UNLESS_GREATER_EQUAL(s_UserCount, 3);

        nn::friends::SetOption(nn::friends::OptionAdmin_CheckUserStatus, 0);
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
    }
};


TEST_F(FriendsPlayHistory, CleanupRelationship)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    for (int i = 0; i < 3; i++)
    {
        nnt::friends::CleanupRelationship(s_Users[i]);
    }
}

TEST_F(FriendsPlayHistory, CreateRelationship)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[1], s_AccountIds[0],
            nn::friends::RequestType_FriendCode));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[0], s_AccountIds[1],
            nn::friends::RequestType_FriendCode));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
}

TEST_F(FriendsPlayHistory, Cleanup)
{
    for (int i = 0; i < 3; i++)
    {
        // 2 回ずつ削除してみる。
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::DeletePlayHistory(s_Users[i]));
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::DeletePlayHistory(s_Users[i]));
    }
}

TEST_F(FriendsPlayHistory, OwnError)
{
    nn::friends::PlayHistoryRegistrationKey key = {};
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryRegistrationKey(&key, s_Users[0], false));

    nn::friends::InAppScreenName name = {"User0", {"ja"}};
    nn::friends::InAppScreenName myName = {"User0", {"ja"}};

    NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultOwnPlayHistoryRegistrationKey,
        nn::friends::AddPlayHistory(s_Users[0], key, name, myName));
}

TEST_F(FriendsPlayHistory, BrokenError)
{
    nn::friends::PlayHistoryRegistrationKey key = {};
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryRegistrationKey(&key, s_Users[0], false));

    key.value[63]++;

    nn::friends::InAppScreenName name = {"User0", {"ja"}};
    nn::friends::InAppScreenName myName = {"User0", {"ja"}};

    NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultPlayHistoryRegistrationKeyBroken,
        nn::friends::AddPlayHistory(s_Users[0], key, name, myName));
}

TEST_F(FriendsPlayHistory, InvalidName)
{
    // 冗長な文字コード
    {
        nn::friends::PlayHistoryRegistrationKey key = {};
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryRegistrationKey(&key, s_Users[1], false));

        nn::friends::InAppScreenName name = {{'\xC0', '\xAF', '\0'}, {"ja"}};
        nn::friends::InAppScreenName myName = {{'\xC0', '\xAF', '\0'}, {"ja"}};

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultInvalidArgument,
            nn::friends::AddPlayHistory(s_Users[0], key, name, myName));
    }
    // 途中までしかない名前
    {
        nn::friends::PlayHistoryRegistrationKey key = {};
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryRegistrationKey(&key, s_Users[1], false));

        nn::friends::InAppScreenName name = {{'\xE3', '\x81', '\x82', '\xE3', '\x81', '\x82', '\xE3', '\0'}, {"ja"}};
        nn::friends::InAppScreenName myName = {{'\xE3', '\x81', '\x82', '\xE3', '\x81', '\0'}, {"ja"}};

        // 登録は成功するが、名前は正常なところまでしかコピーされない。
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::AddPlayHistory(s_Users[0], key, name, myName));

        nn::friends::PlayHistory histories[2];
        int count;

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryList(&count, histories, s_Users[0], 0, NN_ARRAY_SIZE(histories)));

        ASSERT_EQ(count, 1);

        EXPECT_EQ(nn::util::Strnlen(histories[0].GetPlayRecord().name.name, sizeof (name.name)), 6);
        EXPECT_EQ(nn::util::Strnlen(histories[0].GetPlayRecord().myName.name, sizeof (myName.name)), 3);

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::DeletePlayHistory(s_Users[0]));
    }
    // 長すぎる
    {
        nn::friends::PlayHistoryRegistrationKey key = {};
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryRegistrationKey(&key, s_Users[1], false));

        nn::friends::InAppScreenName name = {"123456789012345678901234567890123456789012345678901234567890", {"ja"}};
        nn::friends::InAppScreenName myName = {"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz", {"ja"}};

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::AddPlayHistory(s_Users[0], key, name, myName));

        nn::friends::PlayHistory histories[2];
        int count;

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryList(&count, histories, s_Users[0], 0, NN_ARRAY_SIZE(histories)));

        ASSERT_EQ(count, 1);

        EXPECT_TRUE(histories[0].IsValid());
        EXPECT_STREQ(histories[0].GetPlayRecord().name.name, "12345678901234567890");
        EXPECT_STREQ(histories[0].GetPlayRecord().myName.name, "ABCDEFGHIJKLMNOPQRST");

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::DeletePlayHistory(s_Users[0]));
    }
}

TEST_F(FriendsPlayHistory, InvalidLanguage)
{
    nn::friends::PlayHistoryRegistrationKey key = {};
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryRegistrationKey(&key, s_Users[1], false));

    // 不正な言語コード
    nn::friends::InAppScreenName name = {"", {{'\xC0', '\xAF', '\0'}}};
    nn::friends::InAppScreenName myName = {"", {{'\xC0', '\xAF', '\0'}}};

    NNT_EXPECT_RESULT_FAILURE(nn::friends::ResultInvalidArgument,
        nn::friends::AddPlayHistory(s_Users[0], key, name, myName));
}

TEST_F(FriendsPlayHistory, Add)
{
    for (int i = 0; i < s_UserCount; i++)
    {
        for (int j = 0; j < s_UserCount; j++)
        {
            if (i == j)
            {
                continue;
            }

            {
                nn::friends::PlayHistoryRegistrationKey key = {};
                NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryRegistrationKey(&key, s_Users[j], true));

                nn::friends::InAppScreenName name = {"UserB_Local", {"ja"}};
                nn::friends::InAppScreenName myName = {"UserA", {"ja"}};

                NNT_ASSERT_RESULT_SUCCESS(nn::friends::AddPlayHistory(s_Users[i], key, name, myName));
            }
            {
                nn::friends::PlayHistoryRegistrationKey key = {};
                NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryRegistrationKey(&key, s_Users[j], false));

                nn::friends::InAppScreenName name = {"UserB_Online", {"ja"}};
                nn::friends::InAppScreenName myName = {"UserA", {"ja"}};

                NNT_ASSERT_RESULT_SUCCESS(nn::friends::AddPlayHistory(s_Users[i], key, name, myName));
            }

            if (i == 0 && j == 1)
            {
                nn::friends::FriendDetailedInfo info;

                NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendDetailedInfo(&info, s_Users[0], s_AccountIds[1]));

                EXPECT_STREQ(info.GetLastPlayRecord().name.name, "UserB_Online");
                EXPECT_STREQ(info.GetLastPlayRecord().name.language.string, "ja");
                EXPECT_STREQ(info.GetLastPlayRecord().myName.name, "UserA");
                EXPECT_STREQ(info.GetLastPlayRecord().myName.language.string, "ja");
            }
        }

        // ネットワークサービスアカウントのキャッシュを任意のタイミングで削除しても問題ないかどうかの確認。
        nn::friends::DeleteNetworkServiceAccountCache(s_Users[i]);
    }

    nn::friends::PlayHistory histories[3];
    int count;

    {
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryList(&count, histories, s_Users[0], 0, NN_ARRAY_SIZE(histories)));

        ASSERT_EQ(count, 2);

        // 登録の降順で並んでいるはず。
        EXPECT_TRUE(histories[0].IsValid());
        EXPECT_EQ(histories[0].GetAccountId(), s_AccountIds[2]);
        // 同じネットワークサービスアカウントが指定された場合、設定は上書きされる。
        EXPECT_FALSE(histories[1].IsLocalPlayed());

        EXPECT_TRUE(histories[1].IsValid());
        EXPECT_EQ(histories[1].GetAccountId(), s_AccountIds[1]);
        EXPECT_FALSE(histories[1].IsLocalPlayed());

        nnt::friends::Dump(histories, count);
    }
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryList(&count, histories, s_Users[1], 0, NN_ARRAY_SIZE(histories)));

        ASSERT_EQ(count, 2);

        EXPECT_TRUE(histories[0].IsValid());
        EXPECT_EQ(histories[0].GetAccountId(), s_AccountIds[2]);
        EXPECT_FALSE(histories[1].IsLocalPlayed());

        EXPECT_TRUE(histories[1].IsValid());
        EXPECT_EQ(histories[1].GetAccountId(), s_AccountIds[0]);
        EXPECT_FALSE(histories[1].IsLocalPlayed());

        nnt::friends::Dump(histories, count);
    }
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryList(&count, histories, s_Users[2], 0, NN_ARRAY_SIZE(histories)));

        ASSERT_EQ(count, 2);

        EXPECT_TRUE(histories[0].IsValid());
        EXPECT_EQ(histories[0].GetAccountId(), s_AccountIds[1]);
        EXPECT_FALSE(histories[1].IsLocalPlayed());

        EXPECT_TRUE(histories[1].IsValid());
        EXPECT_EQ(histories[1].GetAccountId(), s_AccountIds[0]);
        EXPECT_FALSE(histories[1].IsLocalPlayed());

        nnt::friends::Dump(histories, count);
    }
}

TEST_F(FriendsPlayHistory, Statistics)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::DeletePlayHistory(s_Users[0]));

    nn::friends::PlayHistoryStatistics statistics;
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryStatistics(&statistics, s_Users[0]));

    EXPECT_EQ(statistics.totalLocalPlayCount, 0);
    EXPECT_EQ(statistics.totalOnlinePlayCount, 0);

    // ローカル 10
    for (int i = 0; i < 10; i++)
    {
        nn::account::NetworkServiceAccountId accountId = {static_cast<uint64_t>(i)};

        nn::friends::PlayHistoryRegistrationKey key = {};
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryRegistrationKey(&key, accountId, true));

        nn::friends::InAppScreenName name = {"User", {"ja"}};
        nn::friends::InAppScreenName myName = {"User", {"ja"}};

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::AddPlayHistory(s_Users[0], key, name, myName));
    }
    // オンライン 20
    for (int i = 0; i < 20; i++)
    {
        nn::account::NetworkServiceAccountId accountId = {static_cast<uint64_t>(1000 + i)};

        nn::friends::PlayHistoryRegistrationKey key = {};
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryRegistrationKey(&key, accountId, false));

        nn::friends::InAppScreenName name = {"User", {"ja"}};
        nn::friends::InAppScreenName myName = {"User", {"ja"}};

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::AddPlayHistory(s_Users[0], key, name, myName));
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryStatistics(&statistics, s_Users[0]));

    EXPECT_EQ(statistics.totalLocalPlayCount, 10);
    EXPECT_EQ(statistics.totalOnlinePlayCount, 20);
}

TEST_F(FriendsPlayHistory, Overflow)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::DeletePlayHistory(s_Users[0]));

    const int MaxValue = nn::friends::PlayHistoryCountMax * 5;

    for (int i = 0; i < MaxValue; i++)
    {
        nn::account::NetworkServiceAccountId accountId = {static_cast<uint64_t>(i)};

        nn::friends::PlayHistoryRegistrationKey key = {};
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryRegistrationKey(&key, accountId, true));

        nn::friends::InAppScreenName name = {"User", {"ja"}};
        nn::friends::InAppScreenName myName = {"User", {"ja"}};

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::AddPlayHistory(s_Users[0], key, name, myName));
    }

    static nn::friends::PlayHistory s_Histories[nn::friends::PlayHistoryCountMax];
    int count;

    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryList(&count, s_Histories, s_Users[0], 0, NN_ARRAY_SIZE(s_Histories)));

    ASSERT_EQ(count, NN_ARRAY_SIZE(s_Histories));

    for (int i = 0; i < count; i++)
    {
        EXPECT_EQ(s_Histories[i].GetAccountId().id, MaxValue - i - 1);
    }

    nn::friends::PlayHistoryStatistics statistics;
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryStatistics(&statistics, s_Users[0]));

    EXPECT_EQ(statistics.totalLocalPlayCount, MaxValue);
    EXPECT_EQ(statistics.totalOnlinePlayCount, 0);
}

TEST_F(FriendsPlayHistory, PerformanceCheck)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::DeletePlayHistory(s_Users[0]));
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::DeletePlayHistory(s_Users[1]));

    // BG 動作によってパフォーマンス計測に影響が出ないようにする。
    nn::friends::DaemonSuspension suspension;
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::SuspendDaemon(&suspension));

    nn::account::UserHandle handle = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&handle, s_Users[0]));

    NN_UTIL_SCOPE_EXIT
    {
        // このタイミングで、BG 書き込みが行われる。
        nn::account::CloseUser(handle);
    };

    nn::friends::PlayHistoryRegistrationKey key1 = {};
    nn::friends::PlayHistoryRegistrationKey key2 = {};

    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryRegistrationKey(&key1, s_Users[1], false));
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetPlayHistoryRegistrationKey(&key2, s_Users[2], false));

    nn::friends::InAppScreenName name = {"User", {"ja"}};
    nn::friends::InAppScreenName myName = {"User", {"ja"}};

    nn::os::Tick s = nn::os::GetSystemTick();

    for (int i = 0; i < 50; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::AddPlayHistory(s_Users[0], key1, name, myName));
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::AddPlayHistory(s_Users[0], key2, name, myName));
    }

    nn::TimeSpan elapsed = (nn::os::GetSystemTick() - s).ToTimeSpan();

    NN_LOG("Elapsed = %lld ms\n", elapsed.GetMilliSeconds());
}
