﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testFriends_Common.h"

#include <nn/friends/detail/service/core/friends_BackgroundTaskManager.h>

using namespace nn::friends::detail::service::core;

namespace
{
    nn::account::Uid g_Users[nn::account::UserCountMax] = {};
    int g_UserCount = 0;
}

TEST(FriendsBackgroundTaskManager, Initialize)
{
    nn::account::Initialize();

    nnt::friends::LoadAccounts(&g_UserCount, g_Users, NN_ARRAY_SIZE(g_Users));
    NN_ABORT_UNLESS_GREATER_EQUAL(g_UserCount, 2);

    for (int i = 0; i < g_UserCount; i++)
    {
        BackgroundTaskManager::GetInstance().NotifyUserAdded(g_Users[i]);
    }
}

TEST(FriendsBackgroundTaskManager, None)
{
    BackgroundTaskManager::TaskInfo info;

    EXPECT_FALSE(BackgroundTaskManager::GetInstance().GetRunnableTask(&info));

    // Open 中のユーザーが 0 人だと何も起こらない。
    BackgroundTaskManager::GetInstance().RequestUpdateListSummary();

    EXPECT_FALSE(BackgroundTaskManager::GetInstance().GetRunnableTask(&info));
}

TEST(FriendsBackgroundTaskManager, SleepAwaked)
{
    // スリープ復帰が行われると、ユーザー／フレンドリスト／ブロックリストの同期処理が発生する。
    BackgroundTaskManager::GetInstance().NotifySleepAwaked();

    BackgroundTaskManager::TaskInfo info;

    // タスク優先度順に処理が行われているかどうかの確認。
    for (int i = 0; i < g_UserCount; i++)
    {
        EXPECT_TRUE(BackgroundTaskManager::GetInstance().GetRunnableTask(&info));
        EXPECT_EQ(info.taskId, BackgroundTaskManager::TaskId_SyncUser);
        EXPECT_EQ(info.uid, g_Users[i]);
        BackgroundTaskManager::GetInstance().NotifyDone(info, nn::ResultSuccess());
    }

    for (int i = 0; i < g_UserCount; i++)
    {
        EXPECT_TRUE(BackgroundTaskManager::GetInstance().GetRunnableTask(&info));
        EXPECT_EQ(info.taskId, BackgroundTaskManager::TaskId_SyncFriendList);
        EXPECT_EQ(info.uid, g_Users[i]);
        BackgroundTaskManager::GetInstance().NotifyDone(info, nn::ResultSuccess());
    }

    for (int i = 0; i < g_UserCount; i++)
    {
        EXPECT_TRUE(BackgroundTaskManager::GetInstance().GetRunnableTask(&info));
        EXPECT_EQ(info.taskId, BackgroundTaskManager::TaskId_SyncBlockedUserList);
        EXPECT_EQ(info.uid, g_Users[i]);
        BackgroundTaskManager::GetInstance().NotifyDone(info, nn::ResultSuccess());
    }

    EXPECT_FALSE(BackgroundTaskManager::GetInstance().GetRunnableTask(&info));
}

TEST(FriendsBackgroundTaskManager, NpnsConnected)
{
    // 接続状態にすることで、ユーザーアカウントの Open 時に GetListSummary タスクが実行される。
    BackgroundTaskManager::GetInstance().NotifyNpnsStateChanged(true);

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(BackgroundTaskManager::RandomWaitMax + 1));
    BackgroundTaskManager::GetInstance().Schedule();

    BackgroundTaskManager::TaskInfo info;

    for (int i = 0; i < g_UserCount; i++)
    {
        EXPECT_TRUE(BackgroundTaskManager::GetInstance().GetRunnableTask(&info));
        EXPECT_EQ(info.taskId, BackgroundTaskManager::TaskId_UpdateUserPresence);
        EXPECT_EQ(info.uid, g_Users[i]);
        BackgroundTaskManager::GetInstance().NotifyDone(info, nn::ResultSuccess());
    }

    for (int i = 0; i < g_UserCount; i++)
    {
        EXPECT_TRUE(BackgroundTaskManager::GetInstance().GetRunnableTask(&info));
        EXPECT_EQ(info.taskId, BackgroundTaskManager::TaskId_GetFriendRequestCount);
        EXPECT_EQ(info.uid, g_Users[i]);
        BackgroundTaskManager::GetInstance().NotifyDone(info, nn::ResultSuccess());
    }

    for (int i = 0; i < g_UserCount; i++)
    {
        EXPECT_TRUE(BackgroundTaskManager::GetInstance().GetRunnableTask(&info));
        EXPECT_EQ(info.taskId, BackgroundTaskManager::TaskId_GetFriendPresence);
        EXPECT_EQ(info.uid, g_Users[i]);
        BackgroundTaskManager::GetInstance().NotifyDone(info, nn::ResultSuccess());
    }

    EXPECT_FALSE(BackgroundTaskManager::GetInstance().GetRunnableTask(&info));
}

TEST(FriendsBackgroundTaskManager, UserOpened)
{
    BackgroundTaskManager::GetInstance().NotifyUserOpened(g_Users[0]);

    BackgroundTaskManager::TaskInfo info;

    EXPECT_TRUE(BackgroundTaskManager::GetInstance().GetRunnableTask(&info));
    EXPECT_EQ(info.taskId, BackgroundTaskManager::TaskId_GetListSummary);
    EXPECT_EQ(info.uid, g_Users[0]);
    BackgroundTaskManager::GetInstance().NotifyDone(info, nn::ResultSuccess());

    EXPECT_TRUE(BackgroundTaskManager::GetInstance().GetRunnableTask(&info));
    EXPECT_EQ(info.taskId, BackgroundTaskManager::TaskId_SyncUser);
    EXPECT_EQ(info.uid, g_Users[0]);
    BackgroundTaskManager::GetInstance().NotifyDone(info, nn::ResultSuccess());

    EXPECT_TRUE(BackgroundTaskManager::GetInstance().GetRunnableTask(&info));
    EXPECT_EQ(info.taskId, BackgroundTaskManager::TaskId_SyncBlockedUserList);
    EXPECT_EQ(info.uid, g_Users[0]);
    BackgroundTaskManager::GetInstance().NotifyDone(info, nn::ResultSuccess());

    EXPECT_FALSE(BackgroundTaskManager::GetInstance().GetRunnableTask(&info));

    BackgroundTaskManager::GetInstance().NotifyUserClosed(g_Users[0]);

    EXPECT_TRUE(BackgroundTaskManager::GetInstance().GetRunnableTask(&info));
    EXPECT_EQ(info.taskId, BackgroundTaskManager::TaskId_SyncUser);
    EXPECT_EQ(info.uid, g_Users[0]);
    BackgroundTaskManager::GetInstance().NotifyDone(info, nn::ResultSuccess());

    EXPECT_FALSE(BackgroundTaskManager::GetInstance().GetRunnableTask(&info));
}

TEST(FriendsBackgroundTaskManager, NpnsDisconnected)
{
    BackgroundTaskManager::GetInstance().NotifyNpnsStateChanged(false);

    BackgroundTaskManager::TaskInfo info;

    EXPECT_FALSE(BackgroundTaskManager::GetInstance().GetRunnableTask(&info));
}

TEST(FriendsBackgroundTaskManager, FacedFriendRequestAdded)
{
    BackgroundTaskManager::GetInstance().NotifyFacedFriendRequestAdded(g_Users[0]);

    BackgroundTaskManager::TaskInfo info;

    EXPECT_TRUE(BackgroundTaskManager::GetInstance().GetRunnableTask(&info));
    EXPECT_EQ(info.taskId, BackgroundTaskManager::TaskId_SendFacedFriendRequest);
    EXPECT_EQ(info.uid, g_Users[0]);
    BackgroundTaskManager::GetInstance().NotifyDone(info, nn::ResultSuccess());

    EXPECT_FALSE(BackgroundTaskManager::GetInstance().GetRunnableTask(&info));
}

TEST(FriendsBackgroundTaskManager, UserPresenceUpdated)
{
    BackgroundTaskManager::GetInstance().NotifyUserPresenceUpdated(g_Users[0]);

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(BackgroundTaskManager::PresenceUpdateInterval + 1));
    BackgroundTaskManager::GetInstance().Schedule();

    BackgroundTaskManager::TaskInfo info;

    EXPECT_TRUE(BackgroundTaskManager::GetInstance().GetRunnableTask(&info));
    EXPECT_EQ(info.taskId, BackgroundTaskManager::TaskId_UpdateUserPresence);
    EXPECT_EQ(info.uid, g_Users[0]);
    BackgroundTaskManager::GetInstance().NotifyDone(info, nn::ResultSuccess());

    EXPECT_FALSE(BackgroundTaskManager::GetInstance().GetRunnableTask(&info));
}

TEST(FriendsBackgroundTaskManager, RequestUpdateFriendProfileImageCache)
{
    BackgroundTaskManager::GetInstance().RequestUpdateFriendProfileImageCache(g_Users[0]);

    BackgroundTaskManager::TaskInfo info;

    EXPECT_TRUE(BackgroundTaskManager::GetInstance().GetRunnableTask(&info));
    EXPECT_EQ(info.taskId, BackgroundTaskManager::TaskId_DownloadFriendProfileImage);
    EXPECT_EQ(info.uid, g_Users[0]);
    BackgroundTaskManager::GetInstance().NotifyDone(info, nn::ResultSuccess());

    EXPECT_FALSE(BackgroundTaskManager::GetInstance().GetRunnableTask(&info));
}
