﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testFriends_Common.h"

#include <nn/friends/detail/service/core/friends_AccountStorageManager.h>
#include <nn/friends/detail/service/core/friends_FileSystem.h>
#include <nn/fs/fs_Context.h>

using namespace nn::friends::detail::service::core;

namespace
{
    nn::account::Uid s_Users[nn::account::UserCountMax] = {};
    int s_UserCount = 0;

    nn::os::ThreadType s_Threads[nn::account::UserCountMax] = {};
    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 s_Stacks[nn::account::UserCountMax][32 * 1024] = {};
}

namespace
{
    void AccessUserDataImpl(int index) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(AccountStorageManager::GetInstance().Mount(s_Users[index]));

        NN_UTIL_SCOPE_EXIT
        {
            AccountStorageManager::GetInstance().Unmount();
        };

        char path[256];
        AccountStorageManager::GetInstance().MakePath(path, sizeof (path), "dummy.bin");

        nn::fs::DeleteFile(path);

        NN_ABORT_UNLESS_RESULT_SUCCESS(FileSystem::CreateFile(path, 4));

        {
            nn::fs::FileHandle handle = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write));

            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handle);
            };

            int32_t n = static_cast<int32_t>(index);

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::WriteFile(handle, 0, &n, 4,
                nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(AccountStorageManager::GetInstance().Commit());

        {
            nn::fs::FileHandle handle = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));

            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handle);
            };

            int32_t n;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(handle, 0, &n, 4));

            NN_ABORT_UNLESS_EQUAL(n, static_cast<int32_t>(index));
        }
    }

    void AccessNetworkServiceAccountDataImpl(int index) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(AccountStorageManager::GetInstance().Mount(s_Users[index]));

        NN_UTIL_SCOPE_EXIT
        {
            AccountStorageManager::GetInstance().Unmount();
        };

        char path[256];

        nn::Result result = AccountStorageManager::GetInstance().
            MakePathWithNetworkServiceAccountDirectory(path, sizeof (path), "nsa_dummy.bin");

        if (nn::friends::ResultNetworkServiceAccountNotLinked::Includes(result))
        {
            NN_LOG("[%02d] Network service account is not linked.\n", index);
            return;
        }

        NN_LOG("[%02d] Path = %s\n", index, path);

        nn::fs::DeleteFile(path);

        NN_ABORT_UNLESS_RESULT_SUCCESS(FileSystem::CreateFile(path, 4));

        {
            nn::fs::FileHandle handle = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write));

            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handle);
            };

            int32_t n = static_cast<int32_t>(index);

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::WriteFile(handle, 0, &n, 4,
                nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(AccountStorageManager::GetInstance().Commit());

        {
            nn::fs::FileHandle handle = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));

            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handle);
            };

            int32_t n;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(handle, 0, &n, 4));

            NN_ABORT_UNLESS_EQUAL(n, static_cast<int32_t>(index));
        }
    }

    void AccessUserDataThread(void* arg) NN_NOEXCEPT
    {
        int index = static_cast<int>(reinterpret_cast<intptr_t>(arg));

        for (int c = 0; c < 10; c++)
        {
            NN_LOG("Access[%02d] c=%03d ...\n", index, c);

            AccessUserDataImpl(index);

            NN_LOG("Access[%02d] c=%03d done!\n", index, c);
        }
    }
}

TEST(FriendsAccountStorageManager, Initialize)
{
    NN_FUNCTION_LOCAL_STATIC(nn::fs::FsContext, s_Context
    (
        [](nn::Result) -> nn::fs::AbortSpecifier
        {
            return nn::fs::AbortSpecifier::ReturnResult;
        }
    ));

    nn::fs::SetCurrentThreadFsContext(&s_Context);

    nn::account::InitializeForSystemService();

    nnt::friends::LoadAccounts(&s_UserCount, s_Users, NN_ARRAY_SIZE(s_Users));
    NN_ABORT_UNLESS_GREATER_EQUAL(s_UserCount, 3);

    // 0x800000000000008X で適当なものを選択。
    AccountStorageManager::GetInstance().SetSystemSaveDataId(0x8000000000000089);
}

TEST(FriendsAccountStorageManager, Mount)
{
    NNT_ASSERT_RESULT_SUCCESS(AccountStorageManager::GetInstance().Mount(s_Users[0]));
    AccountStorageManager::GetInstance().Unmount();

    NNT_ASSERT_RESULT_SUCCESS(AccountStorageManager::GetInstance().Mount(s_Users[1]));
    AccountStorageManager::GetInstance().Unmount();

    NNT_ASSERT_RESULT_SUCCESS(AccountStorageManager::GetInstance().Mount(s_Users[2]));
    AccountStorageManager::GetInstance().Unmount();
}

TEST(FriendsAccountStorageManager, AccessUserData)
{
    for (int i = 0; i < s_UserCount; i++)
    {
        AccessUserDataImpl(i);
    }

    for (int i = 0; i < s_UserCount; i++)
    {
        nn::os::CreateThread(&s_Threads[i], AccessUserDataThread, reinterpret_cast<void*>(static_cast<intptr_t>(i)),
            s_Stacks[i], sizeof (s_Stacks[i]), nn::os::DefaultThreadPriority);
        nn::os::StartThread(&s_Threads[i]);
    }
    for (int i = 0; i < s_UserCount; i++)
    {
        nn::os::DestroyThread(&s_Threads[i]);
    }
}

TEST(FriendsAccountStorageManager, DeleteUnmanagedNetworkServiceAccountDirectory1)
{
    for (int i = 0; i < s_UserCount; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(AccountStorageManager::GetInstance().Mount(s_Users[i]));

        NN_UTIL_SCOPE_EXIT
        {
            AccountStorageManager::GetInstance().Unmount();
        };

        NNT_ASSERT_RESULT_SUCCESS(AccountStorageManager::GetInstance().DeleteUnmanagedNetworkServiceAccountDirectory());
    }
}

TEST(FriendsAccountStorageManager, CreateDummyDirectory)
{
    for (int i = 0; i < s_UserCount; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(AccountStorageManager::GetInstance().Mount(s_Users[i]));

        NN_UTIL_SCOPE_EXIT
        {
            AccountStorageManager::GetInstance().Unmount();
        };

        NNT_ASSERT_RESULT_SUCCESS(FileSystem::CreateFile("friends:/0/dummy.bin", 10));
        NNT_ASSERT_RESULT_SUCCESS(FileSystem::CreateFile("friends:/1/dummy.bin", 10));
        NNT_ASSERT_RESULT_SUCCESS(FileSystem::CreateFile("friends:/2/dummy.bin", 10));
        NNT_ASSERT_RESULT_SUCCESS(FileSystem::CreateFile("friends:/X/dummy.bin", 10));
        NNT_ASSERT_RESULT_SUCCESS(FileSystem::CreateFile("friends:/Y/dummy.bin", 10));
        NNT_ASSERT_RESULT_SUCCESS(FileSystem::CreateFile("friends:/Z/dummy.bin", 10));

        NNT_ASSERT_RESULT_SUCCESS(AccountStorageManager::GetInstance().Commit());
    }
}

TEST(FriendsAccountStorageManager, AccessNetworkServiceAccountData)
{
    for (int i = 0; i < s_UserCount; i++)
    {
        AccessNetworkServiceAccountDataImpl(i);
    }
}

TEST(FriendsAccountStorageManager, DeleteUnmanagedNetworkServiceAccountDirectory2)
{
    for (int i = 0; i < s_UserCount; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(AccountStorageManager::GetInstance().Mount(s_Users[i]));

        NN_UTIL_SCOPE_EXIT
        {
            AccountStorageManager::GetInstance().Unmount();
        };

        NNT_ASSERT_RESULT_SUCCESS(AccountStorageManager::GetInstance().DeleteUnmanagedNetworkServiceAccountDirectory());
    }
}

TEST(FriendsAccountStorageManager, DeleteNetworkServiceAccountDirectory)
{
    for (int i = 0; i < s_UserCount; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(AccountStorageManager::GetInstance().Mount(s_Users[i]));

        NN_UTIL_SCOPE_EXIT
        {
            AccountStorageManager::GetInstance().Unmount();
        };

        NNT_ASSERT_RESULT_SUCCESS(AccountStorageManager::GetInstance().DeleteNetworkServiceAccountDirectory());
    }
}

TEST(FriendsAccountStorageManager, FacedFriendRequestProfileImage)
{
    nn::account::Uid uid1 = {{0x0123456789ABCDEFull, 0x0123456789ABCDEFull}};
    nn::account::Uid uid2 = {};

    char url1[160] = {};
    char url2[160] = {};

    nn::account::NetworkServiceAccountId myAccountId = {0x10};
    nn::account::NetworkServiceAccountId accountId = {0x20};

    AccountStorageManager::MakeFacedFriendRequestProfileImageUrl(url1, sizeof (url1), uid1, myAccountId, accountId);

    NN_LOG("Url = %s\n", url1);

    ASSERT_STREQ(url1, "friends-faced-image:/01234567-89ab-cdef-0123-456789abcdef/0000000000000010/0000000000000020.jpg");

    ASSERT_TRUE(AccountStorageManager::ResolveFacedFriendRequestProfileImageUrl(&uid2, url2, sizeof (url2), url1));

    ASSERT_EQ(uid1, uid2);
    ASSERT_STREQ(url2, "friends:/0000000000000010/faced/0000000000000020.jpg");
}
