﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testFriends_Common.h"

// jd1 環境の NA をリンクする必要がある。
//
// 0: フレンド候補取得 API を実行する
//    → テスト用アカウント（https://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=174256161）

namespace
{
    nn::friends::ExternalApplicationCatalogId TestCatalogId =
    {
        {0x77187a302bb9fcd4, 0xf718aa65e0617d6d}
    };
}

namespace
{
    nn::account::Uid s_Users[nn::account::UserCountMax] = {};
    nn::account::NetworkServiceAccountId s_AccountIds[nn::account::UserCountMax] = {};
    int s_UserCount = 0;
}

namespace
{
    void GetCatalog(nn::settings::Language language) NN_NOEXCEPT
    {
        nn::friends::ExternalApplicationCatalog catalog = {};

        nn::settings::LanguageCode languageCode = nn::settings::LanguageCode::Make(language);

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetExternalApplicationCatalog(&context, &catalog, TestCatalogId, languageCode));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        NN_LOG("----------------------------------------------------------------------\n");
        NN_LOG("Language: %s\n", languageCode.string);
        NN_LOG("----------------------------------------------------------------------\n");
        NN_LOG("Catalog.id = %016llx%016llx\n", catalog.id.value[0], catalog.id.value[1]);
        NN_LOG("Catalog.name = %s\n", catalog.name.name);
        NN_LOG("Catalog.nameLanguage = %s\n", catalog.name.language.string);
        NN_LOG("Catalog.iconImageUrl = %s\n", catalog.iconImageUrl.value);
        NN_LOG("----------------------------------------------------------------------\n");
    }
}

class FriendsFriendCandidate : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::account::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());

        nnt::friends::LoadAccounts(&s_UserCount, s_Users, s_AccountIds, NN_ARRAY_SIZE(s_Users));
        NN_ABORT_UNLESS_GREATER_EQUAL(s_UserCount, 1);

        nn::friends::SetOption(nn::friends::OptionAdmin_CheckUserStatus, 0);
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
    }
};

TEST_F(FriendsFriendCandidate, GetList)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    static nn::friends::FriendCandidate s_Candidates[100];
    int count;

    nn::friends::AsyncContext context;
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendCandidateList(&context, &count,
        s_Candidates, s_Users[0], NN_ARRAY_SIZE(s_Candidates), nn::friends::FriendCandidateType_ExternalApplication));

    nn::os::SystemEvent completionEvent;
    NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

    completionEvent.Wait();

    NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

    nnt::friends::Dump(s_Candidates, count);
}

TEST_F(FriendsFriendCandidate, GetCatalog)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    ASSERT_NO_FATAL_FAILURE(GetCatalog(nn::settings::Language_Japanese));
    ASSERT_NO_FATAL_FAILURE(GetCatalog(nn::settings::Language_AmericanEnglish));
    ASSERT_NO_FATAL_FAILURE(GetCatalog(nn::settings::Language_French));
    ASSERT_NO_FATAL_FAILURE(GetCatalog(nn::settings::Language_German));
    ASSERT_NO_FATAL_FAILURE(GetCatalog(nn::settings::Language_Italian));
    ASSERT_NO_FATAL_FAILURE(GetCatalog(nn::settings::Language_Spanish));
    ASSERT_NO_FATAL_FAILURE(GetCatalog(nn::settings::Language_Chinese));
    ASSERT_NO_FATAL_FAILURE(GetCatalog(nn::settings::Language_Korean));
    ASSERT_NO_FATAL_FAILURE(GetCatalog(nn::settings::Language_Dutch));
    ASSERT_NO_FATAL_FAILURE(GetCatalog(nn::settings::Language_Portuguese));
    ASSERT_NO_FATAL_FAILURE(GetCatalog(nn::settings::Language_Russian));
    ASSERT_NO_FATAL_FAILURE(GetCatalog(nn::settings::Language_Taiwanese));
    ASSERT_NO_FATAL_FAILURE(GetCatalog(nn::settings::Language_BritishEnglish));
    ASSERT_NO_FATAL_FAILURE(GetCatalog(nn::settings::Language_CanadianFrench));
    ASSERT_NO_FATAL_FAILURE(GetCatalog(nn::settings::Language_LatinAmericanSpanish));
    ASSERT_NO_FATAL_FAILURE(GetCatalog(nn::settings::Language_SimplifiedChinese));
    ASSERT_NO_FATAL_FAILURE(GetCatalog(nn::settings::Language_TraditionalChinese));

    // 未知の言語コード
    {
        nn::friends::ExternalApplicationCatalog catalog = {};

        nn::settings::LanguageCode languageCode = {"0"};

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetExternalApplicationCatalog(&context, &catalog, TestCatalogId, languageCode));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        // 名前は取得できないがアイコンは取得できる。
        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        EXPECT_STREQ(catalog.name.language.string, "");
    }
}
