﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testFriends_Common.h"

namespace
{
    nn::account::Uid s_Users[nn::account::UserCountMax] = {};
    nn::account::NetworkServiceAccountId s_AccountIds[nn::account::UserCountMax] = {};
    int s_UserCount = 0;
}

class FriendsBlockedUser : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::account::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());

        nnt::friends::LoadAccounts(&s_UserCount, s_Users, s_AccountIds, NN_ARRAY_SIZE(s_Users));
        NN_ABORT_UNLESS_GREATER_EQUAL(s_UserCount, 3);

        nn::friends::SetOption(nn::friends::OptionAdmin_CheckUserStatus, 0);
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
    }
};

TEST_F(FriendsBlockedUser, CleanupRelationship)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nnt::friends::CleanupRelationship(s_Users[0]);
}

TEST_F(FriendsBlockedUser, Block)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::BlockUser(&context, s_Users[0], s_AccountIds[1],
            nn::friends::BlockReason_BadFriendRequest));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        nn::friends::ApplicationInfo appInfo = {{0xFFFFFFFF00000000}, 0xFFFFFFFF00000000};
        nn::friends::InAppScreenName inAppScreenName = {"InAppName2", {"ja"}};

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::BlockUser(&context, s_Users[0], s_AccountIds[2],
            nn::friends::BlockReason_InApp, appInfo, inAppScreenName));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    // 再送
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::BlockUser(&context, s_Users[0], s_AccountIds[1],
            nn::friends::BlockReason_BadFriendRequest));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultAlreadyBlocked, context.GetResult());
    }
    // 存在しないユーザー
    {
        nn::account::NetworkServiceAccountId accountId = {};

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::BlockUser(&context, s_Users[0], accountId,
            nn::friends::BlockReason_BadFriendRequest));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultNetworkServiceAccountNotExistsInBlock, context.GetResult());
    }
    // 自分
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::BlockUser(&context, s_Users[0], s_AccountIds[0],
            nn::friends::BlockReason_BadFriendRequest));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultOwnNetworkServiceAccountSpecified, context.GetResult());
    }

    nn::friends::BlockedUser users[3];
    int count;

    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetBlockedUserList(&count, users, s_Users[0], 0, NN_ARRAY_SIZE(users)));
    EXPECT_EQ(count, 2);

    EXPECT_TRUE(users[0].IsValid());
    EXPECT_TRUE(users[1].IsValid());

    if (users[0].GetAccountId() == s_AccountIds[1])
    {
        EXPECT_EQ(users[0].GetAccountId(), s_AccountIds[1]);
        EXPECT_EQ(users[0].GetBlockReason(), nn::friends::BlockReason_BadFriendRequest);

        EXPECT_EQ(users[1].GetAccountId(), s_AccountIds[2]);
        EXPECT_EQ(users[1].GetBlockReason(), nn::friends::BlockReason_InApp);
    }
    else
    {
        EXPECT_EQ(users[0].GetAccountId(), s_AccountIds[2]);
        EXPECT_EQ(users[0].GetBlockReason(), nn::friends::BlockReason_InApp);

        EXPECT_EQ(users[1].GetAccountId(), s_AccountIds[1]);
        EXPECT_EQ(users[1].GetBlockReason(), nn::friends::BlockReason_BadFriendRequest);
    }

    EXPECT_FALSE(users[2].IsValid());

    nnt::friends::Dump(users, count);
}

TEST_F(FriendsBlockedUser, Unblock)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::UnblockUser(&context, s_Users[0], s_AccountIds[1]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::UnblockUser(&context, s_Users[0], s_AccountIds[2]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    // 同じ相手をもう一度ブロック解除してみる。
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::UnblockUser(&context, s_Users[0], s_AccountIds[1]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultNotBlocked, context.GetResult());
    }

    nn::friends::BlockedUser users[2];
    int count;

    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetBlockedUserList(&count, users, s_Users[0], 0, NN_ARRAY_SIZE(users)));

    EXPECT_EQ(count, 0);

    nnt::friends::Dump(users, count);
}
