﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "testFriends_Common.h"

namespace nnt { namespace friends {

/*!
    @brief      アカウント情報を読み込みます。

    @param[out] pOutCount   読み込んだユーザーアカウントの数。
    @param[out] pOutUsers   ユーザーアカウントリスト。
    @param[in]  count       ユーザーアカウントリストの要素数。

    @pre
        - pOutCount != nullptr
        - pOutUsers != nullptr
        - count > 0
*/
void LoadAccounts(int* pOutCount, nn::account::Uid* pOutUsers, int count) NN_NOEXCEPT;

/*!
    @brief      アカウント情報を読み込みます。

    @param[out] pOutCount       読み込んだユーザーアカウントの数。
    @param[out] pOutUsers       ユーザーアカウントリスト。
    @param[out] pOutAccountIds  ネットワークサービスアカウントリスト。
    @param[in]  count           ユーザーアカウントリストの要素数。

    @pre
        - pOutCount != nullptr
        - pOutUsers != nullptr
        - pOutAccountIds != nullptr
        - count > 0

    @details
                ネットワークサービスアカウントが紐付いていないユーザーアカウントが 1 つ以上存在する場合、ASSERT します。
*/
void LoadAccounts(int* pOutCount, nn::account::Uid* pOutUsers, nn::account::NetworkServiceAccountId* pOutAccountIds, int count) NN_NOEXCEPT;

/*!
    @brief      ネットワークに接続します。

    @details
                ネットワークに接続できなかった場合、ASSERT します。
*/
void ConnectNetwork() NN_NOEXCEPT;

/*!
    @brief      さまざまなユーザー関係を削除し、クリーンな状態にします。

    @param[in]  user    ユーザーアカウント。

    @details
                本関数は以下の処理を行います。

                - フレンドの削除。
                - フレンド申請（対面）の取り消し。
                - フレンド申請（送信）の取り消し。
                - フレンド申請（受信）の削除。
                - ブロックしたユーザーのブロック解除。
*/
void CleanupRelationship(const nn::account::Uid& user) NN_NOEXCEPT;

/*!
    @brief      ユーザー設定をログにダンプします。

    @param[in]  setting ユーザー設定。
*/
void Dump(nn::friends::UserSetting& setting) NN_NOEXCEPT;

/*!
    @brief      フレンドリストをログにダンプします。

    @param[in]  pFriends    フレンドリスト。
    @param[in]  count       フレンドリストの要素数。

    @pre
        - pFriends != nullptr
*/
void Dump(const nn::friends::Friend* pFriends, int count) NN_NOEXCEPT;

/*!
    @brief      フレンドの詳細情報をログにダンプします。

    @param[in]  info    フレンドの詳細情報。

*/
void Dump(const nn::friends::FriendDetailedInfo& info) NN_NOEXCEPT;

/*!
    @brief      フレンド申請リストをログにダンプします。

    @param[in]  pRequests   フレンド申請リスト。
    @param[in]  count       フレンド申請リストの要素数。
    @param[in]  listType    フレンド申請リストの種類。

    @pre
        - pRequests != nullptr
*/
void Dump(const nn::friends::FriendRequest* pRequests, int count, nn::friends::RequestListType listType) NN_NOEXCEPT;

/*!
    @brief      フレンド候補リストをログにダンプします。

    @param[in]  pCandidates フレンド候補リスト。
    @param[in]  count       フレンド候補リストの要素数。

    @pre
        - pCandidates != nullptr
*/
void Dump(const nn::friends::FriendCandidate* pCandidates, int count) NN_NOEXCEPT;

/*!
    @brief      NNID の情報をログにダンプします。

    @param[in]  userInfo    NNID のユーザー情報。
    @param[in]  pFriends    NNID のフレンドリスト。
    @param[in]  count       NNID のフレンドリストの要素数。

    @pre
        - pFriends != nullptr
*/
void Dump(const nn::friends::NintendoNetworkIdUserInfo& userInfo,
    const nn::friends::NintendoNetworkIdFriend* pFriends, int count) NN_NOEXCEPT;

/*!
    @brief      フレンド候補取得用 SNS アカウントのフレンドリストをログにダンプします。

    @param[in]  pFriends    SNS アカウントのフレンドリスト。
    @param[in]  count       SNS アカウントのフレンドリストの要素数。

    @pre
        - pFriends != nullptr
*/
void Dump(const nn::friends::SnsAccountFriend* pFriends, int count) NN_NOEXCEPT;

/*!
    @brief      ブロックリストをログにダンプします。

    @param[in]  pUsers  ブロックリスト。
    @param[in]  count   ブロックリストの要素数。

    @pre
        - pUsers != nullptr
*/
void Dump(const nn::friends::BlockedUser* pUsers, int count) NN_NOEXCEPT;

/*!
    @brief      いっしょに遊んだ記録をログにダンプします。

    @param[in]  pHistories  いっしょに遊んだユーザーリスト。
    @param[in]  count       いっしょに遊んだユーザーリストの要素数。

    @pre
        - pHistories != nullptr
*/
void Dump(const nn::friends::PlayHistory* pHistories, int count) NN_NOEXCEPT;

/*!
    @brief      プロフィールリストをログにダンプします。

    @param[in]  pProfiles   プロフィールリスト。
    @param[in]  count       プロフィールリストの要素数。

    @pre
        - pProfiles != nullptr
*/
void Dump(const nn::friends::Profile* pProfiles, int count) NN_NOEXCEPT;

/*!
    @brief      拡張プロフィールリストをログにダンプします。

    @param[in]  pProfiles   拡張プロフィールリスト。
    @param[in]  count       拡張プロフィールリストの要素数。

    @pre
        - pProfiles != nullptr
*/
void Dump(const nn::friends::ProfileExtra* pProfiles, int count) NN_NOEXCEPT;

}}
