﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/erpt/erpt_Manager.h>
#include <nn/erpt/erpt_Report.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/msgpack.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nn/err/err_Result.h>
#include <nn/fs/fs_ResultHandler.h>

#include <nn/fatalsrv/fatalsrv_CpuContext.h>
#include <functional>

#include "../../../../Programs/Eris/Sources/Libraries/fatalsrv/fatalsrv_ErrorReport.h"

class FatalWriteErrorReportTest : public testing::Test
{
protected:
    static void SetUpTestCase()
    {
        nn::fs::SetEnabledAutoAbort(false);
        nn::fs::SetResultHandledByApplication(false);
    }

    static void TearDownTestCase()
    {
    }
};

TEST_F(FatalWriteErrorReportTest, Basic)
{
    nn::fatalsrv::Service::Context context = {};
    context.fatalContext.lastResult = nn::err::ResultSystemProgramAbort();

    NNT_EXPECT_RESULT_SUCCESS(nn::fatalsrv::WriteErrorReport(context.fatalContext, context.cpuContext, 0));
}

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

struct FatalErrorReport
{
    std::string programId;
    uint32_t abortType;
    std::string errorCode;
    bool fatalFlag;
    bool abortFlag;

    nn::fatalsrv::CpuContext::ArchType archType;
    uint64_t startAddr64;
    uint64_t backtrace64[nn::fatalsrv::Aarch64Context::MaxBacktraceCount];
    uint32_t backtaraceCount64;
    uint64_t registers64[nn::fatalsrv::Aarch64Context::ValueNameCount];
    nn::util::BitFlagSet<nn::fatalsrv::Aarch64Context::ValueNameCount> registerSetFlag64;

    uint32_t startAddr32;
    uint32_t backtrace32[nn::fatalsrv::Aarch32Context::MaxBacktraceCount];
    uint32_t backtaraceCount32;
    uint32_t registers32[nn::fatalsrv::Aarch32Context::ValueNameCount];
    nn::util::BitFlagSet<nn::fatalsrv::Aarch32Context::ValueNameCount> registerSetFlag32;
};

struct ErrorReportContext
{
    nn::fatalsrv::Service::Context context;
    nn::Bit64 programId;
};

nn::Result CallWriteErrorReport(const ErrorReportContext& context)
{
    NN_RESULT_DO(nn::fatalsrv::WriteErrorReport(context.context.fatalContext, context.context.cpuContext, context.programId));
    NN_RESULT_SUCCESS;
}

void CreateFatalErrorReport(
    FatalErrorReport* out,
    uint32_t abortType,
    nn::fatalsrv::CpuContext::ArchType archType,
    const char* errorCode)
{
    *out = {};
    out->programId = "0123456789abcdef";
    out->abortType = abortType;
    out->errorCode = errorCode;
    out->fatalFlag = true;
    out->abortFlag = true;

    if (archType == nn::fatalsrv::CpuContext::Aarch64)
    {
        out->startAddr64 = 0xffffeeeeddddccccull;
        for (int i = 0; i < nn::fatalsrv::Aarch64Context::MaxBacktraceCount; ++i)
        {
            out->backtrace64[i] = 0x1000000000000000ull + i;
        }
        out->backtaraceCount64 = nn::fatalsrv::Aarch64Context::MaxBacktraceCount;

        for (int i = 0; i < nn::fatalsrv::Aarch64Context::ValueNameCount; ++i)
        {
            out->registers64[i] = 0x2000000000000000ull + i;
            out->registerSetFlag64.Set(i);
        }
    }
    else if (archType == nn::fatalsrv::CpuContext::Aarch32)
    {
        out->startAddr32 = 0x11112222;
        for (int i = 0; i < nn::fatalsrv::Aarch32Context::MaxBacktraceCount; ++i)
        {
            out->backtrace32[i] = 0x30000000 + i;
        }
        out->backtaraceCount32 = nn::fatalsrv::Aarch32Context::MaxBacktraceCount;

        for (int i = 0; i < nn::fatalsrv::Aarch32Context::ValueNameCount; ++i)
        {
            out->registers32[i] = 0x40000000 + i;
            out->registerSetFlag32.Set(i);
        }
    }
}


void StringParser(void* out, size_t maxOutSize, const std::string& key, nn::msgpack::MpWalker& mpWalker)
{
    auto v = mpWalker[key.c_str()];
    if (v.GetSize() > 0)
    {
        const char* pStr;
        uint32_t strSize;
        v.GetString(&pStr, &strSize);


        NN_ASSERT(strSize < maxOutSize);
        std::unique_ptr<char[]> tmp(new char[strSize + 1]);
        tmp[strSize] = '\0';
        std::memcpy(tmp.get(), pStr, strSize);

        auto outString = reinterpret_cast<std::string*>(out);
        *outString = tmp.get();
    }
}

void UInt32Parser(void* out, size_t maxOutSize, const std::string& key, nn::msgpack::MpWalker& mpWalker)
{
    NN_UNUSED(maxOutSize);
    NN_ASSERT(sizeof(uint32_t) <= maxOutSize);

    auto v = mpWalker[key.c_str()];
    if (v.GetSize() > 0)
    {
        v.GetUint(reinterpret_cast<uint32_t*>(out));
    }
}

void UInt64Parser(void* out, size_t maxOutSize, const std::string& key, nn::msgpack::MpWalker& mpWalker)
{
    NN_UNUSED(maxOutSize);
    NN_ASSERT(sizeof(uint64_t) <= maxOutSize);

    auto v = mpWalker[key.c_str()];
    if (v.GetSize())
    {
        v.GetUint(reinterpret_cast<uint64_t*>(out));
    }
}

void BoolParser(void* out, size_t maxOutSize, const std::string& key, nn::msgpack::MpWalker& mpWalker)
{
    NN_UNUSED(maxOutSize);
    NN_ASSERT(sizeof(bool) <= maxOutSize);

    auto v = mpWalker[key.c_str()];
    if (v.GetSize())
    {
        v.GetBoolean(reinterpret_cast<bool*>(out));
    }
}

struct ParseInfo
{
    std::string keyName;
    std::function<void(void*, size_t, const std::string&, nn::msgpack::MpWalker& walker)> parse;
    void* out;
    size_t outSize;
};

void ReadErrorReport(FatalErrorReport* out)
{
    nn::erpt::ReportList reportList;
    nn::erpt::Manager manager;

    NNT_ASSERT_RESULT_SUCCESS(manager.Initialize());
    NN_UTIL_SCOPE_EXIT{ manager.Finalize(); };

    NNT_EXPECT_RESULT_SUCCESS(manager.GetReportList(reportList));
    auto reportCount = reportList.reportCount;

    NN_ABORT_UNLESS(reportCount > 0);

    auto reportId = reportList.Report[0].reportId;

    nn::erpt::Report report;
    NNT_ASSERT_RESULT_SUCCESS(report.Open(reportId));

    int64_t reportSize;
    NNT_ASSERT_RESULT_SUCCESS(report.GetSize(&reportSize));

    std::unique_ptr<uint8_t[]> reportData(new uint8_t[static_cast<uint32_t>(reportSize)]);
    EXPECT_TRUE(reportData != nullptr);

    uint32_t readCount;
    NNT_ASSERT_RESULT_SUCCESS(report.Read(&readCount, reportData.get(), static_cast<uint32_t>(reportSize)));

    nn::msgpack::MpWalker mpWalker;
    mpWalker.Init(reportData.get(), readCount);

    *out = {};

    ParseInfo parseInfoTable[] = {
        { "ProgramId", StringParser, &out->programId, sizeof(out->programId) },
        { "ErrorCode", StringParser, &out->errorCode, sizeof(out->errorCode) },
        { "AbortType", UInt32Parser, &out->abortType, sizeof(out->abortType) },
        { "AbortFlag", BoolParser,   &out->abortFlag, sizeof(out->abortFlag) },
        { "FatalFlag", BoolParser,   &out->fatalFlag, sizeof(out->fatalFlag) },
        { "ProgramMappedAddr64", UInt64Parser,   &out->startAddr64, sizeof(out->startAddr64) },
        { "ProgramMappedAddr32", UInt32Parser,   &out->startAddr32, sizeof(out->startAddr32) },
    };
    const int ParseInfoCount = sizeof(parseInfoTable) / sizeof(ParseInfo);

    for (int i = 0; i < ParseInfoCount; ++i)
    {
        auto& parseInfo = parseInfoTable[i];
        parseInfo.parse(parseInfo.out, parseInfo.outSize, parseInfo.keyName, mpWalker);
    }

    {
        auto v = mpWalker["StackBacktrace64"];
        if (v.GetSize() > 0)
        {
            v.GetArrayCount(&out->backtaraceCount64);
            for (int i = 0; i < out->backtaraceCount64; ++i)
            {
                auto backtrace = v.At(i);
                backtrace.GetUint(&out->backtrace64[i]);
            }
        }
    }
    {
        auto v = mpWalker["StackBacktrace32"];
        if (v.GetSize() > 0)
        {
            v.GetArrayCount(&out->backtaraceCount32);
            for (int i = 0; i < out->backtaraceCount32; ++i)
            {
                auto backtrace = v.At(i);
                backtrace.GetUint(&out->backtrace32[i]);
            }
        }
    }
    {
        auto v = mpWalker["RegisterSetFlag64"];
        if (v.GetSize() > 0)
        {
            v.GetUint(&out->registerSetFlag64._storage[0]);

            auto regs = mpWalker["GeneralRegisterAarch64"];
            for (int i = 0; i < nn::fatalsrv::Aarch64Context::ValueNameCount; ++i)
            {
                if (out->registerSetFlag64.Test(i))
                {
                    regs.At(i).GetUint(&out->registers64[i]);
                }
            }
        }
    }
    {
        auto v = mpWalker["RegisterSetFlag32"];
        if (v.GetSize())
        {
            v.GetUint(&out->registerSetFlag32._storage[0]);

            auto regs = mpWalker["GeneralRegisterAarch32"];
            for (int i = 0; i < nn::fatalsrv::Aarch32Context::ValueNameCount; ++i)
            {
                if (out->registerSetFlag32.Test(i))
                {
                    regs.At(i).GetUint(&out->registers32[i]);
                }
            }
        }
    }
}

void CreateErrorReportContext(
    ErrorReportContext* out,
    nn::os::UserExceptionType exceptionType,
    nn::Result lastResult,
    nn::fatalsrv::CpuContext::ArchType archType)
{
    *out = {};
    out->programId = 0x0123456789abcdefull;
    out->context.fatalContext.lastResult = lastResult;
    out->context.cpuContext.archType = archType;
    out->context.cpuContext.exceptionType = exceptionType;

    if (archType == nn::fatalsrv::CpuContext::Aarch64)
    {
        out->context.cpuContext.aarch64.programMappedAddr = 0xffffeeeeddddccccull;
        for (int i = 0; i < nn::fatalsrv::Aarch64Context::MaxBacktraceCount; ++i)
        {
            out->context.cpuContext.aarch64.backtrace[i] = 0x1000000000000000ull + i;
        }
        out->context.cpuContext.aarch64.backtraceCount = nn::fatalsrv::Aarch64Context::MaxBacktraceCount;

        for (int i = 0; i < nn::fatalsrv::Aarch64Context::ValueNameCount; ++i)
        {
            out->context.cpuContext.aarch64.values[i] = 0x2000000000000000ull + i;
            out->context.cpuContext.aarch64.setFlag.Set(i);
        }
    }
    else if (archType == nn::fatalsrv::CpuContext::Aarch32)
    {
        out->context.cpuContext.aarch32.programMappedAddr = 0xbbbbaaaaull;
        for (int i = 0; i < nn::fatalsrv::Aarch32Context::MaxBacktraceCount; ++i)
        {
            out->context.cpuContext.aarch32.backtrace[i] = 0x30000000ull + i;
        }
        out->context.cpuContext.aarch32.backtraceCount = nn::fatalsrv::Aarch32Context::MaxBacktraceCount;

        for (int i = 0; i < nn::fatalsrv::Aarch32Context::ValueNameCount; ++i)
        {
            out->context.cpuContext.aarch32.values[i] = 0x40000000ull + i;
            out->context.cpuContext.aarch32.setFlag.Set(i);
        }
    }
}

void TestReport(const FatalErrorReport& expected, const FatalErrorReport& actual)
{
    EXPECT_EQ(expected.programId, actual.programId);
    EXPECT_EQ(expected.errorCode, actual.errorCode);
    EXPECT_EQ(expected.abortType, actual.abortType);
    EXPECT_EQ(expected.archType, actual.archType);
    EXPECT_EQ(expected.abortFlag, actual.abortFlag);
    EXPECT_EQ(expected.fatalFlag, actual.fatalFlag);
    EXPECT_EQ(expected.backtaraceCount32, actual.backtaraceCount32);
    EXPECT_EQ(expected.backtaraceCount64, actual.backtaraceCount64);
    EXPECT_EQ(expected.startAddr32, actual.startAddr32);
    EXPECT_EQ(expected.startAddr64, actual.startAddr64);
    EXPECT_EQ(expected.registerSetFlag32, actual.registerSetFlag32);
    EXPECT_EQ(expected.registerSetFlag64, actual.registerSetFlag64);
    EXPECT_TRUE(std::memcmp(expected.backtrace32, actual.backtrace32, sizeof(expected.backtrace32)) == 0);
    EXPECT_TRUE(std::memcmp(expected.backtrace64, actual.backtrace64, sizeof(expected.backtrace64)) == 0);
    EXPECT_TRUE(std::memcmp(expected.registers32, actual.registers32, sizeof(expected.registers32)) == 0);
    EXPECT_TRUE(std::memcmp(expected.registers64, actual.registers64, sizeof(expected.registers64)) == 0);
}

TEST_F(FatalWriteErrorReportTest, ExceptionTypeAbort)
{
    ErrorReportContext context;
    CreateErrorReportContext(&context, nn::os::UserExceptionType::UserExceptionType_None, nn::ResultSuccess(), nn::fatalsrv::CpuContext::Aarch64);
    NNT_EXPECT_RESULT_SUCCESS(CallWriteErrorReport(context));

    FatalErrorReport errorReport;
    ReadErrorReport(&errorReport);

    FatalErrorReport expected;
    CreateFatalErrorReport(&expected, 0, nn::fatalsrv::CpuContext::Aarch64, "2162-0002");

    TestReport(expected, errorReport);
}

TEST_F(FatalWriteErrorReportTest, ExceptionTypeAbortWithResult)
{
    ErrorReportContext context;
    CreateErrorReportContext(&context, nn::os::UserExceptionType::UserExceptionType_None, nn::erpt::ResultAlreadyExist(), nn::fatalsrv::CpuContext::Aarch64);
    NNT_EXPECT_RESULT_SUCCESS(CallWriteErrorReport(context));

    FatalErrorReport errorReport;
    ReadErrorReport(&errorReport);

    FatalErrorReport expected;
    CreateFatalErrorReport(&expected, 0, nn::fatalsrv::CpuContext::Aarch64, "2147-0011");

    TestReport(expected, errorReport);
}

#endif
