﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/init/init_Malloc.h>
#include <nn/fatalsrv/fatalsrv_Server.h>

#include <nn/fatal/fatal_ApiPrivate.h>

#include "../../../../Programs/Eris/Sources/Libraries/fatalsrv/fatalsrv_Environment.h"
#include "../../../../Programs/Eris/Sources/Libraries/fatalsrv/fatalsrv_PowerControl.h"

#include <nn/psm/psm.h>
#include <nn/psm/psm_SystemProcess.h>

namespace {
    class PowerControlButton : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            nn::fatalsrv::InitializeEnvironmentInfo();
        }

        static void TearDownTestCase()
        {
        }


    private:
    };

    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_TaskThreadStack[4 * 1024];

    void TestPowerButton(nn::fatalsrv::PowerButtonObserveTask& task)
    {
        // 実際のタスクスレッドと同じスタックサイズで動かす
        nn::os::ThreadType taskThread;
        nn::os::CreateThread(&taskThread, [](void* arg) {
            auto task = reinterpret_cast<nn::fatalsrv::PowerButtonObserveTask*>(arg);
            task->Run();
        }, &task, g_TaskThreadStack, sizeof(g_TaskThreadStack), nn::os::DefaultThreadPriority);

        nn::os::StartThread(&taskThread);
        nn::os::WaitThread(&taskThread);
        nn::os::DestroyThread(&taskThread);
    }

    class PowerControllerBase : public nn::fatalsrv::IPowerController
    {
    public:
        NN_IMPLICIT PowerControllerBase(nn::os::Event& rebootEvent) NN_NOEXCEPT : m_RebootEvent(rebootEvent) {}
        virtual void Shutdown() NN_NOEXCEPT {  }
        virtual void Reboot() NN_NOEXCEPT { m_RebootEvent.Signal(); }
        virtual nn::psm::BatteryVoltageState GetBatteryVoltageState() NN_NOEXCEPT { return nn::psm::BatteryVoltageState_ShutdownRequired; }
        virtual nn::bpc::SleepButtonState GetSleepButtonState() NN_NOEXCEPT { return nn::bpc::SleepButtonState_Released; }
        virtual int GetMaxShutdownWaitSec() const NN_NOEXCEPT { return 1; }
    private:
        nn::os::Event& m_RebootEvent;
    };
}


TEST_F(PowerControlButton, Basic)
{
    class PowerController : public PowerControllerBase
    {
    public:
        NN_IMPLICIT PowerController(nn::os::Event& rebootEvent) : PowerControllerBase(rebootEvent){}
        virtual nn::bpc::SleepButtonState GetSleepButtonState() NN_NOEXCEPT
        {
            NN_FUNCTION_LOCAL_STATIC(int, s_Count, = 0);
            if (s_Count++ == 0)
            {
                return nn::bpc::SleepButtonState_Released;
            }
            else
            {
                return nn::bpc::SleepButtonState_Pushed;
            }
        }
    };

    nn::os::Event rebootEvent(nn::os::EventClearMode_AutoClear);
    PowerController controller(rebootEvent);

    nn::os::Event errorReportWrittenEvent(nn::os::EventClearMode_AutoClear);
    errorReportWrittenEvent.Signal();

    nn::fatalsrv::PowerButtonObserveTask task;
    task.Initialize(&errorReportWrittenEvent, &controller);

    TestPowerButton(task);
    EXPECT_EQ(true, rebootEvent.TimedWait(nn::TimeSpan::FromSeconds(1)));
}

// INFO: 下記のテストは実際に電源ボタン押しを待ち、そこから再起動する処理を走らせるものです。
//       実際の挙動を確認するときに有効化してください。
#if 0
TEST_F(PowerControlButton, Prod)
{
    nn::os::Event errorReportWrittenEvent(nn::os::EventClearMode_AutoClear);
    errorReportWrittenEvent.Signal();

    nn::fatalsrv::PowerButtonObserveTask task(errorReportWrittenEvent);

    TestPowerButton(task);
}
#endif
