﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/erpt.h>
#include <nn/erpt/erpt_Manager.h>
#include <nn/eupld/eupld_Request.h>
#include <nn/eupld/eupld_Result.h>
#include <nn/eupld/eupld_Control.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nnt.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/eupld/server/eupld_ServerApi.h>
#endif

namespace nnt   {
namespace eupld {

class EupldTest : public ::testing::Test
{
protected:
    static nn::nifm::NetworkConnection& GetNetworkConnection()
    {
        NN_FUNCTION_LOCAL_STATIC(nn::nifm::NetworkConnection, nc);
        return nc;
    }

    virtual void SetUp()
    {
    }

    virtual void TearDown()
    {
    }

    static void SetUpTestCase()
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::nifm::InitializeSystem());
        auto& nc = GetNetworkConnection();
        // SystemMenuApplet triggers manual error upload, thus using RequirementPreset_InternetForApplet.
        nn::nifm::SetRequestRequirementPreset(nc.GetRequestHandle(), nn::nifm::RequirementPreset_InternetForApplet);
        nc.SubmitRequestAndWait();
    }

    static void TearDownTestCase()
    {
        GetNetworkConnection().CancelRequest();
    }
};

TEST(eupld, InitRequest)
{
    nn::eupld::Request uploadRequest;
    nn::Result         result;

    result = uploadRequest.Initialize();
    NNT_ASSERT_RESULT_SUCCESS(result);
}

TEST(eupld, DoubleInitRequest)
{
    nn::eupld::Request uploadRequest;
    nn::Result         result;

    result = uploadRequest.Initialize();
    NNT_ASSERT_RESULT_SUCCESS(result);

    result = uploadRequest.Initialize();
    NNT_ASSERT_RESULT_FAILURE(nn::eupld::ResultAlreadyInitialized, result);
}

TEST_F(EupldTest, UnitializedRequest)
{
    nn::eupld::Request uploadRequest;
    nn::Result         result;

    result = uploadRequest.UploadAll();
    NNT_ASSERT_RESULT_FAILURE(nn::eupld::ResultNotInitialized, result);

    uploadRequest.Finalize();
}

TEST(eupld, DISABLED_SetUrl)
{
    nn::Result         result;
    nn::eupld::Control control;
    const char         tmpUrl[] = "https://receive-%.er.srv.nintendo.net/post";

    result = control.SetUrl(tmpUrl, static_cast<uint32_t>(strlen(tmpUrl) + 1));
    NNT_ASSERT_RESULT_SUCCESS(result);
}

TEST(eupld, GenerateReports)
{
    nn::Result        result;
    const char        errorCode[] = "1234-1234";
    nn::erpt::Context context(nn::erpt::ErrorInfo);

    result = context.Add(nn::erpt::ErrorCode, errorCode, sizeof(errorCode));
    NNT_ASSERT_RESULT_SUCCESS(result);

    for (int type = nn::erpt::ReportType_First;
             type < nn::erpt::ReportType_Last;
             type++)
    {
        result = context.CreateReport(static_cast<nn::erpt::ReportType>(type));
        NNT_ASSERT_RESULT_SUCCESS(result);
    }
}

TEST_F(EupldTest, UploadAll)
{
    nn::eupld::Request          uploadRequest;
    nn::eupld::ReportUploadList reportStatus;
    nn::Result                  result;

    result = uploadRequest.Initialize();
    NNT_ASSERT_RESULT_SUCCESS(result);

    if ((result = uploadRequest.UploadAll()).IsSuccess())
    {
        ASSERT_TRUE(uploadRequest.GetEventPointer()->TimedWait(nn::TimeSpan::FromSeconds(60)));

        result = uploadRequest.GetResult();
        if( nn::eupld::ResultUrlNotSet::Includes(result) )
        {
            NN_LOG("URL is not set. Upload is disabled.\n");
        }
        else
        {
            NNT_ASSERT_RESULT_SUCCESS(result);
        }

        NN_LOG(
            "upload state: %u %u\n",
            result.GetModule(),
            result.GetDescription()
        );

        result = uploadRequest.GetUploadStatus(reportStatus);
        NNT_ASSERT_RESULT_SUCCESS(result);

        for (uint32_t i = 0;
                      i < reportStatus.reportCount;
                      i++)
        {

            char idStr[64];
            reportStatus.Report[i].reportId.u.uuidRFC4122.ToString(idStr, sizeof(idStr));
            NN_LOG(
                "id: %s\n",
                idStr);
            NN_LOG(
                "result: %u %u\n",
                reportStatus.Report[i].result.GetModule(),
                reportStatus.Report[i].result.GetDescription()
            );
        }
    }
    else
    {
        NNT_ASSERT_RESULT_FAILURE(nn::eupld::ResultNoReportsToSend, result);
    }
}

TEST_F(EupldTest, UploadSelected)
{
    nn::eupld::Request          uploadRequest;
    nn::eupld::ReportUploadList reportStatus;
    nn::erpt::ReportList        reportList;
    nn::erpt::Manager           reportManager;
    nn::erpt::ReportId          reportId[nn::erpt::NumberOfReports];
    nn::Result                  result;

    result = reportManager.Initialize();
    NNT_ASSERT_RESULT_SUCCESS(result);

    result = reportManager.GetReportList(reportList, nn::erpt::ReportType_Visible);
    NNT_ASSERT_RESULT_SUCCESS(result);

    for (uint32_t i = 0;
                  i < reportList.reportCount;
                  i++)
    {
        reportId[i] = reportList.Report[i].reportId;
    }

    result = uploadRequest.Initialize();
    NNT_ASSERT_RESULT_SUCCESS(result);

    result = uploadRequest.UploadSelected(reportId, reportList.reportCount);
    NNT_ASSERT_RESULT_SUCCESS(result);

    ASSERT_TRUE(uploadRequest.GetEventPointer()->TimedWait(nn::TimeSpan::FromSeconds(200)));

    result = uploadRequest.GetResult();
    if( nn::eupld::ResultUrlNotSet::Includes(result) )
    {
        NN_LOG("URL is not set. Upload is disabled.\n");
    }
    else
    {
        NNT_ASSERT_RESULT_SUCCESS(result);
    }

    NN_LOG(
        "upload state: %u %u\n",
        result.GetModule(),
        result.GetDescription()
    );

    result = uploadRequest.GetUploadStatus(reportStatus);
    NNT_ASSERT_RESULT_SUCCESS(result);

    for (uint32_t i = 0;
                  i < reportList.reportCount;
                  i++)
    {
        char idStr[64];
        reportStatus.Report[i].reportId.u.uuidRFC4122.ToString(idStr, sizeof(idStr));
        NN_LOG(
            "id: %s\n",
            idStr);
        NN_LOG(
            "result: %u %u\n",
            reportStatus.Report[i].result.GetModule(),
            reportStatus.Report[i].result.GetDescription()
        );
    }
}

TEST_F(EupldTest, CancelUpload)
{
    nn::Result                  result;
    nn::eupld::Request          uploadRequest;
    nn::eupld::ReportUploadList reportStatus;
    nn::erpt::ReportList        reportList;
    nn::erpt::Manager           reportManager;

    result = reportManager.Initialize();
    NNT_ASSERT_RESULT_SUCCESS(result);

    result = reportManager.GetReportList(reportList);
    NNT_ASSERT_RESULT_SUCCESS(result);

    result = uploadRequest.Initialize();
    NNT_ASSERT_RESULT_SUCCESS(result);

    ASSERT_GT(reportList.reportCount, 0u);

    result = uploadRequest.UploadSelected(&reportList.Report[0].reportId, 1);
    NNT_ASSERT_RESULT_SUCCESS(result);

    result = uploadRequest.CancelUpload();
    if (result <= nn::eupld::ResultCancelInProgress())
    {
        ASSERT_TRUE(uploadRequest.GetEventPointer()->TimedWait(nn::TimeSpan::FromSeconds(200)));
    }

    result = uploadRequest.GetResult();
    NNT_ASSERT_RESULT_SUCCESS(result);

    NN_LOG(
        "upload state: %u %u\n",
        result.GetModule(),
        result.GetDescription()
    );

    result = uploadRequest.GetUploadStatus(reportStatus);
    NNT_ASSERT_RESULT_SUCCESS(result);

    for (uint32_t i = 0;
                  i < reportStatus.reportCount;
                  i++)
    {
        char idStr[64];
        reportStatus.Report[i].reportId.u.uuidRFC4122.ToString(idStr, sizeof(idStr));
        NN_LOG(
            "id: %s\n",
            idStr);
        NN_LOG(
            "result: %u %u\n",
            reportStatus.Report[i].result.GetModule(),
            reportStatus.Report[i].result.GetDescription()
        );
    }
}

TEST_F(EupldTest, UploadAbuse)
{
    nn::Result              result;
    nn::eupld::Request      uploadRequest;
    nn::erpt::ReportList    reportList;
    nn::erpt::Manager       reportManager;

    const uint32_t IterationCount = 100;

    result = reportManager.Initialize();
    NNT_ASSERT_RESULT_SUCCESS(result);

    result = reportManager.GetReportList(reportList);
    NNT_ASSERT_RESULT_SUCCESS(result);

    for (int i = 0;
             i < IterationCount;
             i++)
    {
        result = uploadRequest.Initialize();
        NNT_ASSERT_RESULT_SUCCESS(result);

        result = uploadRequest.UploadSelected(&reportList.Report[0].reportId, 1);
        NNT_ASSERT_RESULT_SUCCESS(result);

        NN_LOG("Uploading %d/%d...\n", i + 1, IterationCount );
        auto isUploadCompleted = uploadRequest.GetEventPointer()->TimedWait(nn::TimeSpan::FromSeconds(60));
        EXPECT_TRUE(isUploadCompleted);
        if( !isUploadCompleted )
        {
            auto cancelResult = uploadRequest.CancelUpload();
            if( nn::eupld::ResultCancelInProgress::Includes(cancelResult) )
            {
                ASSERT_TRUE(uploadRequest.GetEventPointer()->TimedWait(nn::TimeSpan::FromSeconds(60)));
            }
            else
            {
                NNT_ASSERT_RESULT_SUCCESS(cancelResult);
            }
            uploadRequest.Finalize();
            continue;
        }

        result = uploadRequest.GetResult();
        if( nn::eupld::ResultUrlNotSet::Includes(result) )
        {
            NN_LOG("URL is not set. Upload is disabled.\n");
        }
        else
        {
            NNT_ASSERT_RESULT_SUCCESS(result);
        }

        uploadRequest.Finalize();
    }
}

}}
