﻿/*
 *  Copyright 2005-2014 Acer Cloud Technology, Inc.
 *  All Rights Reserved.
 *
 *  This software contains confidential information and
 *  trade secrets of Acer Cloud Technology, Inc.
 *  Use, disclosure or reproduction is prohibited without
 *  the prior express written permission of Acer Cloud
 *  Technology, Inc.
 */

/*
 *               Copyright (C) 2010, BroadOn Communications Corp.
 *
 *  These coded instructions, statements, and computer programs contain
 *  unpublished  proprietary information of BroadOn Communications Corp.,
 *  and  are protected by Federal copyright law. They may not be disclosed
 *  to  third  parties or copied or duplicated in any form, in whole or in
 *  part, without the prior written consent of BroadOn Communications Corp.
 *
 */
#include <nnt/nntest.h>
#include <nn/fs.h>

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <iostream>

#include <nn/iosc/iosc.h>
#include <nn/ioscrypto/iosccert.h>
#include <nn/escore/estypes.h>
USING_ES_NAMESPACE
#include <nn/publish/publish.h>
USING_PUBLISH_NAMESPACE
#include <nn/ioscrypto/iosendian.h>

#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>

#include "../Common/testEs_Tkt6.cpp"

static char __ticketFile[] = "00000000.tik";

#define CERTS_SIGNED_WITH_SHA256
#ifdef CERTS_SIGNED_WITH_SHA256
static const u8 issuerName[] = "Root-CA00000004-XS00000009";
#else
static const u8 issuerName[] = "Root-CA00000002-XS00000006";
#endif

static u8 ticketBuf[512 * 1024];

// static ESItemRight rightsEntries[8];

#define CHECK_RESULT( exp, rval )                                         \
    {                                                                     \
        if( ( rv = ( exp ) ) != ( rval ) )                                \
        {                                                                 \
            NN_LOG( "Failure at line %d: returns %d\n", __LINE__, rval ); \
            ADD_FAILURE();                                                \
            if( rv == ES_ERR_OK )                                         \
            {                                                             \
                rv = ES_ERR_INVALID;                                      \
            }                                                             \
            goto out;                                                     \
        }                                                                 \
        else                                                              \
        {                                                                 \
            SUCCEED();                                                    \
        }                                                                 \
    }


#ifdef CERTS_SIGNED_WITH_SHA256
static char rsaPubModFile[] = "xs9_dpki_rsa.pubMod";
static char rsaPrivExpFile[] = "xs9_dpki_rsa.privExp";
#else
static char rsaPubModFile[] = "xs_dpki_rsa.pubMod";
static char rsaPrivExpFile[] = "xs_dpki_rsa.privExp";
#endif
static u8 rsaPubMod[256];
static u8 rsaPrivExp[256];

static char xsEccPrivFile[] = "xs_dpki.eccPvtKey";
static char xsEccPubFile[] = "xs_dpki.eccPubKey";
static IOSCEccPrivateKey xsEccPrivKey;
static IOSCEccPublicKey xsEccPubKey;

static char devCertFile[] = "dev_ng_dpki.cert";
static IOSCEccEccCert devCert;

static char rvlCommonKeyFile[] = "common_dpki.aesKey";
static char evlCommonKeyFile[] = "common_evl_dpki.aesKey";
static char etdCommonKey0File[] = "common_etd_devkey0_npki.aesKey";
static char etdCommonKey1File[] = "common_etd_devkey1_npki.aesKey";

static u8 rvlCommonKey[16];
static u8 evlCommonKey[16];
static u8 etdCommonKey0[16];
static u8 etdCommonKey1[16];

static IOSCSecretKeyHandle hCommonKey;
static u8 commonKeyId;

static void dumpData( const char *msg, u8 *array, int size )
{
    int i;

    NN_LOG( "%s", msg );
    for( i = 0; i < size; i++ )
    {
        if( i % 16 == 0 )
        {
            NN_LOG( "\n" );
        }
        NN_LOG( "%02X ", array[i] );
    }
    NN_LOG( "\n" );
}

static char *makePath( const char *relPath )
{
    static char pathName[256];

    strncpy( pathName, "..\\..\\PkiData\\", sizeof( pathName ) );
    strncat( pathName, relPath, sizeof( pathName ) - strlen(relPath) - 1);

    return pathName;
}

static const char *makeAbsPath( const char *fileName )
{
    static std::string path;

    //#define NNT_ES_SIGLO_ROOT set by nact script or vcxporj to the root path of Siglo.
    const char* kResourcePath = "/Tests/Escore/Sources/Tests/PublishRepublish/testEs_PublishRepublish/";
    path = std::string(NNT_ES_SIGLO_ROOT) + std::string(kResourcePath) + fileName;

    return path.c_str();
}

static int readFile( const char *fileName, void* buffer, size_t size, u32* readSize = NULL)
{
    ESError rv = ES_ERR_OK;

    nn::fs::FileHandle file;
    nn::Result result;
    result = nn::fs::OpenFile(&file, makeAbsPath(makePath(fileName)), nn::fs::OpenMode_Read);
    if ( result.IsFailure() )
    {
        NN_LOG( "Open failed for %s\n", fileName);
        rv = ES_ERR_FAIL;
        return rv;
    }

    size_t fileSize = 0;
    result = nn::fs::ReadFile(&fileSize, file, 0, buffer, size, nn::fs::ReadOption());
    if (fileSize == 0)
    {
        NN_LOG( "read failed for %s\n", fileName);
        rv = ES_ERR_FAIL;
    }
    if (readSize != NULL)
    {
        *readSize = static_cast<u32>(fileSize);
    }

    nn::fs::CloseFile(file);
    return rv;
}

static void writeFile( const char *fileName, const void* buffer, size_t size)
{
    nn::fs::FileHandle file;
    nn::Result result;
    nn::fs::DeleteFile(makeAbsPath(fileName));
    result = nn::fs::CreateFile(makeAbsPath(fileName), size);
    if ( result.IsFailure() )
    {
        NN_LOG( "create failed for %s", fileName );
        GTEST_FATAL_FAILURE_(fileName);
    }

    result = nn::fs::OpenFile(&file, makeAbsPath(fileName), nn::fs::OpenMode_Write);
    if ( result.IsFailure() )
    {
        NN_LOG( "open failed for %s", fileName );
        GTEST_FATAL_FAILURE_(fileName);
    }

    result = nn::fs::WriteFile(file, 0, buffer, size, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
    if (result.IsFailure())
    {
        NN_LOG( "write failed for %s", fileName );
        GTEST_FATAL_FAILURE_(fileName);
    }
    nn::fs::CloseFile(file);
}

/*
 * This program acts as the XS server, so it needs both the
 * RSA key pair (used for signing tickets) and the ECC key pair,
 * used for doing ECDH in ticket personalization.
 */
static void initKeys()
{
    ESError rv = ES_ERR_OK;

    rv = readFile( rsaPubModFile, rsaPubMod, sizeof( rsaPubMod ) );
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        return;
    }

    rv = readFile( rsaPrivExpFile, rsaPrivExp, sizeof( rsaPrivExp ) );
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        return;
    }

    rv = readFile( xsEccPrivFile, xsEccPrivKey, sizeof( xsEccPrivKey ) );
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        return;
    }

    rv = readFile( xsEccPubFile, xsEccPubKey, sizeof( xsEccPubKey ) );
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        return;
    }

    rv = readFile( rvlCommonKeyFile, rvlCommonKey, sizeof( rvlCommonKey ) );
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        return;
    }

    rv = readFile( evlCommonKeyFile, evlCommonKey, sizeof( evlCommonKey ) );
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        return;
    }

    rv = readFile( etdCommonKey0File, etdCommonKey0, sizeof( etdCommonKey0 ) );
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        return;
    }

    rv = readFile( etdCommonKey1File, etdCommonKey1, sizeof( etdCommonKey1 ) );
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        return;
    }
}

/*
 * The device cert is used by the XS server to obtain the device ID
 * and the device public key for ticket personalization
 */
static void loadDevCert()
{
    ESError rv = ES_ERR_OK;

    rv = readFile( devCertFile, &devCert, sizeof( devCert ) );
    EXPECT_EQ( rv, ES_ERR_OK );
    if( rv != ES_ERR_OK )
    {
        return;
    }

    return;
}

static void dumpRightsId(ESRightsId rightsid)
{
    for (int i = 0; i < sizeof(ESRightsId); i++)
    {
        NN_LOG("%02X", rightsid[i]);
    }
    NN_LOG("\n");
}

/*
 * Use an input ticket template to republish a ticket with a different
 * common key or other attributes
 */
static ESError doPublishTicket()
{
    ESError rv = ES_ERR_OK;
    PublishTicket pubTik( tkt6, sizeof( tkt6 ) );
    u32 ticketLen;
    IOSCAesIv aesIv;
    IOSCAesKey titleKey;
    IOSCAesKey titleKeyClr;
    ESRightsId rightsId;
    ESDeviceId deviceId;
    ESDeviceId certDeviceId;
    u8 newCommonKeyId = 0;
    ESTicketId ticketId;
    ESTicketId nTicketId;
    IOSCPublicKeyHandle hDevPubKey;
    IOSCSecretKeyHandle hXsEccPrivKey;
    IOSCSecretKeyHandle hSharedKey;

    /*
     * Dump the template
     */
    CHECK_RESULT( pubTik.DumpTicket(), ES_ERR_OK );

    /*
     * Get the info needed to convert the titleKey
     */
    CHECK_RESULT( pubTik.GetRightsId( &rightsId ), ES_ERR_OK );
    CHECK_RESULT( pubTik.GetTitleKey( titleKey ), ES_ERR_OK );
    CHECK_RESULT( pubTik.GetCommonKeyId( &commonKeyId ), ES_ERR_OK );
    CHECK_RESULT( pubTik.GetDeviceId( &deviceId ), ES_ERR_OK );
    CHECK_RESULT( pubTik.GetTicketId( &ticketId ), ES_ERR_OK );

    NN_LOG( "Data from template:\n" );
    NN_LOG( "Rights ID = \n");
    dumpRightsId(rightsId);
    NN_LOG( "Device ID = %08x\n", (u32)deviceId );
    NN_LOG( "Common Key ID = %d\n", commonKeyId );
    dumpData( "Title Key from ticket:", titleKey, sizeof( titleKey ) );

    /*
     * If ticket is personalized, do the conversion using the device certificate (act as XS server)
     */
    if( deviceId != 0 )
    {
        NN_LOG( "Ticket is personalized!\n" );
        /*
         * Check the device certificate to see if it matches
         */
        sscanf( (const char *)devCert.head.name.deviceId, "%16llx", &certDeviceId );
        if( deviceId != certDeviceId )
        {
            NN_LOG( "Device Id from DevCert file %s (%08x) but ticket is personalized for %08x\n", devCertFile, certDeviceId, deviceId );
            goto out;
        }
        NN_LOG( "Device Id from DevCert matches ticket (%08x)\n", certDeviceId );
        CHECK_RESULT( pubTik.SetDeviceCert( (u8 *)&devCert, sizeof( devCert ) ), ES_ERR_OK );

        /*
         * Use device public key to do ECDH
         */
        rv = IOSC_CreateObject( &hDevPubKey, IOSC_PUBLICKEY_TYPE, IOSC_ECC233_SUBTYPE );
        if( rv != IOSC_ERROR_OK )
        {
            NN_LOG( "Failed to create public key object, rv=%d\n", rv );
            rv = ES_ERR_CRYPTO;
            goto out;
        }
        rv = IOSC_ImportPublicKey( devCert.pubKey, NULL, hDevPubKey );
        if( rv != IOSC_ERROR_OK )
        {
            NN_LOG( "Failed to import device public key, rv=%d\n", rv );
            rv = ES_ERR_CRYPTO;
            goto out;
        }

        rv = IOSC_CreateObject( &hXsEccPrivKey, IOSC_SECRETKEY_TYPE, IOSC_ECC233_SUBTYPE );
        if( rv != IOSC_ERROR_OK )
        {
            NN_LOG( "Failed to create secret key object, rv=%d\n", rv );
            rv = ES_ERR_CRYPTO;
            goto out;
        }
        rv = IOSC_ImportSecretKey( hXsEccPrivKey, 0, 0, IOSC_NOSIGN_NOENC, NULL, NULL, xsEccPrivKey );
        if( rv != IOSC_ERROR_OK )
        {
            NN_LOG( "Failed to import XS ECC Private key, rv=%d\n", rv );
            rv = ES_ERR_CRYPTO;
            goto out;
        }

        rv = IOSC_CreateObject( &hSharedKey, IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE );
        if( rv != IOSC_ERROR_OK )
        {
            NN_LOG( "Failed to create secret key object, rv=%d\n", rv );
            rv = ES_ERR_CRYPTO;
            goto out;
        }

        rv = IOSC_ComputeSharedKey( hXsEccPrivKey, hDevPubKey, hSharedKey );
        if( rv != IOSC_ERROR_OK )
        {
            NN_LOG( "Failed to calculate shared key, rv=%d\n", rv );
            rv = ES_ERR_CRYPTO;
            goto out;
        }

        memset( aesIv, 0, sizeof( aesIv ) );
        nTicketId = ntohll( ticketId );
        memcpy( aesIv, &nTicketId, sizeof( nTicketId ) );
        rv = IOSC_Decrypt( hSharedKey, aesIv, titleKey, sizeof( titleKey ), titleKeyClr );
        if( rv != IOSC_ERROR_OK )
        {
            NN_LOG( "Decrypt failed (rv %d)\n", rv );
            goto out;
        }

        dumpData( "Title Key from ticket after depersonalization:", titleKeyClr, sizeof( titleKeyClr ) );
        memcpy( titleKey, titleKeyClr, sizeof( titleKey ) );

        (void)IOSC_DeleteObject( hSharedKey );
        (void)IOSC_DeleteObject( hXsEccPrivKey );
        (void)IOSC_DeleteObject( hDevPubKey );
    }

    rv = IOSC_CreateObject( &hCommonKey, IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE );
    if( rv != IOSC_ERROR_OK )
    {
        NN_LOG( "Failed to create secret key object, rv=%d\n", rv );
        rv = ES_ERR_CRYPTO;
        goto out;
    }

    rv = IOSC_ImportSecretKey( hCommonKey, 0, 0, IOSC_NOSIGN_NOENC, NULL, NULL, rvlCommonKey );
    if( rv != IOSC_ERROR_OK )
    {
        NN_LOG( "Failed to import common key, rv=%d\n", rv );
        rv = ES_ERR_CRYPTO;
        goto out;
    }

    /*
     * Decrypt the titleKey with the previous common key
     *
     * Initialization Vector is the Rights ID (network byte order)
     */
    memset( aesIv, 0, sizeof( aesIv ) );
    memcpy( aesIv, rightsId, sizeof( rightsId ) );
    rv = IOSC_Decrypt( hCommonKey, aesIv, titleKey, sizeof( titleKey ), titleKeyClr );
    if( rv != IOSC_ERROR_OK )
    {
        NN_LOG( "Decrypt failed (rv %d)\n", rv );
        goto out;
    }
    (void)IOSC_DeleteObject( hCommonKey );

    dumpData( "Title Key (Cleartext):", titleKeyClr, sizeof( titleKeyClr ) );

    /*
     * Now re-encrypt the Title Key with the new common key
     */
    rv = IOSC_CreateObject( &hCommonKey, IOSC_SECRETKEY_TYPE, IOSC_ENC_SUBTYPE );
    if( rv != IOSC_ERROR_OK )
    {
        NN_LOG( "Failed to create secret key object, rv=%d\n", rv );
        rv = ES_ERR_CRYPTO;
        goto out;
    }

    rv = IOSC_ImportSecretKey( hCommonKey, 0, 0, IOSC_NOSIGN_NOENC, NULL, NULL, evlCommonKey );
    if( rv != IOSC_ERROR_OK )
    {
        NN_LOG( "Failed to import common key, rv=%d\n", rv );
        rv = ES_ERR_CRYPTO;
        goto out;
    }

    memset( aesIv, 0, sizeof( aesIv ) );
    memcpy( aesIv, rightsId, sizeof(rightsId) );
    rv = IOSC_Encrypt( hCommonKey, aesIv, titleKeyClr, sizeof( titleKeyClr ), titleKey );
    if( rv != IOSC_ERROR_OK )
    {
        NN_LOG( "Encrypt failed (rv %d)\n", rv );
        goto out;
    }
    (void)IOSC_DeleteObject( hCommonKey );

    dumpData( "Title Key (re-encrypted with new CK):", titleKey, sizeof( titleKey ) );

    /*
     * Reset to common key id 0 for applications and key id 1 for system titles
     */
    CHECK_RESULT( pubTik.SetCommonKeyId( newCommonKeyId ), ES_ERR_OK );
    CHECK_RESULT( pubTik.SetTitleKey( titleKey ), ES_ERR_OK );

#ifdef USE_OBSOLETE_CODE
    /*
     * Set some item rights, since those are ignored in a template
     */
    rightsEntries[0].type = ES_ITEM_RIGHT_PERMANENT;
    rightsEntries[0].right.perm.referenceId[0] = 0x42;
    rightsEntries[0].right.perm.referenceId[15] = 0x77;
    rightsEntries[0].right.perm.referenceIdAttr = 0xbabecafe;
    rightsEntries[1].type = ES_ITEM_RIGHT_SUBSCRIPTION;
    rightsEntries[1].right.sub.limit = 27;
    rightsEntries[1].right.sub.referenceId[7] = 0xa5;
    rightsEntries[1].right.sub.referenceId[11] = 0xfe;
    rightsEntries[1].right.sub.referenceIdAttr = 0xdeadbeef;
    rightsEntries[2].type = ES_ITEM_RIGHT_PERMANENT;
    rightsEntries[2].right.perm.referenceId[0] = 0x58;
    rightsEntries[2].right.perm.referenceId[15] = 0x24;
    rightsEntries[2].right.perm.referenceIdAttr = 0xdeafdadd;
    rightsEntries[3].type = ES_ITEM_RIGHT_CONTENT_CONSUMPTION;
    rightsEntries[3].right.cntLmt.index = 3;
    rightsEntries[3].right.cntLmt.code = ES_LC_DURATION_TIME;
    rightsEntries[3].right.cntLmt.limit = 10;
    rightsEntries[4].type = ES_ITEM_RIGHT_ACCESS_TITLE;
    rightsEntries[4].right.accTitle.accessTitleId = 0x5000004100000043LL;
    rightsEntries[4].right.accTitle.accessTitleMask = 0xff0000ff00ff00ffLL;
    rightsEntries[5].type = ES_ITEM_RIGHT_CONTENT_CONSUMPTION;
    rightsEntries[5].right.cntLmt.index = 5;
    rightsEntries[5].right.cntLmt.code = ES_LC_ELAPSED_TIME;
    rightsEntries[5].right.cntLmt.limit = 142;
    /*
     * For titles with more than 1024 content items, this method must
     * be used to specify which contents are activated by this ticket.
     * It can also be used for < 1024 content titles.  Each record
     * specifies the access bit mask for 1024 contents starting at
     * the "offset" value.
     */
    rightsEntries[6].type = ES_ITEM_RIGHT_CONTENT;
    rightsEntries[6].right.cnt.offset = 0;
    rightsEntries[6].right.cnt.accessMask[0] = 0x21;
    rightsEntries[6].right.cnt.accessMask[62] = 0xf0;
    rightsEntries[6].right.cnt.accessMask[63] = 0x1f;
    rightsEntries[6].right.cnt.accessMask[126] = 0x07;
    rightsEntries[7].type = ES_ITEM_RIGHT_CONTENT;
    rightsEntries[7].right.cnt.offset = 1024;
    rightsEntries[7].right.cnt.accessMask[0] = 0x50;
    rightsEntries[7].right.cnt.accessMask[62] = 0xa5;
    rightsEntries[7].right.cnt.accessMask[63] = 0xe5;
    rightsEntries[7].right.cnt.accessMask[127] = 0x99;
    CHECK_RESULT( pubTik.SetItemRights( rightsEntries, 8 ), ES_ERR_OK );
#endif
    CHECK_RESULT( pubTik.SignTicket( issuerName, rsaPubMod, rsaPrivExp ), ES_ERR_OK );

    /*
     * If the ticket is personalized, re-personalize it
     */
    if( deviceId != 0 )
    {
        NN_LOG( "Ticket is personalized!\n" );
        CHECK_RESULT( pubTik.PersonalizeTicket( xsEccPrivKey, sizeof( xsEccPrivKey ) ), ES_ERR_OK );
        CHECK_RESULT( pubTik.GetTitleKey( titleKey ), ES_ERR_OK );
        dumpData( "Title Key after personalization:", titleKey, sizeof( titleKey ) );
    }

    CHECK_RESULT( pubTik.GetTicketLen( &ticketLen ), ES_ERR_OK );
    if( ticketLen > sizeof( ticketBuf ) )
    {
        NN_LOG( "Static ticket buffer too small (need %d)\n", ticketLen );
        rv = ES_ERR_INVALID;
        goto out;
    }
    CHECK_RESULT( pubTik.OutputTicket( ticketBuf, ticketLen ), ES_ERR_OK );
    CHECK_RESULT( pubTik.DumpTicket(), ES_ERR_OK );

    /*
     * Now write it out to a file
     */
    writeFile(__ticketFile, ticketBuf, ticketLen);

out:

    NN_LOG( "doPublishTicket returns %d\n", rv );
    return rv;
}  // NOLINT (readability/fn_size)

/**
    @brief FS用アロケータ
 */
void* Allocate(size_t size)
{
    return std::malloc(size);
}

/**
    @brief FS用デアロケータ
 */
void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    std::free(p);
}


TEST( PublishRepublishTest, PublishRepublish )
{
    ESError rv = ES_ERR_OK;

    nn::fs::SetAllocator(Allocate, Deallocate);
    nn::fs::MountHostRoot();

    initKeys();
    loadDevCert();

    rv = doPublishTicket();
    EXPECT_EQ( rv, ES_ERR_OK );
}
