﻿/*
 *  Copyright 2005-2014 Acer Cloud Technology, Inc.
 *  All Rights Reserved.
 *
 *  This software contains confidential information and
 *  trade secrets of Acer Cloud Technology, Inc.
 *  Use, disclosure or reproduction is prohibited without
 *  the prior express written permission of Acer Cloud
 *  Technology, Inc.
 */

/* test of RNG functions
 */

#include <nnt/nntest.h>
//#include <types.h>
#include <nn/iosc/iosc.h>
#include <nn/csl/aes.h>

#ifdef LINUX
#include <cstdio>
#include <cstring>
#include <cstdlib>
#else
#include <nn/ioslibc/ioslibc.h>
#endif

#include "testEs_Testcommon.h"
#include "testEs_Testutils.h"

#ifdef NET_CARD
#include <sc.h>
const int STACK_SIZE = ( ( 256 + 32 ) * 1024 );
const u8 _initStack[STACK_SIZE];
const u32 _initStackSize = sizeof( _initStack );
const u32 _initPriority = 10;
const int NUM_TICKS = 200;
const int CLOCK_FREQ = 100;

u32 getNumTicks( u32 start, u32 end )
{
    u32 diff;
    diff = ( end >= start ) ? end - start : end + ( 0xFFFFFFFF - start );
    return diff;
}
#endif

#ifdef RVL
#include <iop.h>
extern u32 getNumTicks( u32, u32 );
const int NUM_TICKS = 128;
const int CLOCK_FREQ = 243;
#endif

#ifdef LINUX
#define ALIGN_SIZE 16 // NOLINT(readability/define)
#else
#define ALIGN_SIZE 64 // NOLINT(readability/define)
#endif

#define TESTNAME "RngTest"

#define ERROR_CHECK( err )                                                            \
    if( err != IOSC_ERROR_OK )                                                        \
    {                                                                                 \
        TUL_LogFail( TESTNAME, "error %d in IOSC call at line %d\n", err, __LINE__ ); \
        EXIT( -1 );                                                                   \
    }

/* this whole test is only defined for 20000 bits */

static const short MINRUN[7] = {0, 2315, 1114, 527, 240, 103, 103};
static const short MAXRUN[7] = {0, 2685, 1386, 723, 384, 209, 209};

const int LONGRUN = 26;
const int MINONES = 9725;
const int MAXONES = 10275;

/*
#define MINPOKE 2.16
#define MAXPOKE 46.17
*/
const int MINPOKE = 2;
const int MAXPOKE = 46;

/*
 * Table lookup for number of 1 bits in a byte
 */
const unsigned char Popcount[] = {0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4, 1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5, 1, 2, 2, 3, 2,
                                  3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 1, 2, 2, 3, 2, 3, 3, 4, 2, 3,
                                  3, 4, 3, 4, 4, 5, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5,
                                  6, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5, 2, 3, 3, 4,
                                  3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 3, 4, 4, 5, 4, 5, 5, 6, 4,
                                  5, 5, 6, 5, 6, 6, 7, 2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6,
                                  6, 7, 3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8};

void endrun( int last, int run, u32 runs[][7], int *nerrs )
{
    if( run >= LONGRUN )
    {
        LOG_FAIL( "Run of %d %s", run, last ? "ones" : "zeros" );
        ++( *nerrs );
    }
    if( run > 6 )
    {
        run = 6;
    }
    ++runs[last][run];
}

int dofips( u8 *b, int size )
{
    int i;
    u8 *p;
    u32 c;
    u32 X;
    u32 last;
    u32 run;
    u32 poker[16];
    u32 runs[2][7];
    int nerrs = 0;
    int rv;

    /* monobit test */
    for( p = b, c = 0; p < &b[size]; p++ )
    {
        c += Popcount[*p];
    }

    /* monobit test */
    if( c <= MINONES || MAXONES <= c )
    {
        LOG_FAIL( "Monobit test, ones = %d", c );
        ++nerrs;
    }
    else
    {
        LOG_PASS( "Monobit test, ones = %d", c );
    }

    /* poker test */
    memset( poker, 0, sizeof poker );
    for( p = b; p < &b[size]; p++ )
    {
        ++poker[*p & 0xF];
        ++poker[( *p >> 4 ) & 0xF];
    }

    X = 0;
    for( i = 0; i < 16; i++ )
    {
        X += static_cast<u32>( poker[i] * poker[i] );
#ifdef DEBUG
        LOG_MESSAGE( "poker[%d] %d, cumulative value-> %d", i, poker[i], X );
#endif
    }
#ifdef DEBUG
    LOG_MESSAGE( "after summing poker values" );
    LOG_MESSAGE( "before X->%d", X );
#endif

    X = 16 * X / 5000 - 5000;
#ifdef DEBUG
    LOG_MESSAGE( "after X-> %d", X );
#endif
    if( ( X <= MINPOKE ) || ( MAXPOKE <= X ) )
    {
        LOG_FAIL( "sample fails poker test, X = %d (m/M = %d/%d)", X, MINPOKE, MAXPOKE );
        ++nerrs;
    }
    else
    {
        LOG_PASS( "poker test, X = %d (m/M = %d/%d)", X, MINPOKE, MAXPOKE );
    }

    /* runs test */

    memset( runs, 0, sizeof( runs ) );
    last = ( b[0] >> 7 ) & 1;
    run = 0;
    for( p = b; p < &b[size]; p++ )
    {
        c = *p;
        for( i = 7; i >= 0; --i )
        {
            if( ( ( c >> i ) & 1 ) != last )
            {
                endrun( last, run, runs, &nerrs );
                run = 0;
                last = ( c >> i ) & 1;
            }
            ++run;
        }
    }
    endrun( last, run, runs, &nerrs );

    for( run = 1; run <= 6; run++ )
    {
        for( last = 0; last <= 1; last++ )
        {
            if( runs[last][run] <= static_cast<u32>( MINRUN[run] ) )
            {
                LOG_FAIL( "too few runs of %d %s: %d", run, last ? "ones" : "zeros", runs[last][run] );
                ++nerrs;
            }
            else if( runs[last][run] >= static_cast<u32>( MAXRUN[run] ) )
            {
                LOG_FAIL( "too many runs of %d %s: %d", run, last ? "ones" : "zeros", runs[last][run] );
                ++nerrs;
            }
        }
    }

    if( nerrs == 0 )
    {
        LOG_PASS( "All statistical tests PASS" );
        rv = IOSC_ERROR_OK;
    }
    else
    {
        LOG_FAIL( "At least some statistical tests FAIL" );
        rv = IOSC_ERROR_INVALID;
    }

    return rv;
}

u8 fipsData[2500];

int doRngTests()
{
    IOSCError err;
    int iter;
    u8 rand[IOSC_MAX_RAND_BYTES];
#ifdef PERF
    u32 startTime, endTime;
#endif

    /*
     * Test of GenerateRand
     */
    for( iter = 0; iter < 10; iter++ )
    {
        err = IOSC_GenerateRand( rand, IOSC_MAX_RAND_BYTES );
        ERROR_CHECK( err );
        LOG_MESSAGE( "%02x %02x %02x %02x", rand[0], rand[1], rand[2], rand[3] );
    }
#ifdef PERF
    startTime = IO_READ( SYS_TIMER_REG );
    err = IOSC_GenerateRand( rand, IOSC_MAX_RAND_BYTES );
    endTime = IO_READ( SYS_TIMER_REG );
    NN_LOG( "Number of Ticks for IOSC_GenerateRand = %d, Time = %d us per %d bytes\n", getNumTicks( startTime, endTime ),
            getNumTicks( startTime, endTime ) * NUM_TICKS / CLOCK_FREQ, IOSC_MAX_RAND_BYTES );
#endif

    /*
     * Test of FIPS
     */
    for( iter = 0; iter < sizeof( fipsData ) / 100; iter++ )
    {
        err = IOSC_GenerateRand( fipsData + 100 * iter, 100 );
        ERROR_CHECK( err );
    }

#ifdef DO_FIPS_TEST
    dofips( fipsData, sizeof( fipsData ) );
#endif
    return 0;
}

#if defined( SEPARATE_MAIN_PROGRAM )

int main( int argc, char **argv )
{
    IOSC_Initialize();
    doRngTests();
    EXIT( 0 );
    return 0;
}
#endif
