﻿/*
 *  Copyright 2005-2014 Acer Cloud Technology, Inc.
 *  All Rights Reserved.
 *
 *  This software contains confidential information and
 *  trade secrets of Acer Cloud Technology, Inc.
 *  Use, disclosure or reproduction is prohibited without
 *  the prior express written permission of Acer Cloud
 *  Technology, Inc.
 */

/*
 *               Copyright (C) 2009, BroadOn Communications Corp.
 *
 *  These coded instructions, statements, and computer programs contain
 *  unpublished  proprietary information of BroadOn Communications Corp.,
 *  and  are protected by Federal copyright law. They may not be disclosed
 *  to  third  parties or copied or duplicated in any form, in whole or in
 *  part, without the prior written consent of BroadOn Communications Corp.
 *
 */

#include <cstdio>
#include <cstring>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>

#include <nn/istorage/istorage.h>

#include <nnt/escoreUtil/testEscore_util_istorage.h>

USING_RESULT_NAMESPACE

namespace nnt { namespace escore { namespace util {

MemoryInputStream::MemoryInputStream(const void *buffer, u32 size)
{
    rBuf = (const u8 *) buffer;
    rSize = size;
    rPos = 0;
}


MemoryInputStream::~MemoryInputStream()
{
    /* Do nothing */
}


RESULT_NAMESPACE::Result
MemoryInputStream::TryRead(s32 *pOut, void *buffer, u32 size)
{
    RESULT_NAMESPACE::Result res;
    u32 bytes;

    res.Set(0);

    if (rPos + size > rSize) {
        bytes = (u32)(rSize - rPos);
    } else {
        bytes = size;
    }

    memcpy(buffer, &rBuf[rPos], bytes);
    *pOut = (s32) bytes;
    rPos += bytes;

    return res;
}


RESULT_NAMESPACE::Result
MemoryInputStream::TrySetPosition(s64 position)
{
    RESULT_NAMESPACE::Result res;

    res.Set(0);

    if (position >= rSize) {
        res.Set(-1);
        goto end;
    } else {
        rPos = position;
    }

end:
    return res;
}


RESULT_NAMESPACE::Result
MemoryInputStream::TryGetPosition(s64 *pOut) const
{
    RESULT_NAMESPACE::Result res;

    res.Set(0);

    *pOut = rPos;

    return res;
}


RESULT_NAMESPACE::Result
MemoryInputStream::TryGetSize(s64 *pOut) const
{
    RESULT_NAMESPACE::Result res;

    res.Set(0);

    *pOut = rSize;

    return res;
}

MemoryOutputStream::MemoryOutputStream(void *buffer, u32 size)
{
    wBuf = (u8 *) buffer;
    wSize = size;
    wPos = 0;
}


MemoryOutputStream::~MemoryOutputStream()
{
    /* Do nothing */
}


RESULT_NAMESPACE::Result
MemoryOutputStream::TryWrite(s32 *pOut, const void *buffer, u32 size, bool flush)
{
    NN_UNUSED(flush);
    RESULT_NAMESPACE::Result res;
    u32 bytes;

    res.Set(0);

    if (wPos + size > wSize) {
        bytes = (u32)(wSize - wPos);
    } else {
        bytes = size;
    }

    memcpy(&wBuf[wPos], buffer, bytes);
    *pOut = (s32) bytes;
    wPos += bytes;

    return res;
}


RESULT_NAMESPACE::Result
MemoryOutputStream::TrySetPosition(s64 position)
{
    RESULT_NAMESPACE::Result res;

    res.Set(0);

    if (position >= wSize) {
        res.Set(-1);
        goto end;
    } else {
        wPos = position;
    }

end:
    return res;
}


RESULT_NAMESPACE::Result
MemoryOutputStream::TryGetPosition(s64 *pOut) const
{
    RESULT_NAMESPACE::Result res;

    res.Set(0);

    *pOut = wPos;

    return res;
}


RESULT_NAMESPACE::Result
MemoryOutputStream::TryGetSize(s64 *pOut) const
{
    RESULT_NAMESPACE::Result res;

    res.Set(0);

    *pOut = wSize;

    return res;
}


/*
 * FileStream class for Linux
 */
RESULT_NAMESPACE::Result
FileStream::TryInitialize(const char *pathName, u32 openMode)
{
    int mode = 0;   // stay away from 'text' files
    RESULT_NAMESPACE::Result res;
    nn::Result result;

    res.Set(0);

    if (f_Stream != NULL) {
        nn::fs::CloseFile(*f_Stream);
        delete f_Stream;
    }

    f_Stream = new nn::fs::FileHandle();
    f_Position = 0;
    f_Size = 0;
    f_Mode = (OpenMode) openMode;

    if ((openMode & OPEN_MODE_READ) == OPEN_MODE_READ) mode |= nn::fs::OpenMode_Read;
    if ((openMode & OPEN_MODE_WRITE) == OPEN_MODE_WRITE) mode |= nn::fs::OpenMode_Write;
    if ((openMode & OPEN_MODE_CREATE) == OPEN_MODE_CREATE)
    {
        nn::fs::DeleteFile(pathName);
        result = nn::fs::CreateFile(pathName, 0);
        if (result.IsFailure())
        {
            NN_LOG( "create failed for %s", pathName );
            res.Set(-1);
        }
        mode |= nn::fs::OpenMode_AllowAppend;
    }

    result = nn::fs::OpenFile(f_Stream, pathName, mode);
    if ( result.IsFailure() )
    {
        NN_LOG( "open failed for %s", pathName );
        res.Set(-1);
    }

    /*
     * Calculate the file size
     */
    int64_t fileSize;
    if ((openMode & OPEN_MODE_READ) == OPEN_MODE_READ) {
        nn::fs::GetFileSize(&fileSize, *f_Stream);
        f_Size = static_cast<s64>(fileSize);
    } else if ((openMode & OPEN_MODE_WRITE) == OPEN_MODE_WRITE) {
        nn::fs::GetFileSize(&fileSize, *f_Stream);
        f_Size = static_cast<s64>(fileSize);
    }

    /*
     * No need to reset pointers, since this code honors f_Position
     */

    return res;
}


void
FileStream::Initialize(const char *pathName, u32 openMode)
{
    RESULT_NAMESPACE::Result res;

    res = FileStream::TryInitialize(pathName, openMode);

    return;
}


void
FileStream::Finalize()
{
    if (f_Stream != NULL) {
        nn::fs::CloseFile(*f_Stream);
        delete f_Stream;
        f_Stream = NULL;
    }

    return;
}


s32
FileStream::Read(void *buffer, size_t size)
{
    s32 rCount;
    nn::Result result;
    if (f_Stream == NULL || (f_Mode & OPEN_MODE_READ) != OPEN_MODE_READ) {
        rCount = -1;
        goto end;
    }

    size_t readSize;
    result = nn::fs::ReadFile(&readSize, *f_Stream, f_Position, buffer, size);
    if (result.IsFailure())
    {
        NN_LOG( "read failed");
    }
    rCount = static_cast<s32>(readSize);

    f_Position += readSize;

end:
    return rCount;
}


RESULT_NAMESPACE::Result
FileStream::TryRead(s32 *pOut, void *buffer, u32 size)
{
    s32 rCount;
    RESULT_NAMESPACE::Result res;

    rCount = FileStream::Read(buffer, size);
    *pOut = rCount;

    res.Set( rCount >= 0 ? 0 : -1 );

    return res;
}


s32
FileStream::Write(const void *buffer, size_t size, bool flush)
{
    s32 wCount = 0;
    s64 startPos;
    s64 startSize;
    nn::Result result;

    if (f_Stream == NULL || (f_Mode & OPEN_MODE_WRITE) != OPEN_MODE_WRITE) {
        wCount = -1;
        goto end;
    }

    startPos = f_Position;
    startSize = f_Size;

    result = nn::fs::WriteFile(*f_Stream, f_Position, buffer, size,
        flush ? nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush) :
        nn::fs::WriteOption::MakeValue(0));
    if (result.IsFailure()) {
        wCount = -1;
        goto end;
    }

    f_Position += size;
    wCount = (s32)(f_Position - startPos);

    int64_t fileSize;
    nn::fs::GetFileSize(&fileSize, *f_Stream);
    f_Size = static_cast<s64>(fileSize);

end:
    return wCount;
}


RESULT_NAMESPACE::Result
FileStream::TryWrite(s32 *pOut, const void *buffer, u32 size, bool flush)
{
    s32 wCount;
    RESULT_NAMESPACE::Result res;

    wCount = FileStream::Write(buffer, size, flush);
    *pOut = wCount;

    res.Set( wCount >= 0 ? 0 : -1 );

    return res;
}


RESULT_NAMESPACE::Result
FileStream::TrySeek(s64 position, PositionBase base)
{
    RESULT_NAMESPACE::Result res;
    s64 newPos = -1LL;  // catch the case that "base" is bogus

    if (f_Stream == NULL) {
        goto end;
    }

    if (base == POSITION_BASE_BEGIN) {
        newPos = position;
    } else if (base == POSITION_BASE_END) {
        newPos = f_Size + position;
    } else if (base == POSITION_BASE_CURRENT) {
        newPos = f_Position + position;
    }

end:
    /*
     * Make sure the position is valid, beyond current EOF is ok
     */
    if (newPos < 0) {
        res.Set(-1);
    } else {
        f_Position = newPos;
        res.Set(0);
    }

    return res;
}


void
FileStream::Seek(s64 position, PositionBase base)
{
    RESULT_NAMESPACE::Result res;

    res = FileStream::TrySeek(position, base);

    return;
}


RESULT_NAMESPACE::Result
FileStream::TryGetSize(s64 *pOut) const
{
    RESULT_NAMESPACE::Result res;

    res.Set(0);
    *pOut = f_Size;

    return res;
}


RESULT_NAMESPACE::Result
FileStream::TrySetSize(s64 size)
{
    RESULT_NAMESPACE::Result res;

    if (size < 0) {
        res.Set(-1);
    } else {
        res.Set(0);
        f_Size = size;
    }

    /*
     * XXX should this code seek to size - 1 and write 1?
     */

    return res;
}


RESULT_NAMESPACE::Result
FileStream::TryGetPosition(s64 *pOut) const
{
    RESULT_NAMESPACE::Result res;

    res.Set(0);
    *pOut = f_Position;

    return res;
}


RESULT_NAMESPACE::Result
FileStream::TrySetPosition(s64 position)
{
    RESULT_NAMESPACE::Result res;

    if (position < 0) {
        res.Set(-1);
    } else {
        res.Set(0);
        f_Position = position;
    }

    return res;
}

}}}
