﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/es.h>
#include <nn/fs.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/es/es_MountName.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt/nntest.h>
#include <nnt/esUtil/testEs_CertificateData.h>
#include <nnt/esUtil/testEs_Utility.h>
#include "es_MaxTicketCount.h"

// チケットの最大数
const int MaxTicketCount = 8192;

// 最大のチケットのサイズ
const int TicketSize = 704;

// テスト用チケットを読み込むためのマウント名
const char* MountName = "ticketdata";

// テスト用 Common チケットのファイルパス
const char* TicketTestDataPath = "ticketdata:/Tests/Es/Sources/Tests/ManyTicketInstall/testEs_ManyTicketInstall/TestTicketData.bin";

class Ticket : public ::testing::Test
{
protected:
    virtual void SetUp()
    {
        nn::es::DeleteAllCommonTicket();
        nn::es::DeleteAllPersonalizedTicket();
    }

    virtual void TearDown()
    {
    }

    static void SetUpTestCase()
    {
        // サービスオブジェクトへの参照の取得
        nn::es::Initialize();
        nn::es::DeleteAllCommonTicket();

        nn::fs::MountHost(MountName, NNT_ES_SIGLO_ROOT);
    }

    static void TearDownTestCase()
    {
        nn::es::DeleteAllCommonTicket();
        // 共有ポインタの解放
        nn::es::Finalize();

        nn::fs::Unmount(MountName);
    }
};

TEST_F(Ticket, InstallManyTicket)
{
    char ticketPath[200];
    nn::util::SNPrintf(ticketPath, sizeof(ticketPath), "%s", TicketTestDataPath);

    nn::fs::FileHandle fileHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, ticketPath, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(fileHandle); };

    // リストファイルサイズ取得
    int64_t fileSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, fileHandle));

    std::unique_ptr<char[]> ticketBuffer(new char[static_cast<size_t>(fileSize)]);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, 0, ticketBuffer.get(), static_cast<size_t>(fileSize)));

    const size_t certificateSize = nnt::es::caNewCertSize + nnt::es::xsNewCertSize;
    char certificateBuffer[certificateSize];

    std::memcpy(certificateBuffer, nnt::es::caNewCert, nnt::es::caNewCertSize);
    std::memcpy(certificateBuffer + nnt::es::caNewCertSize, nnt::es::xsNewCert, nnt::es::xsNewCertSize);

    // テスト用チケットデータをインポートする
    for (int i = 0; i < MaxTicketCount; i++)
    {
        NN_LOG("%d / %d ticket importing.\n", i + 1, MaxTicketCount);

        char ticketData[TicketSize];
        nn::es::TicketInfo ticketInfo;

        std::memcpy(ticketData, ticketBuffer.get() + TicketSize * i, TicketSize);
        ticketInfo = nnt::es::CreateTicketInfo(ticketData, TicketSize);

        nn::Result result = nn::es::ImportTicket(ticketData, TicketSize, certificateBuffer, certificateSize);

        if (i < nn::es::MaxCommonTicketCount)
        {
            EXPECT_TRUE(result.IsSuccess());
            EXPECT_EQ(i + 1, nn::es::CountCommonTicket());

            bool hasTicket;
            nn::es::OwnTicket(&hasTicket, &ticketInfo.rightsId, 1);
            EXPECT_TRUE(hasTicket);

            size_t outTicketSize;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::es::GetCommonTicketSize(&outTicketSize, ticketInfo.rightsId));
            EXPECT_EQ(TicketSize, outTicketSize);

            char outTicketBuffer[TicketSize];
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::es::GetCommonTicketData(&outTicketSize, outTicketBuffer, TicketSize, ticketInfo.rightsId));
            EXPECT_EQ(TicketSize, outTicketSize);
            EXPECT_TRUE(std::memcmp(ticketData, outTicketBuffer, TicketSize) == 0);

            nn::es::TicketInfo outTicketInfo;
            int ticketInfoCount = nn::es::GetTicketInfo(&outTicketInfo, 1, &ticketInfo.rightsId, 1);
            EXPECT_EQ(1, ticketInfoCount);
            EXPECT_TRUE(nnt::es::operator==(outTicketInfo, ticketInfo));
        }
        else
        {
            EXPECT_TRUE(nn::es::ResultTicketFull::Includes(result));
            EXPECT_EQ(nn::es::MaxCommonTicketCount, nn::es::CountCommonTicket());

            bool hasTicket;
            nn::es::OwnTicket(&hasTicket, &ticketInfo.rightsId, 1);
            EXPECT_FALSE(hasTicket);

            size_t outTicketSize;
            result = nn::es::GetCommonTicketSize(&outTicketSize, ticketInfo.rightsId);
            EXPECT_TRUE(nn::es::ResultRightsIdNotFound::Includes(result));

            char outTicketBuffer[TicketSize];
            result = nn::es::GetCommonTicketData(&outTicketSize, outTicketBuffer, TicketSize, ticketInfo.rightsId);
            EXPECT_TRUE(nn::es::ResultRightsIdNotFound::Includes(result));

            nn::es::TicketInfo outTicketInfo;
            int ticketInfoCount = nn::es::GetTicketInfo(&outTicketInfo, 1, &ticketInfo.rightsId, 1);
            EXPECT_EQ(0, ticketInfoCount);
        }
    }

    int count = nn::es::CountCommonTicket();
    NN_SDK_ASSERT_EQUAL(nn::es::MaxCommonTicketCount, count);
    NN_UNUSED(count);
}

